package com.af.v4.system.common.redis;

import org.json.JSONArray;
import org.json.JSONObject;
import org.redisson.api.RBucket;
import org.redisson.api.RLock;
import org.redisson.api.RMap;
import org.redisson.api.RedissonClient;
import org.redisson.codec.JsonJacksonCodec;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.time.Duration;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

/**
 * Redis服务
 *
 * @author Mr.river
 */
@Component
public class RedisService {

    private static final Logger LOGGER = LoggerFactory.getLogger(RedisService.class);

    /**
     * redisson客户端
     */
    private final RedissonClient redissonClient;

    public RedisService(RedissonClient REDISSON_CLIENT) {
        redissonClient = REDISSON_CLIENT;
    }

    public <T> void set(String key, T value) {
        set(key, value, -1);
    }

    public <T> void set(String key, T value, Object timeoutSec) {
        long timeoutSecValue = Long.parseLong(timeoutSec.toString());
        RBucket<T> rBucket = redissonClient.getBucket(key, JsonJacksonCodec.INSTANCE);
        if (timeoutSecValue > 0) {
            rBucket.set(value, Duration.ofSeconds(timeoutSecValue));
        } else {
            rBucket.set(value);
        }
    }

    public <T> void setHash(String hashKey, Map<String, T> value) {
        setHash(hashKey, value, -1);
    }

    public <T> void setHash(String hashKey, Map<String, T> value, Object timeoutSec) {
        long timeoutSecValue = Long.parseLong(timeoutSec.toString());
        RMap<String, T> rMap = redissonClient.getMap(hashKey);
        rMap.putAll(value);
        if (timeoutSecValue > 0) {
            rMap.expire(Duration.ofSeconds(timeoutSecValue));
        }
    }

    public JSONObject getHash(String hashKey) {
        return new JSONObject(redissonClient.getMap(hashKey));
    }

    public <T> T getHash(String hashKey, String key) {
        RMap<String, T> map = redissonClient.getMap(hashKey);
        return map.get(key);
    }

    public <T> void setHashKey(String hashKey, String key, T value) {
        RMap<String, T> map = redissonClient.getMap(hashKey);
        map.put(key, value);
    }

    public void deleteHashKey(String hashKey, String key) {
        redissonClient.getMap(hashKey).remove(key);
    }

    public Boolean hasHashKey(String hashKey, String key) {
        return redissonClient.getMap(hashKey).containsKey(key);
    }

    public void lock(String key, Runnable runnable) {
        lock(key, 10, 20, runnable);
    }

    public void lock(String key, Integer waitTime, Integer leaseTime, Runnable runnable) {
        lock(key, waitTime, leaseTime, () -> {
            runnable.run();
            return null;
        });
    }

    public <T> T lock(String key, Supplier<T> supplier) {
        return lock(key, 10, 20, supplier);
    }

    public <T> T lock(String key, Integer waitTime, Integer leaseTime, Supplier<T> supplier) {
        RLock lock = redissonClient.getLock("LOCK-" + key);
        boolean isLock;
        try {
            isLock = lock.tryLock(waitTime, leaseTime, TimeUnit.SECONDS);
            if (isLock) {
                return supplier.get();
            } else {
                throw new RuntimeException("操作失败，该数据受锁保护中。");
            }
        } catch (InterruptedException e) {
            throw new RuntimeException("锁" + lock.getName() + "获取失败", e);
        } finally {
            if (lock.isHeldByCurrentThread()) {
                lock.unlock();
            }
        }
    }


    public <T> T get(String key) {
        RBucket<T> bucket = redissonClient.getBucket(key, JsonJacksonCodec.INSTANCE);
        return bucket.get();
    }

    public boolean hasKey(String key) {
        return redissonClient.getBucket(key, JsonJacksonCodec.INSTANCE).isExists();
    }

    public void delete(String key) {
        redissonClient.getBucket(key, JsonJacksonCodec.INSTANCE).delete();
    }

    public void deleteList(Iterable<Object> keys) {
        keys.forEach(item -> delete(item.toString()));
    }

    public void deleteList(JSONArray keys) {
        deleteList(keys.toList());
    }
}
