package com.af.v4.system.common.redis;

import org.json.JSONObject;
import org.redisson.api.RBucket;
import org.redisson.api.RLock;
import org.redisson.api.RMap;
import org.redisson.api.RedissonClient;
import org.redisson.codec.JsonJacksonCodec;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * Redis服务
 *
 * @author Mr.river
 */
@Component
public class RedisService {

    private static final Logger LOGGER = LoggerFactory.getLogger(RedisService.class);
    /**
     * redisson客户端
     */
    private final RedissonClient REDISSON_CLIENT;

    public RedisService(RedissonClient REDISSON_CLIENT) {
        this.REDISSON_CLIENT = REDISSON_CLIENT;
    }

    public <T> void set(String key, T value) {
        set(key, value, -1);
    }

    public <T> void set(String key, T value, long timeoutSec) {
        RBucket<T> rBucket = REDISSON_CLIENT.getBucket(key, JsonJacksonCodec.INSTANCE);
        if (timeoutSec > 0) {
            rBucket.set(value, timeoutSec, TimeUnit.SECONDS);
        } else {
            rBucket.set(value);
        }
    }

    public <T> void setHash(String hashKey, Map<String, T> value) {
        setHash(hashKey, value, -1);
    }

    public <T> void setHash(String hashKey, Map<String, T> value, long timeoutSec) {
        RMap<String, T> rMap = REDISSON_CLIENT.getMap(hashKey);
        rMap.putAll(value);
        if (timeoutSec > 0) {
            rMap.expire(Duration.ofSeconds(timeoutSec));
        }
    }

    public JSONObject getHash(String hashKey) {
        return new JSONObject(REDISSON_CLIENT.getMap(hashKey));
    }

    public <T> T getHash(String hashKey, String key) {
        RMap<String, T> map = REDISSON_CLIENT.getMap(hashKey);
        return map.get(key);
    }

    public <T> void setHashKey(String hashKey, String key, T value) {
        RMap<String, T> map = REDISSON_CLIENT.getMap(hashKey);
        map.put(key, value);
    }

    public void deleteHashKey(String hashKey, String key) {
        REDISSON_CLIENT.getMap(hashKey).remove(key);
    }

    public Boolean hasHashKey(String hashKey, String key) {
        return REDISSON_CLIENT.getMap(hashKey).containsKey(key);
    }

    public void lock(String key, Runnable runnable) {
        lock(key, 10, 20, runnable);
    }

    public void lock(String key, Integer waitTime, Integer leaseTime, Runnable runnable) {
        RLock lock = REDISSON_CLIENT.getLock("LOCK-" + key);
        boolean isLock;
        try {
            isLock = lock.tryLock(waitTime, leaseTime, TimeUnit.SECONDS);
            if (isLock) {
                runnable.success();
            } else {
                throw new RuntimeException("修改失败，该数据受锁保护中。");
            }
        } catch (InterruptedException e) {
            LOGGER.error("锁" + lock.getName() + "获取失败");
            e.printStackTrace();
        } finally {
            if (lock.isHeldByCurrentThread()) {
                lock.unlock();
            }
        }
    }


    public <T> T get(String key) {
        RBucket<T> bucket = REDISSON_CLIENT.getBucket(key, JsonJacksonCodec.INSTANCE);
        return bucket.get();
    }

    public boolean hasKey(String key) {
        return REDISSON_CLIENT.getBucket(key, JsonJacksonCodec.INSTANCE).isExists();
    }

    public void delete(String key) {
        REDISSON_CLIENT.getBucket(key, JsonJacksonCodec.INSTANCE).delete();
    }

    public void deleteList(Iterable<String> keys) {
        keys.forEach(this::delete);
    }

    public void deleteList(JSONObject keys) {
        deleteList(new ArrayList<>(keys.keySet()));
    }

    @FunctionalInterface
    public interface Runnable {
        void success();
    }
}
