package com.af.v4.system.common.plugins.json;

import com.af.v4.system.common.plugins.core.CommonTools;
import com.af.v4.system.common.plugins.io.IOTools;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.dataformat.xml.XmlMapper;
import com.fasterxml.jackson.dataformat.xml.ser.ToXmlGenerator;
import org.json.JSONArray;
import org.json.JSONObject;
import org.json.JSONParserConfiguration;

import java.io.IOException;
import java.math.BigDecimal;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * json工具类
 *
 * @author Mr.river
 */
public class JsonTools {

    private static final ObjectMapper objectMapper;

    static {
        objectMapper = new ObjectMapper();
        objectMapper.setSerializationInclusion(JsonInclude.Include.NON_EMPTY);
        objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
    }

    /**
     * 给一个JSONObject追加一个JSONObject中的所有值(deep clone)
     *
     * @param obj       原始 JSONObject
     * @param addObj    要合并的 JSONObject
     * @param modifyObj 是否直接修改原始 obj
     * @param fields    筛选 addObj 中需要合并的 fields
     * @return 合并后的 JSONObject。如果 modifyObj 为 true，则返回修改后的原始对象；否则返回新的对象。
     */
    public static JSONObject addJSON(JSONObject obj, JSONObject addObj, Boolean modifyObj, JSONArray fields) {
        if (addObj == null) {
            throw new NullPointerException("追加的JSONObject不存在！");
        }

        Set<String> fieldSet;

        if (fields == null || fields.isEmpty()) {
            fieldSet = addObj.keySet();
        } else {
            fieldSet = fields.toList().stream()
                    .map(Object::toString)
                    .collect(Collectors.toSet());
        }

        JSONObject targetObj = modifyObj ? obj : new JSONObject(obj.toMap());

        for (String key : fieldSet) {
            Object item = addObj.opt(key);
            switch (item) {
                case null -> {
                }
                case JSONObject json -> {
                    JSONObject cloneItem = new JSONObject(json.toMap());
                    targetObj.put(key, cloneItem);
                }
                case JSONArray array -> {
                    JSONArray cloneItem = new JSONArray(array.toList());
                    targetObj.put(key, cloneItem);
                }
                default -> targetObj.put(key, item);
            }
        }
        return targetObj;
    }

    public static JSONObject addJSON(JSONObject obj, JSONObject addObj, Boolean modifyObj) {
        return addJSON(obj, addObj, modifyObj, null);
    }

    public static JSONObject addJSON(JSONObject obj, JSONObject addObj, JSONArray fields) {
        return addJSON(obj, addObj, true, fields);
    }

    public static JSONObject addJSON(JSONObject obj, JSONObject addObj) {
        return addJSON(obj, addObj, true);
    }

    /**
     * 将字符串转化为json
     *
     * @param str 要转为json的字符串
     * @return 原生json
     */
    public static JSONObject convertToJson(String str) {
        return convertToJson(str, false);
    }

    public static JSONObject convertToJson(String str, Boolean isOrdered) {
        if (str == null || str.isEmpty()) {
            return new JSONObject();
        }
        if (isOrdered) {
            return new JSONObject(str, new JSONParserConfiguration().setOrdered(true));
        } else {
            return new JSONObject(str);
        }
    }

    public static JSONObject convertToJson(JSONObject json) {
        return json;
    }

    public static JSONObject getOrderedJson() {
        return new JSONObject(new JSONParserConfiguration().setOrdered(true));
    }

    public static String jsonConvertToXml(JSONObject json, String rootName, boolean writeXmlDeclaration, String charset) {
        try {
            ObjectMapper jsonMapper = new ObjectMapper();
            XmlMapper xmlMapper = new XmlMapper();

            // 不让 XmlMapper 自动写 XML 声明，我们自己控制
            xmlMapper.configure(ToXmlGenerator.Feature.WRITE_XML_DECLARATION, false);
            xmlMapper.configure(ToXmlGenerator.Feature.UNWRAP_ROOT_OBJECT_NODE, true);

            // 包装 JSON，使之以指定根节点为名
            JSONObject toXmlJson = new JSONObject();
            toXmlJson.put(rootName, json);

            JsonNode jsonNode = jsonMapper.readTree(toXmlJson.toString());

            // 将 JsonNode 转换为格式化的 XML 字符串（不含声明）
            String xmlBody = xmlMapper.writerWithDefaultPrettyPrinter().writeValueAsString(jsonNode);

            // 手动拼接 XML 声明
            if (writeXmlDeclaration) {
                return "<?xml version=\"1.0\" encoding=\"" + charset + "\"?>\n" + xmlBody;
            } else {
                return xmlBody;
            }

        } catch (Exception e) {
            throw new RuntimeException("JSON 转换为 XML 过程中发生错误", e);
        }
    }

    public static String jsonConvertToXml(JSONObject json, String rootName, boolean writeXmlDeclaration) {
        return jsonConvertToXml(json, rootName, writeXmlDeclaration, "UTF-8");
    }

    /**
     * 将 JSON 转换为 XML 字符串。
     *
     * @param json 原生JSON。
     * @return XML 字符串。
     */
    public static String jsonConvertToXml(JSONObject json) {
        try {
            ObjectMapper jsonMapper = new ObjectMapper();
            XmlMapper xmlMapper = new XmlMapper();
            // 配置 XmlMapper 控制是否写入 XML 声明
            xmlMapper.configure(ToXmlGenerator.Feature.WRITE_XML_DECLARATION, true);
            // 配置 XmlMapper 控制是否展开根对象节点
            xmlMapper.configure(ToXmlGenerator.Feature.UNWRAP_ROOT_OBJECT_NODE, true);
            JSONObject toXmlJson = new JSONObject();
            toXmlJson.put("xml", json);
            JsonNode jsonNode = jsonMapper.readTree(toXmlJson.toString());
            // 将 JsonNode 转换为格式化的 XML 字符串
            return xmlMapper.writerWithDefaultPrettyPrinter().writeValueAsString(jsonNode);
        } catch (Exception e) {
            throw new RuntimeException("JSON 转换为 XML 过程中发生错误", e);
        }
    }

    /**
     * 将 XML 字符串转换为 JSON 字符串。
     *
     * @param xmlStr XML 字符串。
     * @return 原生JSON。
     */
    public static JSONObject xmlConvertToJson(String xmlStr, String charset) {
        try {
            XmlMapper xmlMapper = new XmlMapper();
            ObjectMapper jsonMapper = new ObjectMapper();
            JsonNode xmlNode = xmlMapper.readTree(xmlStr.getBytes(charset));
            return new JSONObject(jsonMapper.writeValueAsString(xmlNode));
        } catch (IOException e) {
            throw new RuntimeException("XML 转换为 JSON 过程中发生错误", e);
        }
    }

    public static JSONObject xmlConvertToJson(String xmlStr) {
        String defaultCharset = java.nio.charset.Charset.defaultCharset().name();
        return xmlConvertToJson(xmlStr, defaultCharset);
    }

    /**
     * 把字符串转换成JSON数组
     *
     * @param str 需要转换的String
     * @return 转换完成后的json数组
     */
    public static JSONArray parseArray(String str) {
        return new JSONArray(str);
    }

    public static JSONArray parseArray(JSONArray json) {
        return json;
    }

    /**
     * 返回一个JSON数组实例
     *
     * @return JSON数组实例
     */
    public static JSONArray getArray() {
        return new JSONArray();
    }

    /**
     * 返回一个制定长度的JSON数组实例
     *
     * @param val 长度
     * @return JSONArray
     */
    public static JSONArray getArray(Object val) {
        int len = Integer.parseInt(val.toString());
        return new JSONArray(new String[len]);
    }

    /**
     * 返回一个JSON实例
     *
     * @return JSON数组实例
     */
    public static JSONObject getJson() {
        return new JSONObject();
    }

    /**
     * JSON键值对替换（保留原JSON中的键值对）
     *
     * @param oldObj 被替换的JSON
     * @param newObj 替换的JSON
     * @return 处理后的JSON
     */
    public static JSONObject replaceJSON(JSONObject oldObj, JSONObject newObj) {
        for (String key : newObj.keySet()) {
            //如果新JSON中有原JSON的键值对
            oldObj.put(key, newObj.get(key));
        }
        return oldObj;
    }

    /**
     * 读取JSON文件，返回JSON对象
     *
     * @param path 文件路径
     * @return JSONObject
     */
    public static JSONObject readJsonFile(String path) {
        return new JSONObject(IOTools.readText(path));
    }

    public static JSONObject readJsonFile(String path, Boolean isOrdered) {
        if (isOrdered) {
            return new JSONObject(IOTools.readText(path), new JSONParserConfiguration().setOrdered(true));
        } else {
            return new JSONObject(IOTools.readText(path));
        }
    }


    /**
     * 读取JSON文件，返回JSON数组
     *
     * @param path 文件路径
     * @return JSONArray
     */
    public static JSONArray readJsonArrayFile(String path) {
        return new JSONArray(IOTools.readText(path));
    }

    /**
     * 根据参数字典格式化Json
     *
     * @param object Json
     * @param params 参数字典
     * @return 格式化后的Json
     */
    public static JSONObject formatJsonByParams(JSONObject object, JSONObject params) {
        if (!object.isEmpty()) {
            return new JSONObject(CommonTools.formatStrByParams(object.toString(), params));
        }
        return object;
    }

    /**
     * 根据参数字典格式化Json集合
     *
     * @param array  Json集合
     * @param params 参数字典
     * @return 格式化后的Json集合
     */
    public static JSONArray formatJsonArrayByParams(JSONArray array, JSONObject params) {
        if (!array.isEmpty()) {
            return new JSONArray(CommonTools.formatStrByParams(array.toString(), params));
        }
        return array;
    }

    /**
     * 将json转化为实体POJO
     *
     * @param json org.json
     * @param obj  转换的实体类
     * @return 实体POJO
     */
    public static <T> T toParse(JSONObject json, Class<T> obj) throws JsonProcessingException {
        return toParse(json.toString(), obj);
    }

    public static <T> T toParse(String json, Class<T> obj) throws JsonProcessingException {
        return objectMapper.readValue(json, obj);
    }

    public static <T> List<T> toParseList(JSONArray json, Class<T> obj) throws JsonProcessingException {
        return objectMapper.readValue(json.toString(), getCollectionType(List.class, obj));
    }

    /**
     * 将实体POJO转换为json
     *
     * @param entity 被转换的实体类
     * @return org.json
     */
    public static JSONObject toJSON(Object entity) throws JsonProcessingException {
        return new JSONObject(objectMapper.writeValueAsString(entity));
    }

    public static <T> JSONArray toJsonArray(List<T> entity) throws JsonProcessingException {
        return new JSONArray(objectMapper.writeValueAsString(entity));
    }

    /**
     * 获取泛型的Collection Type
     *
     * @param collectionClass 泛型的Collection
     * @param elementClasses  实体bean
     * @return JavaType Java类型
     */
    private static JavaType getCollectionType(Class<?> collectionClass, Class<?>... elementClasses) {
        return objectMapper.getTypeFactory().constructParametricType(collectionClass, elementClasses);
    }

    /**
     * 对象转json
     *
     * @param obj 任意类型对象(需要 getter 和 setter 方法)
     * @return json对象
     */
    public static <T> JSONObject objToJson(T obj) throws JsonProcessingException {
        return new JSONObject(objectMapper.writeValueAsString(obj));
    }

    /**
     * 获取JSONObject实例中Key的集合
     *
     * @param object JSONObject实例
     * @return Key集合
     */
    public static JSONArray getKeys(JSONObject object) {
        return new JSONArray(object.keySet());
    }

    /**
     * 包装结算场景JSON
     * 应用场景：对于售气收费，为匹配气价的计算，将json字符串进行相应的包装
     * 未包装前:{"data":{"price":"1","amount":"12"}}
     * 包装后:{"data":[{"price":"1","amount":"12"}]}
     *
     * @param json      json对象
     * @param priceType 气价类型
     * @return 包装后的json数组
     */
    public static JSONArray packSettleJson(JSONObject json, String priceType) {
        JSONArray result = new JSONArray();
        if ("fixed".equals(priceType)) {
            JSONObject value = new JSONObject();
            value.put("f_unitprice", new BigDecimal(String.valueOf(json.get("f_unitprice"))));
            result.put(value);
        } else if ("mixed".equals(priceType)) {
            JSONArray array = json.getJSONArray("detailprice");
            JSONObject obj = new JSONObject();
            obj.put("f_hybridprice1",
                    new BigDecimal(String.valueOf(array.getJSONObject(0).get("f_price"))));
            obj.put("f_hybridprice2",
                    new BigDecimal(String.valueOf(array.getJSONObject(1).get("f_price"))));
            obj.put("f_hybridprice1rate",
                    new BigDecimal(String.valueOf(array.getJSONObject(0).get("f_ratio"))));
            result.put(obj);
        } else if ("staired".equals(priceType)) {
            JSONArray array = json.getJSONArray("detailprice");
            for (Object object : array) {
                JSONObject obj = new JSONObject();
                obj.put("price", new BigDecimal(String.valueOf(((JSONObject) object).get("f_price"))));
                obj.put("amount", new BigDecimal(String.valueOf(((JSONObject) object).get("f_amount"))));
                result.put(obj);
            }
            JSONObject obj = new JSONObject();
            obj.put("sumamount", json.get("sumamount"));
            result.put(obj);
        }
        return result;
    }

    /**
     * 判断JSONObject对象中是否包含key
     *
     * @param obj json对象
     * @param key 是否包含的key键
     * @return boolean值的返回值
     */
    public static boolean isContains(JSONObject obj, String key) {
        return obj.has(key);
    }

    /**
     * 获取json对象中的key
     *
     * @param obj json对象
     * @return key的数组
     */
    public static JSONArray getJsonKeys(JSONObject obj) {
        return new JSONArray(obj.keySet());
    }

    /**
     * JSONObject中key是否为空
     *
     * @param obj json对象
     * @param key 需要查看的key
     * @return 返回的boolean值
     */
    public static boolean isNull(JSONObject obj, String key) {
        return obj.isNull(key);
    }

    /**
     * JSONObject是否为空
     *
     * @param obj 查看的json对象
     * @return 返回boolean
     */
    public static boolean isNull(JSONObject obj) {
        return obj.isEmpty();
    }

    /**
     * 对参数进行验证
     *
     * @param obj    传入Json对象
     * @param fields 传入一个对象
     * @throws Exception 异常
     */
    public static void validateField(JSONObject obj, Object fields) throws Exception {
        String errors = null;

        for (Object s : fields instanceof JSONObject ? new JSONObject(fields.toString()).keySet().toArray() : ((String) fields).split(",")) {
            if ("".equals(obj.get((String) s)) || (obj.get((String) s)) == null) {
                errors = (errors == null ? s : errors + s) + " ";
            }
        }
        if (errors != null) {
            throw new Exception(errors + "字段值不能为空！！！");
        }
    }

    /**
     * 从jo中移除colsToBeRemoved中的列
     *
     * @param colsToBeRemoved json对象
     * @param jo              json对象
     * @return 移除完返回jo对象
     */
    public static JSONObject remove(JSONObject colsToBeRemoved, JSONObject jo) {
        Set<String> keySet1 = colsToBeRemoved.keySet();
        for (String s : keySet1) {
            jo.remove(s);
        }
        return jo;
    }

    /**
     * 给一个JSONObject中添加对象
     *
     * @param obj      json对象
     * @param addLabel 键
     * @param addVale  值
     * @return 添加完成后返回obj
     */
    public static JSONObject add(JSONObject obj, String addLabel, Object addVale) {
        obj.put(addLabel, addVale);
        return obj;
    }

    /**
     * JSONObject串删除id
     *
     * @param obj json对象
     * @param key obj中需要移除的键
     * @return 返回obj
     */
    public static JSONObject removeKey(JSONObject obj, String key) {
        obj.remove(key);
        return obj;
    }

    /**
     * 这个方法是将json转换成Map
     *
     * @param object json对象
     * @return 转换后的map
     */
    public static Map<String, Object> toHashMap(JSONObject object) {
        return object.toMap();
    }

    /**
     * 比较两个JSONObject,  返回的JSONArray格式为[{field:xxx, oldVal:xxx, newVal: xxx}]
     *
     * @param jo1 新值
     * @param jo2 旧值
     * @return JSONArray
     */
    public static JSONArray findDiff(JSONObject jo1, JSONObject jo2) {
        JSONArray jsonArray;
        Set<String> keySet1 = jo1.keySet();
        jsonArray = new JSONArray();
        for (String s : keySet1) {
            if (isValueEquals(jo1, jo2, s)) {
                JSONObject jo = new JSONObject();
                jo.put("field", s);
                jo.put("newVal", jo1.get(s) == null ? null : jo1.get(s)
                        .toString());
                jo.put("oldVal", jo2.get(s) == null ? null : jo2.get(s)
                        .toString());
                jsonArray.put(jo);
            }
        }
        return jsonArray;
    }

    /**
     * 判断两个json中值是否相同
     *
     * @param jo1 新值
     * @param jo2 旧值
     * @return 判断结果
     */
    public static boolean isValueEquals(JSONObject jo1, JSONObject jo2, String s) {
        // 针对级联json数据
        if ((jo1.isNull(s) && !jo2.isNull(s))
                || (!jo1.isNull(s) && jo2.isNull(s))) {
            return false;
        }
        if (jo1.isNull(s) && (jo2.isNull(s) || "null".equals(jo2.get(s)))) {
            return false;
        }
        if (jo1.get(s) instanceof Integer) {
            if (jo2.get(s) instanceof BigDecimal) {
                return new BigDecimal((Integer) jo1.get(s))
                        .compareTo((BigDecimal) jo2.get(s)) != 0;
            } else {
                try {
                    return new BigDecimal((Integer) jo1.get(s))
                            .compareTo(new BigDecimal(Integer.parseInt(jo2
                                    .get(s).toString()))) != 0;
                } catch (Exception e) {
                    return false;
                }
            }
        }
        if (jo1.get(s) instanceof String) {
            if (jo2.get(s) instanceof String) {
                return !jo1.get(s).equals(jo2.get(s));
            } else {
                try {
                    // 界面传过来的是字符串，库里拿到的是BigDecimal类型
                    return BigDecimal.valueOf(Double.parseDouble(jo1.get(s).toString()))
                            .compareTo((BigDecimal) jo2.get(s)) != 0;
                } catch (Exception e) {
                    return false;
                }
            }
        }
        if (jo1.get(s) instanceof Double) {
            if (jo2.get(s) instanceof Integer) {
                return BigDecimal.valueOf((Double) jo1.get(s))
                        .compareTo(new BigDecimal((Integer) jo2.get(s))) != 0;
            } else {
                try {
                    return BigDecimal.valueOf((Double) jo1.get(s))
                            .compareTo(BigDecimal.valueOf(Double.parseDouble(jo2
                                    .get(s).toString()))) != 0;
                } catch (Exception e) {
                    return false;
                }
            }
        }
        return !(jo2.get(s).equals(jo1.get(s)));
    }

    /**
     * 应用场景：对于售气收费，为匹配气价的计算，将json字符串进行相应的包装
     * 未包装前:{"data":{"price":"1","amount":"12"}}
     * 包装后:{"data":[{"price":"1","amount":"12"}]}
     *
     * @param json      json对象
     * @param priceType 气价类型
     * @return 包装后的json数组
     * @see #packSettleJson(JSONObject, String)
     */
    @Deprecated
    public static JSONArray packJson(JSONObject json, String priceType) {
        return packSettleJson(json, priceType);
    }

    /**
     * 把对象数组里的某个字段内容，转换成'name1', 'name2'的形式
     *
     * @param array 对象数组
     * @param field String
     * @return 转换后的String
     */
    public static String toString(JSONArray array, String field) {
        StringBuilder result = new StringBuilder();
        for (Object object : array) {
            JSONObject obj = (JSONObject) object;
            String value = obj.optString(field);
            if (!"".contentEquals(result)) {
                result.append(",");
            }
            result.append("'").append(value).append("'");
        }
        return result.toString();
    }

    /**
     * @param array    对象数组
     * @param pageSize Integer
     * @return JSONArray
     */
    public static JSONArray page(JSONArray array, Integer pageSize) {
        JSONArray result = new JSONArray();
        int size = array.length();
        int pageCount = (int) Math.ceil((double) size / pageSize);
        for (int i = 0; i < pageCount; i++) {
            JSONArray page = new JSONArray();
            // 计算当前页的结束索引，取最小值是为了防止索引越界
            int end = Math.min((i + 1) * pageSize, size);
            for (int j = i * pageSize; j < end; j++) {
                page.put(array.get(j));
            }
            result.put(page);
        }
        return result;
    }

    // 对字段key排序之后重新构建JSONObject
    public static JSONObject sortJSONObjectByDescendingKey(JSONObject json) {
        Map<String, Object> sortedMap = json.keySet().stream()
                .sorted(String::compareTo)
                .collect(Collectors.toMap(key -> key, json::get, (oldValue, newValue) -> oldValue, LinkedHashMap::new));
        return new JSONObject(sortedMap);
    }
}
