package com.af.v4.system.common.plugins.core;

import org.json.JSONArray;
import org.json.JSONObject;

import java.math.BigDecimal;
import java.text.NumberFormat;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

/**
 * 金额格式化工具类
 *
 * @author Mr.river
 */
public class PriceFormatTools {
    static final NumberFormat NF;
    static final NumberFormat NF_TWO_DECIMAL;

    static {
        // 4位小数格式化
        NF = NumberFormat.getInstance();
        NF.setGroupingUsed(false);
        // 最小有效位数: 2
        NF.setMinimumFractionDigits(2);
        // 最大有效位数: 4
        NF.setMaximumFractionDigits(4);
        
        // 2位小数格式化
        NF_TWO_DECIMAL = NumberFormat.getInstance();
        NF_TWO_DECIMAL.setGroupingUsed(false);
        // 最小有效位数: 2
        NF_TWO_DECIMAL.setMinimumFractionDigits(2);
        // 最大有效位数: 2
        NF_TWO_DECIMAL.setMaximumFractionDigits(2);
    }

    /**
     * jsonObject 批量格式化（4位小数）
     *
     * @param object 待格式化 {@code jsonObject} 对象
     */
    public static void formatNumber(JSONObject object) {
        formatNumberWithFormat(object, NF);
    }

    /**
     * jsonArray批量格式化（4位小数）
     *
     * @param object 待格式化 {@code jsonArray} 对象
     */
    public static void formatNumber(JSONArray object) {
        formatNumberWithFormat(object, NF);
    }

    /**
     * jsonObject 批量格式化（2位小数）
     *
     * @param object 待格式化 {@code jsonObject} 对象
     */
    public static void formatNumberTwoDecimal(JSONObject object) {
        formatNumberWithFormat(object, NF_TWO_DECIMAL);
    }

    /**
     * jsonArray批量格式化（2位小数）
     *
     * @param object 待格式化 {@code jsonArray} 对象
     */
    public static void formatNumberTwoDecimal(JSONArray object) {
        formatNumberWithFormat(object, NF_TWO_DECIMAL);
    }

    /**
     * jsonObject 批量格式化（4位小数）- 排除指定字段
     *
     * @param object 待格式化 {@code jsonObject} 对象
     * @param excludeFields 要排除的字段名，可变参数
     */
    public static void formatNumber(JSONObject object, String... excludeFields) {
        formatNumberWithFormat(object, NF, excludeFields);
    }

    /**
     * jsonArray批量格式化（4位小数）- 排除指定字段
     *
     * @param object 待格式化 {@code jsonArray} 对象
     * @param excludeFields 要排除的字段名，可变参数
     */
    public static void formatNumber(JSONArray object, String... excludeFields) {
        formatNumberWithFormat(object, NF, excludeFields);
    }

    /**
     * jsonObject 批量格式化（2位小数）- 排除指定字段
     *
     * @param object 待格式化 {@code jsonObject} 对象
     * @param excludeFields 要排除的字段名，可变参数
     */
    public static void formatNumberTwoDecimal(JSONObject object, String... excludeFields) {
        formatNumberWithFormat(object, NF_TWO_DECIMAL, excludeFields);
    }

    /**
     * jsonArray批量格式化（2位小数）- 排除指定字段
     *
     * @param object 待格式化 {@code jsonArray} 对象
     * @param excludeFields 要排除的字段名，可变参数
     */
    public static void formatNumberTwoDecimal(JSONArray object, String... excludeFields) {
        formatNumberWithFormat(object, NF_TWO_DECIMAL, excludeFields);
    }

    /**
     * 使用指定格式化器格式化JSONObject
     *
     * @param object 待格式化 {@code jsonObject} 对象
     * @param formatter 格式化器
     */
    private static void formatNumberWithFormat(JSONObject object, NumberFormat formatter) {
        formatNumberWithFormat(object, formatter, new String[0]);
    }

    /**
     * 使用指定格式化器格式化JSONArray
     *
     * @param object 待格式化 {@code jsonArray} 对象
     * @param formatter 格式化器
     */
    private static void formatNumberWithFormat(JSONArray object, NumberFormat formatter) {
        formatNumberWithFormat(object, formatter, new String[0]);
    }

    /**
     * 使用指定格式化器格式化JSONObject - 排除指定字段
     *
     * @param object 待格式化 {@code jsonObject} 对象
     * @param formatter 格式化器
     * @param excludeFields 要排除的字段名
     */
    private static void formatNumberWithFormat(JSONObject object, NumberFormat formatter, String[] excludeFields) {
        Set<String> excludeSet = new HashSet<>(Arrays.asList(excludeFields));
        Iterator<String> iterator = object.keys();
        while (iterator.hasNext()) {
            String key = iterator.next();
            Object value = object.get(key);
            if (value instanceof BigDecimal) {
                if (excludeSet.contains(key)) {
                    continue;
                }
                object.put(key, formatNumberBase(value, formatter));
            } else if (value instanceof JSONObject) {
                formatNumberWithFormat((JSONObject) value, formatter, excludeFields);
            } else if (value instanceof JSONArray) {
                formatNumberWithFormat((JSONArray) value, formatter, excludeFields);
            }
        }
    }

    /**
     * 使用指定格式化器格式化JSONArray - 排除指定字段
     *
     * @param object 待格式化 {@code jsonArray} 对象
     * @param formatter 格式化器
     * @param excludeFields 要排除的字段名
     */
    private static void formatNumberWithFormat(JSONArray object, NumberFormat formatter, String[] excludeFields) {
        Iterator<Object> iterator = object.iterator();
        int index = 0;
        while (iterator.hasNext()) {
            Object value = iterator.next();
            if (value instanceof BigDecimal) {
                object.put(index, formatNumberBase(value, formatter));
            } else if (value instanceof JSONObject) {
                formatNumberWithFormat((JSONObject) value, formatter, excludeFields);
            } else if (value instanceof JSONArray) {
                formatNumberWithFormat((JSONArray) value, formatter, excludeFields);
            }
            index++;
        }
    }

    /**
     * 格式化对象 - 会把所有的对象转化为字符串 - 不建议使用
     * 使用4位小数
     *
     * @param value 待格式化对象
     * @return 格式化字符串
     */
    public static String formatNumberBase(Object value) {
        return NF.format(value);
    }

    /**
     * 格式化对象 - 使用指定格式化器
     *
     * @param value 待格式化对象
     * @param formatter 格式化器
     * @return 格式化字符串
     */
    private static String formatNumberBase(Object value, NumberFormat formatter) {
        return formatter.format(value);
    }
    
    /**
     * 格式化对象（2位小数）- 会把所有的对象转化为字符串 - 不建议使用
     *
     * @param value 待格式化对象
     * @return 格式化字符串
     */
    public static String formatNumberBaseTwoDecimal(Object value) {
        return NF_TWO_DECIMAL.format(value);
    }
}
