package com.af.v4.system.common.plugins.core;

import cn.hutool.core.lang.UUID;
import cn.hutool.core.util.RandomUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.crypto.digest.MD5;
import org.apache.commons.codec.digest.DigestUtils;
import org.json.JSONArray;
import org.json.JSONObject;

import java.io.UnsupportedEncodingException;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Set;

/**
 * 常用工具类
 *
 * @author Mr.river
 */
public class CommonTools {
    /**
     * 获取一个类的实例（反射）
     *
     * @param className 类名（com.af.plugins.CommonTools）
     * @param param     构造方法参数
     * @return 实例
     */
    public static Object getInstance(String className, JSONArray param) {
        Class<?> classInstance;
        try {
            //加载类
            classInstance = Class.forName(className);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
        if (param != null && !param.isEmpty()) {
            //参数数组
            Object[] objects = new Object[param.length()];
            //参数类型数组
            Class<?>[] paramClasses = new Class[param.length()];
            for (int i = 0; i < param.length(); i++) {
                //获取参数
                objects[i] = param.get(i);
                //获取参数类型
                Class<?> clazz = objects[i].getClass();
                try {
                    //如果是基本数据类型
                    Field f = clazz.getDeclaredField("TYPE");
                    paramClasses[i] = ((Class<?>) f.get(null));
                } catch (NoSuchFieldException e) {
                    //如果是类类型
                    paramClasses[i] = clazz;
                } catch (IllegalAccessException e) {
                    throw new RuntimeException(e);
                }
            }
            //根据构造方法获取实例
            try {
                Constructor<?> constructor = classInstance.getDeclaredConstructor(paramClasses);
                constructor.setAccessible(true);
                return constructor.newInstance(objects);
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        } else {
            try {
                Constructor<?> constructor = classInstance.getDeclaredConstructor();
                constructor.setAccessible(true);
                return constructor.newInstance();
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        }
    }

    /**
     * 获取一个类的实例（无参）
     *
     * @param className 类名（com.af.plugins.CommonTools）
     * @return 实例
     */
    public static Object getInstance(String className) {
        return getInstance(className, null);
    }

    /**
     * 获取一个UUID
     *
     * @param isSimple 为true时，UUID没有"-"分隔
     * @return UUID字符串
     */
    public static String getUUID(Boolean isSimple) {
        return UUID.fastUUID().toString(isSimple);
    }

    /**
     * 获取一个用"-"分隔的UUID
     *
     * @return UUID字符串
     */
    public static String getUUID() {
        return getUUID(false);
    }

    /**
     * MD5加密
     *
     * @param param 需要加密的参数
     * @return MD5加密后的字符串
     */
    public static String md5(String param) {
        return MD5.create().digestHex(param);
    }

    /**
     * MD5加密（加盐）
     *
     * @param param 需要加密的参数
     * @param salt  盐值
     * @return MD5加密后的字符串
     */
    public static String md5(String param, String salt) {
        return MD5.create().setSalt(salt.getBytes(StandardCharsets.UTF_8)).digestHex(param);
    }

    /**
     * MD5加签。
     *
     * @param text    要加签的字符串
     * @param key     秘钥
     * @param charset 编码格式
     * @param length  位数 16或32
     * @param caseSensitive 是否大小写敏感 如果为 true，则返回大写形式的 MD5 哈希值，否则返回小写形式。
     * @return 加签后的字符串
     */
    public static String MD5Sign(String text, String key, String charset, int length, boolean caseSensitive) {
        try {
            String signText = text + key;
            String md5Hex = DigestUtils.md5Hex(signText.getBytes(charset));

            if (length > 0 && length < 32) {
                md5Hex = md5Hex.substring(0, length);
            }
            return caseSensitive ? md5Hex.toUpperCase() : md5Hex.toLowerCase();
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException("MD5加签过程中出现错误", e);
        }
    }

    /**
     * MD5 验签方法
     *
     * @param content  原始字符串
     * @param key      秘钥
     * @param charset  编码格式
     * @param length   位数
     * @param upperCase 是否大写
     * @param sign     待验证的签名
     * @return 验签结果
     */
    public static boolean MD5Verify(String content, String key, String charset, int length, boolean upperCase, String sign) {
        String calculatedSign = MD5Sign(content, key, charset, length, upperCase);
        return calculatedSign.equals(sign);
    }



    /**
     * 获取一个指定范围的伪随机数
     *
     * @param min 最小值(包含)
     * @param max 最大值(包含)
     * @return 伪随机数
     */
    public static int getRandomNumber(Integer min, Integer max) {
        return RandomUtil.randomInt(min, max + 1);
    }

    /**
     * 判断传入字符是否为数字
     *
     * @param str 传入字符
     * @return 是否为数字
     */
    public static boolean isNumeric(String str) {
        return NumberUtil.isNumeric(str);
    }

    /**
     * 精确加法运算
     *
     * @param o1 被加数
     * @param o2 加数
     * @return 和
     */
    public static BigDecimal add(Object o1, Object o2) {
        return CalculateUtil.add(o1, o2);
    }

    /**
     * 精确加法运算(带精度)
     * WARN：对目标按照精度直接进行截取，即向下舍入操作
     *
     * @param scale 精度
     * @param o1    被加数
     * @param o2    加数
     * @return 和
     */
    public static BigDecimal add(Object o1, Object o2, Integer scale) {
        return CalculateUtil.add(o1, o2, scale);
    }

    /**
     * 精确减法运算
     *
     * @param o1 被减数
     * @param o2 减数
     * @return 差
     */
    public static BigDecimal sub(Object o1, Object o2) {
        return CalculateUtil.sub(o1, o2);
    }

    /**
     * 精确减法运算 可以传入精度
     * WARN：对目标按照精度直接进行截取，即向下舍入操作
     *
     * @param scale 精度
     * @param o1    被减数
     * @param o2    减数
     * @return 差
     */
    public static BigDecimal sub(Object o1, Object o2, Integer scale) {
        return CalculateUtil.sub(o1, o2, scale);
    }

    /**
     * 精确乘法运算
     *
     * @param o1 被乘数
     * @param o2 乘数
     * @return 积
     */
    public static BigDecimal mul(Object o1, Object o2) {
        return CalculateUtil.mul(o1, o2);
    }

    /**
     * 精确乘法运算 可以传入精度
     * WARN：对目标按照精度直接进行截取，即向下舍入操作
     *
     * @param scale 精度
     * @param o1    被乘数
     * @param o2    乘数
     * @return 积
     */
    public static BigDecimal mul(Object o1, Object o2, Integer scale) {
        return CalculateUtil.mul(o1, o2, scale);
    }

    /**
     * 提供(相对)精确的除法运算,当发生除不尽的情况的时候,精确到小数点后10位,后面的四舍五入
     *
     * @param o1 被除数
     * @param o2 除数
     * @return 商
     */
    public static BigDecimal div(Object o1, Object o2) {
        return CalculateUtil.div(o1, o2);
    }

    /**
     * 提供(相对)精确的除法运算,当发生除不尽的情况时,由scale指定精确度,后面的四舍五入
     *
     * @param o2    对象
     * @param o1    对象
     * @param scale 精度
     * @return 商
     */
    public static BigDecimal div(Object o1, Object o2, Integer scale) {
        return CalculateUtil.div(o1, o2, scale);
    }

    /**
     * 提供(相对)精确的除法运算,当发生除不尽的情况时,由scale指定精确度,后面的向上取整
     *
     * @param o2    对象
     * @param o1    对象
     * @param scale 精度
     * @return 商
     */
    public static BigDecimal divUp(Object o1, Object o2, Integer scale) {
        return CalculateUtil.div(o1, o2, scale, RoundingMode.UP);
    }

    /**
     * 字符串分割
     *
     * @param str       字符串
     * @param separator 以什么方式分割
     * @return JSONArray 集合
     */
    public static JSONArray split(String str, String separator) {
        return new JSONArray(StrUtil.split(str, separator));
    }

    /**
     * Url参数转Json
     *
     * @param urlParams url参数
     * @return JSONObject
     */
    public static JSONObject url2json(String urlParams) {
        JSONObject result = new JSONObject();
        String[] split = urlParams.split("&");
        for (String entry : split) {
            String[] mapEntry = entry.split("=");
            String value = mapEntry.length > 1 ? mapEntry[1] : "";
            result.put(mapEntry[0], value);
        }
        return result;
    }

    /**
     * 指定字符串是否在字符串中出现过
     *
     * @param str       字符串
     * @param searchStr 被查找的字符串
     * @return 是否包含
     */
    public static boolean isContains(String str, String searchStr) {
        return StrUtil.contains(str, searchStr);
    }

    /**
     * 合并一个JSON数组里的所有字符串，形式为：'a','b'
     *
     * @param array 传入一个json数组
     * @return 合并后的字符串
     */

    public static String union(JSONArray array) {
        StringBuilder result = new StringBuilder();
        array.forEach(item -> result.append("'").append(item).append("',"));
        result.deleteCharAt(result.length() - 1);
        return result.toString();
    }

    public static String union(String[] array) {
        StringBuilder result = new StringBuilder();
        for (String item : array) {
            result.append("'").append(item).append("',");
        }
        result.deleteCharAt(result.length() - 1);
        return result.toString();
    }

    public static String union(JSONArray array, String column) {
        StringBuilder result = new StringBuilder();
        array.forEach(item -> {
            if (((JSONObject) item).has(column)) {
                result.append("'").append(((JSONObject) item).get(column)).append("',");
            }
        });
        if (!result.isEmpty()) {
            result.deleteCharAt(result.length() - 1);
            return result.toString();
        } else {
            return null;
        }
    }

    /**
     * 合并一个ArrayList里的所有元素，形式为：'a','b'
     *
     * @param list 传入一个ArrayList
     * @return 合并后的字符串
     */
    public static String union(ArrayList<?> list) {
        if (list == null || list.isEmpty()) {
            return "";
        }

        StringBuilder result = new StringBuilder();
        for (Object item : list) {
            result.append("'").append(item).append("',");
        }
        result.deleteCharAt(result.length() - 1);
        return result.toString();
    }

    /**
     * 根据参数字典格式化字符串
     * <p>
     * Examples:
     * 需要格式化的参数字符串，格式：http://{baseUrl}/rs/logic/{logicName}
     * 参数映射字典，如：key=baseUrl, value=baidu.com ; key=logicName, value=beta
     *
     * @param str    字符串
     * @param params 参数字典
     * @return 格式化后的字符串
     */
    public static String formatStrByParams(String str, JSONObject params) {
        Set<String> keys = params.keySet();
        for (String key : keys) {
            str = str.replace("{" + key + "}", params.get(key).toString());
        }
        return str;
    }

    /**
     * 返回字符串中第一个分隔符前或后的内容
     *
     * @param input 输入字符串
     * @param delimiter 分隔符（字符串）
     * @param after 如果为 true 返回分隔符后的内容，否则返回分隔符前的内容
     * @param keepOriginalIfNotFound 是否在找不到分隔符或分隔符在末尾时返回原字符串
     * @return 分隔符前/后的字符串，或根据 keepOriginalIfNotFound 返回原字符串 / 空串
     */
    public static String substringAroundFirst(String input, String delimiter, boolean after, boolean keepOriginalIfNotFound) {
        if (input == null || delimiter == null || delimiter.isEmpty()) return input;
        int index = input.indexOf(delimiter);
        if (index == -1 || index + delimiter.length() >= input.length()) {
            return keepOriginalIfNotFound ? input : "";
        }
        return after ? input.substring(index + delimiter.length()) : input.substring(0, index);
    }



}
