package com.af.v4.system.common.plugins.core;

import cn.hutool.crypto.SecureUtil;
import cn.hutool.crypto.asymmetric.KeyType;
import cn.hutool.crypto.asymmetric.RSA;

import javax.crypto.Cipher;
import javax.crypto.SecretKey;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import java.nio.charset.StandardCharsets;
import java.security.KeyFactory;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Signature;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.Base64;
import java.util.HashMap;
import java.util.Map;

/**
 * 加解密工具类
 *
 * @author Mr.river
 */
public class SecureTools {

    /**
     * AES加密
     *
     * @param plainText 待加密内容
     * @param key       密钥
     * @return 加密后的字符串
     */
    public static String AESEncrypt(final String plainText, final String key) {
        return SecureUtil.aes(ConvertTools.base64Decode(key.getBytes(StandardCharsets.UTF_8))).encryptBase64(plainText);
    }

    /**
     * AES解密
     *
     * @param cipherText 待解密内容
     * @param key        密钥
     * @return 解密后的字符串
     */
    public static String AESDecrypt(final String cipherText, final String key) {
        return SecureUtil.aes(ConvertTools.base64Decode(key.getBytes(StandardCharsets.UTF_8))).decryptStr(cipherText);
    }


    /**
     * AES解密
     * CBC模式解密
     *
     * @param combinedText 格式：Base64(iv):Base64(ciphertext)
     */
    public static String AESDecryptCBC(String combinedText, String hexKey) {
        try {
            // 分离 IV 和 密文
            String[] parts = combinedText.split(":");
            String ivBase64 = parts[0];
            String encryptedBase64 = parts[1];

            // Base64 解码 IV 和密文
            byte[] iv = Base64.getDecoder().decode(ivBase64);
            byte[] encryptedBytes = Base64.getDecoder().decode(encryptedBase64);

            // 创建 AES 密钥（从十六进制字符串解析）
            SecretKey secretKeySpec = new SecretKeySpec(hexStringToByteArray(hexKey), "AES");

            // 使用 CBC 模式，解密
            IvParameterSpec ivParameterSpec = new IvParameterSpec(iv);
            Cipher cipher = Cipher.getInstance("AES/CBC/PKCS5Padding");
            cipher.init(Cipher.DECRYPT_MODE, secretKeySpec, ivParameterSpec);

            // 解密
            byte[] decryptedBytes = cipher.doFinal(encryptedBytes);

            // 将解密后的字节数组转换为字符串
            return new String(decryptedBytes);
        } catch (Exception e) {
            throw new RuntimeException("解密失败", e);
        }
    }

    // 将十六进制字符串转换为字节数组
    private static byte[] hexStringToByteArray(String s) {
        int len = s.length();
        byte[] data = new byte[len / 2];
        for (int i = 0; i < len; i += 2) {
            data[i / 2] = (byte) ((Character.digit(s.charAt(i), 16) << 4)
                    + Character.digit(s.charAt(i + 1), 16));
        }
        return data;
    }

    /**
     * 生成RSA密钥对（2048位）
     *
     * @return 包含Base64编码公钥和私钥的Map
     */
    public static Map<String, String> generateRSAKeyPair() {
        RSA rsa = SecureUtil.rsa();
        Map<String, String> keyMap = new HashMap<>();
        keyMap.put("publicKey", rsa.getPublicKeyBase64());
        keyMap.put("privateKey", rsa.getPrivateKeyBase64());
        return keyMap;
    }

    /**
     * RSA公钥加密
     *
     * @param plainText 明文
     * @param publicKey Base64编码的公钥
     */
    public static String RSAEncrypt(String plainText, String publicKey) {
        try {
            RSA rsa = SecureUtil.rsa(null, publicKey);
            return rsa.encryptBase64(plainText, StandardCharsets.UTF_8, KeyType.PublicKey);
        } catch (Exception e) {
            throw new RuntimeException("RSA加密失败", e);
        }
    }

    /**
     * RSA私钥解密
     *
     * @param cipherText Base64编码的密文
     * @param privateKey Base64编码的私钥
     */
    public static String RSADecrypt(String cipherText, String privateKey) {
        try {
            RSA rsa = SecureUtil.rsa(privateKey, null);
            return rsa.decryptStr(cipherText, KeyType.PrivateKey);
        } catch (Exception e) {
            throw new RuntimeException("RSA解密失败", e);
        }
    }


    /**
     * 使用私钥签名
     *
     * @param data       原始数据
     * @param privateKey Base64编码的私钥
     */

    public static String sign(String data, String privateKey) throws Exception {
        PrivateKey key = KeyFactory.getInstance("RSA")
                .generatePrivate(new PKCS8EncodedKeySpec(Base64.getDecoder().decode(privateKey)));
        Signature signature = Signature.getInstance("SHA256WithRSA");
        signature.initSign(key);
        signature.update(data.getBytes(StandardCharsets.UTF_8));
        return Base64.getEncoder().encodeToString(signature.sign());
    }


    /**
     * 使用公钥验签
     *
     * @param data      原始数据
     * @param sign      签名
     * @param publicKey Base64编码的公钥
     */

    public static boolean verify(String data, String sign, String publicKey) throws Exception {
        PublicKey key = KeyFactory.getInstance("RSA")
                .generatePublic(new X509EncodedKeySpec(Base64.getDecoder().decode(publicKey)));

        Signature signature = Signature.getInstance("SHA256WithRSA");
        signature.initVerify(key);
        signature.update(data.getBytes(StandardCharsets.UTF_8));
        return signature.verify(Base64.getDecoder().decode(sign));
    }
}
