package com.af.v4.system.common.plugins.io;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.concurrent.atomic.AtomicReference;

/**
 * IO工具类
 *
 * @author Mr.river
 */
public final class IOTools {

    private static final Logger log = LoggerFactory.getLogger(IOTools.class);

    /**
     * 相对路径默认头
     */
    private static final String DEFAULT_PATH_PREFIX = "/";

    /**
     * 获取文件输入流
     *
     * @param path     文件路径
     * @param runnable 获取stream后操作
     */
    public static void getStream(String path, StreamRunnable runnable) {
        getStream(path, runnable, notFindPath -> {
        });
    }

    /**
     * 获取文件输入流
     *
     * @param path        文件路径
     * @param runnable    获取stream后操作
     * @param notFindable stream为null后操作
     */
    public static void getStream(String path, StreamRunnable runnable, NotFindable notFindable) {
        String relativePath = formatRelativePath(path);
        try (InputStream stream = IOTools.class.getResourceAsStream(relativePath)) {
            if (stream != null) {
                runnable.run(stream);
            } else {
                notFindable.run(path);
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static String formatRelativePath(String path) {
        // 格式校验
        if (path == null) {
            throw new RuntimeException(new IllegalArgumentException("传入的参数为: " + path + ", 请检查参数格式"));
        } else {
            if (!path.startsWith(DEFAULT_PATH_PREFIX)) {
                path = DEFAULT_PATH_PREFIX + path;
            }
        }
        return path;
    }

    /**
     * 判断文件是否存在
     *
     * @param path 文件路径
     * @return 是否存在
     */
    public static boolean hasResource(String path) {
        return IOTools.class.getResource(formatRelativePath(path)) != null;
    }

    /**
     * 获取文本内容
     *
     * @param path 文件路径
     * @return 文本内容
     */
    public static String readText(String path) {
        AtomicReference<String> builder = new AtomicReference<>();
        getStream(path, stream -> {
            try (BufferedReader bufferedReader = new BufferedReader(new InputStreamReader(stream, StandardCharsets.UTF_8))) {
                String line;
                StringBuilder sb = new StringBuilder();
                while ((line = bufferedReader.readLine()) != null) {
                    sb.append(line).append("\r\n");
                }
                builder.set(sb.toString());
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }, (fileName) -> {
            throw new RuntimeException(fileName + "文件不存在");
        });
        return builder.get();
    }

    @FunctionalInterface
    public interface StreamRunnable {
        void run(InputStream stream);
    }

    @FunctionalInterface
    public interface NotFindable {
        void run(String notFindPath);
    }
}
