package com.af.v4.system.common.plugins.http;

import com.af.v4.system.common.plugins.http.config.HttpClientConfig;
import com.af.v4.system.common.plugins.http.core.BaseHttpPoolUtil;
import com.af.v4.system.common.plugins.http.core.HttpConnectionPoolUtil;
import org.apache.hc.client5.http.classic.methods.*;
import org.json.JSONObject;

import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.Iterator;


/**
 * HTTP请求工具类
 *
 * @author Mr.river
 */
public class RestTools {

    /**
     * 发送通用Http请求
     *
     * @param value              请求参数
     * @param headers            请求头
     * @param config             HttpClient配置文件
     * @param isStandardResponse 是否是标准响应（返回响应body和header）
     * @param base               请求类型
     * @return 请求结果
     */
    public static String request(String value, String headers, HttpClientConfig config, Boolean isStandardResponse, HttpUriRequestBase base) {
        JSONObject headerJson = BaseHttpPoolUtil.buildHeader(headers);
        return HttpConnectionPoolUtil.request(value, headerJson.toString(), config, isStandardResponse, base);
    }

    public static String request(String value, String headers, HttpClientConfig config, HttpUriRequestBase base) {
        return request(value, headers, config, false, base);
    }

    /**
     * 发送post请求
     *
     * @param path               请求路径
     * @param value              请求体
     * @param headers            请求头
     * @param isStandardResponse 是否是标准响应（返回响应body和header）
     * @return String结果
     */
    public static String post(String path, String value, String headers, Boolean isStandardResponse) {
        return request(value, headers, null, isStandardResponse, new HttpPost(path));
    }

    public static String post(String path, String value, String headers) {
        return post(path, value, headers, false);
    }

    public static String post(String path, JSONObject value) {
        return post(path, value == null ? null : value.toString(), null);
    }

    public static String postByFormData(String path, String value, String headers, Boolean isStandardResponse) {
        return HttpConnectionPoolUtil.requestFormUrlEncoded(value, headers, null, isStandardResponse, new HttpPost(path));
    }

    public static String postByFormData(String path, String value, String headers) {
        return postByFormData(path, value, headers, false);
    }

    public static String postByFormData(String path, JSONObject value) {
        return postByFormData(path, value == null ? null : value.toString(), null);
    }

    /**
     * 发送get请求
     *
     * @param path               请求路径
     * @param headers            请求头
     * @param isStandardResponse 是否是标准响应（返回响应body和header）
     * @return String结果
     */
    public static String get(String path, String headers, Boolean isStandardResponse) {
        return request(null, headers, null, isStandardResponse, new HttpGet(path));
    }

    public static String get(String path, String headers) {
        return get(path, headers, false);
    }

    public static String get(String path) {
        return get(path, null);
    }

    /**
     * 发送put请求
     *
     * @param path               请求路径
     * @param value              请求参数
     * @param headers            请求头
     * @param isStandardResponse 是否是标准响应（返回响应body和header）
     * @return String结果
     */
    public static String put(String path, String value, String headers, Boolean isStandardResponse) {
        return request(value, headers, null, isStandardResponse, new HttpPut(path));
    }

    public static String put(String path, String value, String headers) {
        return put(path, value, headers, false);
    }

    public static String put(String path, JSONObject value) {
        return put(path, value == null ? null : value.toString(), null);
    }

    /**
     * 发送delete请求
     *
     * @param path               请求路径
     * @param value              请求参数
     * @param headers            请求头
     * @param isStandardResponse 是否是标准响应（返回响应body和header）
     * @return String结果
     */
    public static String delete(String path, String value, String headers, Boolean isStandardResponse) {
        return request(value, headers, null, isStandardResponse, new HttpDelete(path));
    }

    public static String delete(String path, String value, String headers) {
        return delete(path, value, headers, false);
    }

    public static String delete(String path, JSONObject value) {
        return delete(path, value.toString(), null);
    }

    /**
     * REST请求路径拼接参数
     *
     * @param url    请求路径
     * @param params 请求参数
     * @return String结果
     */
    public static String formatURL(String url, JSONObject params) {
        //设置表单长度为30个字节*N个请求参数
        int capacity = params.length() * 30;
        //拼接请求路径
        StringBuilder buffer = new StringBuilder(capacity);
        buffer.append(url).append("?");
        //取出JSON里面的请求参数，添加到路径中。
        Iterator<String> keys = params.keys();
        while (keys.hasNext()) {
            String key = keys.next();
            String val = String.valueOf(params.get(key));
            if (!"".equals(val)) {
                buffer.append(key).append("=").append(val);
            } else {
                buffer.append(key);
            }
            if (keys.hasNext()) {
                buffer.append("&");
            }
        }
        return buffer.toString();
    }

    /**
     * REST请求路径拼接参数
     *
     * @param url    请求路径
     * @param params 请求参数
     * @return String结果
     */
    public static String formatAndEncodeURL(String url, JSONObject params) {
        String urlStr = formatURL(url, params);
        return urlStr.substring(0, urlStr.indexOf("?") + 1) + URLEncoder.encode(urlStr.substring(urlStr.indexOf("?") + 1), StandardCharsets.UTF_8);
    }

    /**
     * 获取标准格式的请求路径
     *
     * @param protocolType 协议类型（http,https,ftp...）
     * @param url          请求ip,域名
     * @param port         端口，默认80
     * @param paramStr     需要拼接的路径
     * @return 标准格式的请求路径
     * @deprecated Logic目前可以在字符串中使用//，因此该接口不再必要
     */
    @Deprecated(since = "1.1.8")
    public static String getStandardURL(String protocolType, String url, String port, String paramStr) {
        if (port == null) {
            port = "80";
        }
        url = protocolType + "://" + url + ":" + port;
        if (paramStr != null) {
            url += paramStr;
        }
        return url;
    }
}
