package com.af.v4.system.common.plugins.other;

import org.json.JSONArray;
import org.json.JSONObject;

/**
 * 树状数据转换工具类
 *
 * @author 张振宇
 */
public class ToTreeTools {

    /**
     * 列表转树状数据
     *
     * @param arr   数据 [{id,name,parentId},{id,name,parentId}]
     * @param id    自定义组装成功数据的id  默认：id
     * @param pid   自定义组装成功数据的父id  默认：parentid
     * @param label 自定义组装取值数据  默认：name 含义：如果配置的是 ’label‘ 则取每条数据的 中的’label‘属性作为 返回数据的label属性
     * @param value 自定义组装取值数据  默认：value 含义：如果配置的是 ’value‘ 则取每条数据的 中的’value‘属性作为 返回数据的value属性
     * @param child 自定义子数组的属性名  默认：children
     * @return 一个tree
     * @apiNote <p>每个节点的id需要唯一 如有重复可加标识符，如：</p>
     * select 'parent' + cast(id as varchar) id, f_name, null f_typeid
     * from t_iot_device_type
     * union all
     * select 'children' + cast(id as varchar) id, f_brand, 'parent' + cast(isnull(f_typeid, 0) as varchar) f_typeid
     * from t_iot_device_brand;
     */
    public static JSONArray listToTree(JSONArray arr, String id, String pid, String label, String value, String child) {
        JSONObject jsonObject = new JSONObject();
        jsonObject.put("id", id);
        jsonObject.put("parentid", pid);
        jsonObject.put("name", label);
        jsonObject.put("value", value);
        jsonObject.put("children", child);
        return toTree(arr, jsonObject);
    }

    public static JSONArray listToTree(JSONArray arr) {
        JSONObject jsonObject = new JSONObject();
        jsonObject.put("id", "id");
        jsonObject.put("parentid", "parentid");
        jsonObject.put("name", "label");
        jsonObject.put("value", "value");
        jsonObject.put("children", "children");
        return toTree(arr, jsonObject);
    }


    /**
     * 转换为树状数据
     *
     * @param data   原始数据
     * @param config 配置
     * @return 树状结构数据
     */
    public static JSONArray toTree(JSONArray data, JSONObject config) {
        JSONObject childrenListMap = new JSONObject();
        JSONObject nodeIds = new JSONObject();
        JSONArray tree = new JSONArray();
        String cid = config.optString("id");
        String cParentId = config.optString("parentid");
        String cLabel = config.optString("name");
        String cValue = config.optString("value");
        String cChildren = config.optString("children");
        for (Object jsonObject : data) {
            String parentId = ((JSONObject) jsonObject).optString(cParentId, "顶层");
            String id = ((JSONObject) jsonObject).optString(cid);
            if (!childrenListMap.has(parentId) && childrenListMap.opt(parentId) == null) {
                childrenListMap.put(parentId, new JSONArray());
            }
            nodeIds.put(id, jsonObject);
            childrenListMap.getJSONArray(parentId).put(jsonObject);
        }
        for (Object jsonObject : data) {
            String parentId = ((JSONObject) jsonObject).optString(cParentId);
            if (!nodeIds.has(parentId) && nodeIds.opt(parentId) == null) {
                tree.put(jsonObject);
            }
        }
        for (Object obj : tree) {
            adaptToChildrenList((JSONObject) obj, childrenListMap, cid, cLabel, cValue, cChildren);
        }
        return tree;
    }


    public static void adaptToChildrenList(JSONObject obj, JSONObject childrenListMap, String cid, String cLabel, String cValue, String cChildren) {
        obj.put("label", obj.optString(cLabel));
        obj.put("value", obj.optString(cValue));
        int totalCount = 0;

        if (childrenListMap.has(obj.optString(cid)) && childrenListMap.optString(obj.optString(cid)) != null) {
            JSONArray childrenArray = childrenListMap.getJSONArray(obj.optString(cid));
            obj.put(cChildren, childrenArray);

            // 递归处理子节点，并累加count值
            for (Object object : childrenArray) {
                adaptToChildrenList((JSONObject) object, childrenListMap, cid, cLabel, cValue, cChildren);
                totalCount += ((JSONObject) object).optInt("count", 0);
            }
        }

        // 如果当前节点有自己的count值，也加上
        int nodeCount = obj.optInt("count", 0);
        totalCount += nodeCount;

        // 设置父节点的count值为子节点的count总和
        obj.put("count", totalCount);
    }

}
