package com.af.v4.system.common.plugins.http.core;

import com.af.v4.system.common.plugins.http.config.CloseHttpLogHandler;
import com.af.v4.system.common.plugins.http.config.HttpClientConfig;
import com.af.v4.system.common.plugins.http.core.response.DefaultAsyncResponse;
import org.apache.hc.client5.http.async.methods.SimpleHttpRequest;
import org.apache.hc.client5.http.async.methods.SimpleHttpResponse;
import org.apache.hc.client5.http.config.RequestConfig;
import org.apache.hc.client5.http.impl.async.CloseableHttpAsyncClient;
import org.apache.hc.client5.http.impl.async.HttpAsyncClients;
import org.apache.hc.client5.http.impl.nio.PoolingAsyncClientConnectionManager;
import org.apache.hc.client5.http.impl.nio.PoolingAsyncClientConnectionManagerBuilder;
import org.apache.hc.client5.http.ssl.DefaultClientTlsStrategy;
import org.apache.hc.client5.http.ssl.NoopHostnameVerifier;
import org.apache.hc.client5.http.ssl.TrustSelfSignedStrategy;
import org.apache.hc.core5.concurrent.FutureCallback;
import org.apache.hc.core5.http.config.Registry;
import org.apache.hc.core5.http.config.RegistryBuilder;
import org.apache.hc.core5.http.nio.ssl.TlsStrategy;
import org.apache.hc.core5.reactor.IOReactorConfig;
import org.apache.hc.core5.ssl.SSLContexts;
import org.apache.hc.core5.util.Timeout;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.security.KeyManagementException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;

/**
 * HttpAsyncClient连接池实现
 *
 * @author Mr.river
 * @see com.af.v4.system.common.plugins.http.RestAsyncTools
 */
public class HttpAsyncConnectionPoolUtil extends BaseHttpPoolUtil {

    private static final Logger LOGGER = LoggerFactory.getLogger(HttpAsyncConnectionPoolUtil.class);
    /**
     * 相当于线程锁,用于线程安全
     */
    private final static Object SYNC_LOCK = new Object();
    /**
     * 发送HTTP异步请求的客户端单例
     */
    private static volatile CloseableHttpAsyncClient httpClient;

    /**
     * 获取httpclient的异步https实例（需要SSL证书）
     *
     * @param sslSocketFactory SSL构建器
     * @return httpclient实例
     */
    public static CloseableHttpAsyncClient getHttpClient(DefaultClientTlsStrategy sslSocketFactory) {
        return createHttpClient(sslSocketFactory);
    }

    /**
     * 构建httpclient实例
     *
     * @return httpclient实例
     */
    private static CloseableHttpAsyncClient createHttpClient() {
        return createHttpClient(null);
    }

    /**
     * 构建httpclient实例
     *
     * @param sslSocketFactory SSL构建器
     * @return httpclient实例
     */
    private static CloseableHttpAsyncClient createHttpClient(DefaultClientTlsStrategy sslSocketFactory) {
        //设置https相关信息
        if (sslSocketFactory == null) {
            LOGGER.info("创建HTTP异步客户端会话");
            try {
                sslSocketFactory = new DefaultClientTlsStrategy(
                        SSLContexts.custom().loadTrustMaterial(null, new TrustSelfSignedStrategy()).build(),
                        NoopHostnameVerifier.INSTANCE
                );
            } catch (NoSuchAlgorithmException | KeyManagementException | KeyStoreException e) {
                throw new RuntimeException(e);
            }
        } else {
            LOGGER.info("创建https异步客户端会话");
        }
        Registry<TlsStrategy> registry = RegistryBuilder.<TlsStrategy>create()
                .register("https", sslSocketFactory)
                .build();
        // 配置io线程
        IOReactorConfig ioReactorConfig = IOReactorConfig.custom()
                .setIoThreadCount(Runtime.getRuntime().availableProcessors() * 2 + 1)
                .setSoKeepAlive(true)
                .build();
        PoolingAsyncClientConnectionManager manager =
                PoolingAsyncClientConnectionManagerBuilder.create()
                        .setTlsStrategy(registry.lookup("https"))
                        .build();

        /* 设置连接参数 **/
        // 最大连接数
        manager.setMaxTotal(HttpClientConfig.HTTP_MAX_POOL_SIZE);
        // 路由最大连接数
        manager.setDefaultMaxPerRoute(HttpClientConfig.HTTP_MAX_POOL_SIZE);

        HttpClientConfig config = new HttpClientConfig.Builder()
                .connectTimeout(20000)
                .socketTimeout(60 * 5 * 1000).build();
        RequestConfig requestConfig = RequestConfig.custom()
                .setConnectionRequestTimeout(Timeout.ofSeconds(config.getConnectTimeout()))
                .setConnectTimeout(Timeout.ofSeconds(config.getConnectTimeout()))
                .setResponseTimeout(Timeout.ofSeconds(config.getSocketTimeout())).build();

        return HttpAsyncClients.custom()
                .setDefaultRequestConfig(requestConfig)
                .setIOReactorConfig(ioReactorConfig)
                .setConnectionManager(manager)
                .build();
    }

    /**
     * 获取HTTP异步请求 httpclient实例
     *
     * @return httpclient实例
     */
    public static CloseableHttpAsyncClient getHttpClient() {
        if (httpClient == null) {
            //多线程下同时调用getHttpClient容易导致重复创建httpClient对象的问题,所以加上了同步锁
            synchronized (SYNC_LOCK) {
                if (httpClient == null) {
                    httpClient = createHttpClient();
                }
            }
        }
        httpClient.start();
        return httpClient;
    }

    /**
     * 发送通用HTTP异步请求
     *
     * @param value          请求参数
     * @param headersStr     请求头
     * @param config         HttpClient配置文件
     * @param base           请求类型
     * @param httpClient     client对象
     * @param futureCallback 异步回调处理函数
     */
    public static void request(String value,
                               String headersStr,
                               HttpClientConfig config,
                               SimpleHttpRequest base,
                               CloseableHttpAsyncClient httpClient,
                               FutureCallback<SimpleHttpResponse> futureCallback) {
        if (httpClient == null) {
            httpClient = getHttpClient();
        }
        if (futureCallback == null) {
            futureCallback = new DefaultAsyncResponse();
        }
        //配置请求参数
//        setRequestConfig(config, base);
        //设置请求头
        setHeaders(base, headersStr);
        //设置请求体
        if (base.getBody() == null) {
            setBody(base, value);
        }
        httpClient.execute(base, futureCallback);
        String url = base.getScheme() + "://" + base.getAuthority() + base.getPath();
        boolean disableLogPrint = CloseHttpLogHandler.getDisableLogPrintValue();
        if (!disableLogPrint) {
            LOGGER.info("api: {}，header: {}，body: {}，type: async", url, headersStr, value);
        }
    }
}
