package com.af.v4.system.common.plugins.http.core;

import com.af.v4.system.common.plugins.http.config.HttpClientConfig;
import org.apache.hc.client5.http.classic.methods.HttpUriRequestBase;
import org.apache.hc.client5.http.config.ConnectionConfig;
import org.apache.hc.client5.http.config.RequestConfig;
import org.apache.hc.client5.http.impl.DefaultHttpRequestRetryStrategy;
import org.apache.hc.client5.http.impl.classic.CloseableHttpClient;
import org.apache.hc.client5.http.impl.classic.HttpClients;
import org.apache.hc.client5.http.impl.io.PoolingHttpClientConnectionManager;
import org.apache.hc.client5.http.impl.io.PoolingHttpClientConnectionManagerBuilder;
import org.apache.hc.client5.http.protocol.HttpClientContext;
import org.apache.hc.client5.http.ssl.*;
import org.apache.hc.core5.http.HeaderElement;
import org.apache.hc.core5.http.HttpHeaders;
import org.apache.hc.core5.http.message.BasicHeaderElementIterator;
import org.apache.hc.core5.ssl.SSLContexts;
import org.apache.hc.core5.util.TimeValue;
import org.apache.hc.core5.util.Timeout;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StopWatch;

import java.io.IOException;
import java.security.KeyManagementException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicReference;

/**
 * HttpClient连接池实现
 *
 * @author Mr.river
 * @see com.af.v4.system.common.plugins.http.RestTools
 */
public class HttpConnectionPoolUtil extends BaseHttpPoolUtil {

    private static final Logger LOGGER = LoggerFactory.getLogger(HttpConnectionPoolUtil.class);
    /**
     * httpClient默认key名
     */
    private final static String HTTP_CLIENT_KEY = "http";
    /**
     * 客户端单例，其中http请求固定key名为http，https根据自行传入的key名获取client
     */
    private static final Map<String, CloseableHttpClient> CLIENT_MAP = new ConcurrentHashMap<>(3);

    /**
     * 构建httpclient实例
     *
     * @param clientKey        httpclient key
     * @param sslSocketFactory SSL构建器
     * @return httpclient实例
     */
    private static CloseableHttpClient createHttpClient(String clientKey, TlsSocketStrategy sslSocketFactory) {
        return CLIENT_MAP.computeIfAbsent(clientKey, (key) -> {
            HttpClientConfig config = new HttpClientConfig.Builder().build();
            TlsSocketStrategy realTlsSocketStrategy;
            //设置https相关信息
            if (sslSocketFactory == null) {
                try {
                    realTlsSocketStrategy = new DefaultClientTlsStrategy(
                            SSLContexts.custom().loadTrustMaterial(TrustAllStrategy.INSTANCE).build(),
                            HostnameVerificationPolicy.CLIENT,
                            NoopHostnameVerifier.INSTANCE
                    );

                } catch (NoSuchAlgorithmException | KeyManagementException | KeyStoreException e) {
                    throw new RuntimeException(e);
                }
            } else {
                realTlsSocketStrategy = sslSocketFactory;
            }
            PoolingHttpClientConnectionManager manager = PoolingHttpClientConnectionManagerBuilder.create()
                    .setTlsSocketStrategy(realTlsSocketStrategy)
                    .setMaxConnTotal(HttpClientConfig.HTTP_MAX_POOL_SIZE)
                    .setMaxConnPerRoute(HttpClientConfig.HTTP_MAX_ROUTE_SIZE)
                    .setDefaultConnectionConfig(
                            ConnectionConfig.custom()
                                    .setConnectTimeout(Timeout.ofMilliseconds(config.getConnectTimeout()))
                                    .build()
                    ).build();
            return HttpClients.custom()
                    .setKeepAliveStrategy((response, context) -> {
                        // Honor 'keep-alive' header
                        BasicHeaderElementIterator it = new BasicHeaderElementIterator(
                                response.headerIterator(HttpHeaders.KEEP_ALIVE));
                        while (it.hasNext()) {
                            HeaderElement he = it.next();
                            String param = he.getName();
                            String value = he.getValue();
                            if (value != null && "timeout".equalsIgnoreCase(param)) {
                                try {
                                    return TimeValue.ofSeconds(Long.parseLong(value));
                                } catch (NumberFormatException ignore) {
                                }
                            }
                        }
                        // Keep alive for 5 seconds only
                        return TimeValue.ofSeconds(5);
                    })
                    .setConnectionManager(manager)
                    .setRetryStrategy(new DefaultHttpRequestRetryStrategy(0, TimeValue.ZERO_MILLISECONDS))
                    .evictExpiredConnections()
                    .evictIdleConnections(TimeValue.ofMilliseconds(HttpClientConfig.HTTP_IDEL_TIME_OUT))
                    .setDefaultRequestConfig(
                            RequestConfig.custom()
                                    .setResponseTimeout(Timeout.ofMilliseconds(config.getSocketTimeout()))
                                    .build()
                    )
                    .build();
        });
    }

    /**
     * 获取httpclient的http实例
     *
     * @param sslSocketFactory SSL构建器
     * @return httpclient实例
     */
    public static CloseableHttpClient getHttpClient(String tlsName, TlsSocketStrategy sslSocketFactory) {
        String clientKeyName;
        if (sslSocketFactory == null) {
            clientKeyName = HTTP_CLIENT_KEY;
        } else {
            clientKeyName = tlsName;
        }
        return createHttpClient(clientKeyName, sslSocketFactory);
    }

    public static CloseableHttpClient getHttpClient() {
        return getHttpClient(null, null);
    }

    /**
     * 发送通用HTTP请求
     *
     * @param value              请求参数
     * @param headersStr         请求头
     * @param config             请求配置
     * @param isStandardResponse 是否是标准响应（返回响应body和header）
     * @param base               请求类型
     * @return 请求结果
     */
    public static String request(String value, String headersStr, HttpClientConfig config, Boolean isStandardResponse, HttpUriRequestBase base) {
        return request(value, headersStr, config, base, isStandardResponse, null);
    }

    public static String request(String value, String headersStr, HttpClientConfig config, HttpUriRequestBase base) {
        return request(value, headersStr, config, false, base);
    }

    public static String requestFormUrlEncoded(String value, String headersStr, HttpClientConfig config, Boolean isStandardResponse, HttpUriRequestBase base) {
        return requestFormUrlEncoded(value, headersStr, config, base, isStandardResponse, null);
    }

    public static String requestFormUrlEncoded(String value, String headersStr, HttpClientConfig config, HttpUriRequestBase base) {
        return requestFormUrlEncoded(value, headersStr, config, false, base);
    }

    /**
     * 发送通用HTTP请求
     *
     * @param value              请求参数
     * @param headersStr         请求头
     * @param base               请求类型
     * @param httpClient         client对象
     * @param isStandardResponse 是否是标准响应（返回响应body和header）
     * @return 请求结果
     */
    public static String request(String value,
                                 String headersStr,
                                 HttpClientConfig config,
                                 HttpUriRequestBase base,
                                 Boolean isStandardResponse,
                                 CloseableHttpClient httpClient) {
        String url = base.getScheme() + "://" + base.getAuthority() + base.getPath();
        if (httpClient == null) {
            httpClient = getHttpClient();
        }
        //配置请求参数
        setRequestConfig(config, base);
        //设置请求头
        setHeaders(base, headersStr);
        //设置请求体
        if (base.getEntity() == null) {
            setBody(base, value);
        }
        StopWatch stopWatch = new StopWatch();
        stopWatch.start();
        final AtomicReference<String> result = new AtomicReference<>();
        try {
            httpClient.execute(base, HttpClientContext.create(), response -> {
                String responseContext = getResponseData(response, isStandardResponse);
                result.set(responseContext);
                return response;
            });
        } catch (IOException e) {
            result.set(BaseHttpPoolUtil.buildErrorResult(e));
        } finally {
            BaseHttpPoolUtil.printLog(LOGGER, "sync", stopWatch, url, headersStr, value, result.get());
        }
        return result.get();
    }

    public static String requestFormUrlEncoded(String value,
                                               String headersStr,
                                               HttpClientConfig config,
                                               HttpUriRequestBase base,
                                               Boolean isStandardResponse,
                                               CloseableHttpClient httpClient) {
        setUrlEncodedBody(base, value);
        return request(value, headersStr, config, base, isStandardResponse, httpClient);
    }

    public static String requestFormUrlEncoded(String value,
                                               String headersStr,
                                               HttpClientConfig config,
                                               HttpUriRequestBase base,
                                               CloseableHttpClient httpClient) {
        return requestFormUrlEncoded(value, headersStr, config, base, false, httpClient);
    }
}
