package com.af.v4.system.common.plugins.http.core;

import com.af.v4.system.common.plugins.http.config.HttpClientConfig;
import com.af.v4.system.common.plugins.http.core.response.AsyncResponse;
import com.af.v4.system.common.plugins.http.core.response.DefaultAsyncResponse;
import org.apache.hc.client5.http.async.methods.SimpleHttpRequest;
import org.apache.hc.client5.http.config.ConnectionConfig;
import org.apache.hc.client5.http.config.RequestConfig;
import org.apache.hc.client5.http.impl.DefaultHttpRequestRetryStrategy;
import org.apache.hc.client5.http.impl.async.CloseableHttpAsyncClient;
import org.apache.hc.client5.http.impl.async.HttpAsyncClients;
import org.apache.hc.client5.http.impl.nio.PoolingAsyncClientConnectionManager;
import org.apache.hc.client5.http.impl.nio.PoolingAsyncClientConnectionManagerBuilder;
import org.apache.hc.client5.http.ssl.DefaultClientTlsStrategy;
import org.apache.hc.client5.http.ssl.HostnameVerificationPolicy;
import org.apache.hc.client5.http.ssl.NoopHostnameVerifier;
import org.apache.hc.client5.http.ssl.TrustAllStrategy;
import org.apache.hc.core5.http.HeaderElement;
import org.apache.hc.core5.http.HttpHeaders;
import org.apache.hc.core5.http.message.BasicHeaderElementIterator;
import org.apache.hc.core5.http.nio.ssl.TlsStrategy;
import org.apache.hc.core5.reactor.IOReactorConfig;
import org.apache.hc.core5.ssl.SSLContexts;
import org.apache.hc.core5.util.TimeValue;
import org.apache.hc.core5.util.Timeout;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.security.KeyManagementException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * HttpAsyncClient连接池实现
 *
 * @author Mr.river
 * @see com.af.v4.system.common.plugins.http.RestAsyncTools
 */
public class HttpAsyncConnectionPoolUtil extends BaseHttpPoolUtil {

    private static final Logger LOGGER = LoggerFactory.getLogger(HttpAsyncConnectionPoolUtil.class);
    /**
     * httpClient默认key名
     */
    private final static String HTTP_CLIENT_KEY = "http";
    /**
     * 客户端单例，其中http请求固定key名为http，https根据自行传入的key名获取client
     */
    private static final Map<String, CloseableHttpAsyncClient> CLIENT_MAP = new ConcurrentHashMap<>(3);

    /**
     * 构建httpclient实例
     *
     * @param clientKey httpclient key
     * @param strategy  SSL构建器
     * @return httpclient实例
     */
    private static CloseableHttpAsyncClient createHttpClient(String clientKey, TlsStrategy strategy) {
        return CLIENT_MAP.computeIfAbsent(clientKey, (key) -> {
            HttpClientConfig config = new HttpClientConfig.Builder().build();
            TlsStrategy realTlsStrategy;
            //设置https相关信息
            if (strategy == null) {
                try {
                    realTlsStrategy = new DefaultClientTlsStrategy(
                            SSLContexts.custom().loadTrustMaterial(TrustAllStrategy.INSTANCE).build(),
                            HostnameVerificationPolicy.CLIENT,
                            NoopHostnameVerifier.INSTANCE
                    );

                } catch (NoSuchAlgorithmException | KeyManagementException | KeyStoreException e) {
                    throw new RuntimeException(e);
                }
            } else {
                realTlsStrategy = strategy;
            }
            PoolingAsyncClientConnectionManager manager = PoolingAsyncClientConnectionManagerBuilder.create()
                    .setTlsStrategy(realTlsStrategy)
                    .setMaxConnTotal(HttpClientConfig.HTTP_MAX_POOL_SIZE)
                    .setMaxConnPerRoute(HttpClientConfig.HTTP_MAX_ROUTE_SIZE)
                    .setDefaultConnectionConfig(
                            ConnectionConfig.custom()
                                    .setConnectTimeout(Timeout.ofMilliseconds(config.getConnectTimeout()))
                                    .build()
                    ).build();
            return HttpAsyncClients.custom()
                    .setKeepAliveStrategy((response, context) -> {
                        // Honor 'keep-alive' header
                        BasicHeaderElementIterator it = new BasicHeaderElementIterator(
                                response.headerIterator(HttpHeaders.KEEP_ALIVE));
                        while (it.hasNext()) {
                            HeaderElement he = it.next();
                            String param = he.getName();
                            String value = he.getValue();
                            if (value != null && "timeout".equalsIgnoreCase(param)) {
                                try {
                                    return TimeValue.ofSeconds(Long.parseLong(value));
                                } catch (NumberFormatException ignore) {
                                }
                            }
                        }
                        // Keep alive for 5 seconds only
                        return TimeValue.ofSeconds(5);
                    })
                    .setConnectionManager(manager)
                    .setIOReactorConfig(IOReactorConfig.custom()
                            .setIoThreadCount(Math.max(2, Runtime.getRuntime().availableProcessors()))
                            .setSoKeepAlive(true)
                            .build())
                    .setRetryStrategy(new DefaultHttpRequestRetryStrategy(0, TimeValue.ZERO_MILLISECONDS))
                    .evictExpiredConnections()
                    .evictIdleConnections(TimeValue.ofMilliseconds(HttpClientConfig.HTTP_IDEL_TIME_OUT))
                    .setDefaultRequestConfig(
                            RequestConfig.custom()
                                    .setResponseTimeout(Timeout.ofMilliseconds(config.getSocketTimeout()))
                                    .build()
                    )
                    .build();
        });
    }

    /**
     * 获取异步httpclient的http实例
     *
     * @param strategy SSL构建器
     * @return httpclient实例
     */
    public static CloseableHttpAsyncClient getHttpClient(String tlsName, TlsStrategy strategy) {
        String clientKeyName;
        if (strategy == null) {
            clientKeyName = HTTP_CLIENT_KEY;
        } else {
            clientKeyName = tlsName;
        }
        CloseableHttpAsyncClient client = createHttpClient(clientKeyName, strategy);
        client.start();
        return client;
    }

    public static CloseableHttpAsyncClient getHttpClient() {
        return getHttpClient(null, null);
    }

    /**
     * 发送通用HTTP异步请求
     *
     * @param value          请求参数
     * @param headersStr     请求头
     * @param config         HttpClient配置文件
     * @param base           请求类型
     * @param httpClient     client对象
     * @param futureCallback 异步回调处理函数
     */
    public static void request(String value,
                               String headersStr,
                               HttpClientConfig config,
                               SimpleHttpRequest base,
                               Boolean isStandardResponse,
                               CloseableHttpAsyncClient httpClient,
                               AsyncResponse futureCallback) {
        if (httpClient == null) {
            httpClient = getHttpClient();
        }
        if (futureCallback == null) {
            futureCallback = new DefaultAsyncResponse();
        }
        //配置请求参数
        setRequestConfig(config, base);
        //设置请求头
        setHeaders(base, headersStr);
        //设置请求体
        if (base.getBody() == null) {
            setBody(base, value);
        }
        String url = base.getScheme() + "://" + base.getAuthority() + base.getPath();
        futureCallback.setContext(url, headersStr, value, isStandardResponse);
        httpClient.execute(base, futureCallback);
    }
}
