package com.af.v4.system.common.plugins.core;

import cn.hutool.core.date.DateField;
import cn.hutool.core.date.DateUtil;
import org.json.JSONArray;

import java.text.SimpleDateFormat;
import java.util.*;

/**
 * SQL 工具类
 * 提供动态 SQL 生成功能，支持分表 Union 查询
 *
 * <p>占位符语法: [ROW] (大写，方括号包裹)</p>
 * <p>设计用于在 Logic 引擎中使用，避免与 Logic 的 {} 变量语法冲突</p>
 *
 * @author Mr.cursor
 */
public class SqlTools {

    private static final String PLACEHOLDER = "[ROW]";

    // 日期格式常量
    private static final String FORMAT_YEAR = "yyyy";
    private static final String FORMAT_MONTH = "yyyyMM";
    private static final String FORMAT_DAY = "yyyyMMdd";

    /**
     * 校验数组参数
     *
     * @param arr 数组参数（JSONArray 或 List）
     * @throws IllegalArgumentException 如果参数为空
     */
    private static void validateArrayParam(Object arr) {
        if (arr == null) {
            throw new IllegalArgumentException("参数 arr 不能为空");
        }

        if (arr instanceof JSONArray && ((JSONArray) arr).isEmpty()) {
            throw new IllegalArgumentException("参数 arr 不能为空");
        }

        if (arr instanceof List && ((List<?>) arr).isEmpty()) {
            throw new IllegalArgumentException("参数 arr 不能为空");
        }
    }

    /**
     * 校验 SQL 参数
     *
     * @param sql SQL 模板字符串
     * @throws IllegalArgumentException 如果参数为空或空白
     */
    private static void validateSqlParam(String sql) {
        if (sql == null || sql.trim().isEmpty()) {
            throw new IllegalArgumentException("参数 sql 不能为空");
        }
    }

    /**
     * 校验日期格式参数
     *
     * @param format 日期格式字符串
     * @throws IllegalArgumentException 如果格式不支持
     */
    private static void validateDateFormat(String format) {
        if (format == null || (!format.equals(FORMAT_YEAR)
                && !format.equals(FORMAT_MONTH)
                && !format.equals(FORMAT_DAY))) {
            throw new IllegalArgumentException(
                    "不支持的日期格式: " + format + ", 仅支持 "
                            + FORMAT_YEAR + "/" + FORMAT_MONTH + "/" + FORMAT_DAY
            );
        }
    }

    /**
     * 根据数组生成 UNION ALL 查询语句
     *
     * <p>使用场景：需要对多个分表执行相同的查询并合并结果</p>
     *
     * <p>使用示例：</p>
     * <pre>
     * JSONArray arr = new JSONArray("[1,2,3]");
     * String sql = "select id, name from t_userinfo_[ROW] where status = '1'";
     * String result = SqlTools.renderUnionSql(arr, sql);
     * // 生成:
     * // select * from (
     * // select id, name from t_userinfo_1 where status = '1'
     * // union all
     * // select id, name from t_userinfo_2 where status = '1'
     * // union all
     * // select id, name from t_userinfo_3 where status = '1'
     * // )
     * </pre>
     *
     * <p>在 Logic 中使用：</p>
     * <pre>
     * result = sqlTools.renderUnionSql([1,2,3],
     *   "select * from t_user_[ROW] where status = '{data.status}'"
     * )
     * // Logic 会先替换 {data.status}，然后 [ROW] 传递给此函数处理
     * </pre>
     *
     * @param arr 数组，每个元素将替换 SQL 模板中的 [ROW] 占位符
     * @param sql SQL 模板，如果包含 [ROW] 占位符则进行替换，否则返回原始 SQL
     * @return 生成的 UNION ALL 查询语句，外层包裹 select * from (...)；如果没有占位符则返回原始 SQL
     * @throws IllegalArgumentException 如果参数为空
     */
    public static String renderUnionSql(JSONArray arr, String sql) {
        return renderUnionSql(arr, sql, true);
    }

    /**
     * 根据数组生成 UNION ALL 查询语句，支持控制是否包装外层
     *
     * @param arr       数组，每个元素将替换 SQL 模板中的 [ROW] 占位符
     * @param sql       SQL 模板，如果包含 [ROW] 占位符则进行替换，否则返回原始 SQL
     * @param wrapOuter 是否包装外层 select * from (...)，true-包装（默认），false-不包装
     * @return 生成的 UNION ALL 查询语句；如果没有占位符则返回原始 SQL
     * @throws IllegalArgumentException 如果参数为空
     */
    public static String renderUnionSql(JSONArray arr, String sql, Boolean wrapOuter) {
        return renderUnionSqlInternal(arr, sql, wrapOuter);
    }

    /**
     * 核心 SQL 生成逻辑（私有方法）
     */
    private static String renderUnionSqlInternal(JSONArray arr, String sql, Boolean wrapOuter) {
        // 参数校验
        validateArrayParam(arr);
        validateSqlParam(sql);

        // 如果没有占位符，直接返回原始 SQL
        if (!sql.contains(PLACEHOLDER)) {
            return sql;
        }

        // 使用 StringBuilder 高效拼接 SQL
        StringBuilder result = new StringBuilder();

        if (wrapOuter) {
            result.append("select * from (\n");
        }

        // 遍历数组，生成每个子查询
        for (int i = 0; i < arr.length(); i++) {
            Object value = arr.get(i);

            // 替换占位符生成子查询
            String subQuery = sql.replace(PLACEHOLDER, String.valueOf(value));
            result.append(subQuery);

            // 添加 union all（最后一个不需要）
            if (i < arr.length() - 1) {
                result.append("\nunion all\n");
            }
        }

        if (wrapOuter) {
            result.append("\n) t");  // 添加别名 t，解决 SQL Server 兼容性问题
        }

        return result.toString();
    }

    /**
     * 根据数组生成 UNION ALL 查询语句（List 重载版本）
     *
     * <p>此方法是 {@link #renderUnionSql(JSONArray, String)} 的重载版本，支持 List 参数</p>
     *
     * @param arr List 数组，每个元素将替换 SQL 模板中的 [ROW] 占位符
     * @param sql SQL 模板，如果包含 [ROW] 占位符则进行替换，否则返回原始 SQL
     * @return 生成的 UNION ALL 查询语句，外层包裹 select * from (...)；如果没有占位符则返回原始 SQL
     * @throws IllegalArgumentException 如果参数为空
     */
    public static String renderUnionSql(List<?> arr, String sql) {
        return renderUnionSql(arr, sql, true);
    }

    /**
     * 根据数组生成 UNION ALL 查询语句（List 重载版本，支持控制包装）
     *
     * @param arr       List 数组，每个元素将替换 SQL 模板中的 [ROW] 占位符
     * @param sql       SQL 模板，如果包含 [ROW] 占位符则进行替换，否则返回原始 SQL
     * @param wrapOuter 是否包装外层 select * from (...)，true-包装（默认），false-不包装
     * @return 生成的 UNION ALL 查询语句；如果没有占位符则返回原始 SQL
     * @throws IllegalArgumentException 如果参数为空
     */
    public static String renderUnionSql(List<?> arr, String sql, Boolean wrapOuter) {
        // 参数校验
        validateArrayParam(arr);

        // 将 List 转换为 JSONArray
        JSONArray jsonArray = new JSONArray(arr);

        // 调用核心方法
        return renderUnionSqlInternal(jsonArray, sql, wrapOuter);
    }

    /**
     * 根据日期范围生成 UNION ALL 查询语句
     *
     * <p>使用场景：需要对按日期分表的多个表执行查询并合并结果</p>
     *
     * <p>支持的日期格式：</p>
     * <ul>
     *   <li>yyyy - 按年分表（如 2023, 2024）</li>
     *   <li>yyyyMM - 按月分表（如 202301, 202302）</li>
     *   <li>yyyyMMdd - 按日分表（如 20230101, 20230102）</li>
     * </ul>
     *
     * <p>使用示例：</p>
     * <pre>
     * JSONArray arr = new JSONArray("['2023-01-01', '2023-03-31']");
     * String sql = "select * from t_order_[ROW] where user_id = '1001'";
     * String format = "yyyyMM";
     * String result = SqlTools.renderUnionSqlForDate(arr, sql, format);
     * // 生成 202301, 202302, 202303 三个月的 UNION ALL 查询
     * </pre>
     *
     * <p>在 Logic 中使用：</p>
     * <pre>
     * result = sqlTools.renderUnionSqlForDate(
     *   ["2023-01-01", "2023-12-31"],
     *   "select * from t_order_[ROW] where user_id = '{data.userId}'",
     *   "yyyyMM"
     * )
     * </pre>
     *
     * @param arr    日期字符串数组，支持格式 yyyy-MM-dd 或 yyyy-MM-dd HH:mm:ss
     * @param sql    SQL 模板，如果包含 [ROW] 占位符则进行替换，否则返回原始 SQL
     * @param format 日期格式化规则，仅支持 yyyy / yyyyMM / yyyyMMdd
     * @return 生成的 UNION ALL 查询语句，外层包裹 select * from (...)；如果没有占位符则返回原始 SQL
     * @throws IllegalArgumentException 如果参数非法、日期格式错误或 format 不支持
     */
    public static String renderUnionSqlForDate(JSONArray arr, String sql, String format) {
        return renderUnionSqlForDate(arr, sql, format, true);
    }

    /**
     * 根据日期范围生成 UNION ALL 查询语句，支持控制是否包装外层
     *
     * @param arr       日期字符串数组，支持格式 yyyy-MM-dd 或 yyyy-MM-dd HH:mm:ss
     * @param sql       SQL 模板，如果包含 [ROW] 占位符则进行替换，否则返回原始 SQL
     * @param format    日期格式化规则，仅支持 yyyy / yyyyMM / yyyyMMdd
     * @param wrapOuter 是否包装外层 select * from (...)，true-包装（默认），false-不包装
     * @return 生成的 UNION ALL 查询语句；如果没有占位符则返回原始 SQL
     * @throws IllegalArgumentException 如果参数非法、日期格式错误或 format 不支持
     */
    public static String renderUnionSqlForDate(JSONArray arr, String sql, String format, Boolean wrapOuter) {
        return renderUnionSqlForDateInternal(arr, sql, format, null, null, wrapOuter);
    }

    /**
     * 根据日期范围生成 UNION ALL 查询语句，支持日期范围限制
     *
     * @param arr     日期字符串数组，支持格式 yyyy-MM-dd 或 yyyy-MM-dd HH:mm:ss；如果为空则使用 maxDate
     * @param sql     SQL 模板，如果包含 [ROW] 占位符则进行替换，否则返回原始 SQL
     * @param format  日期格式化规则，仅支持 yyyy / yyyyMM / yyyyMMdd
     * @param minDate 最小日期限制（格式 yyyy-MM-dd），null 表示不限制
     * @param maxDate 最大日期限制（格式 yyyy-MM-dd），null 时使用当前日期
     * @return 生成的 UNION ALL 查询语句，外层包裹 select * from (...)；如果没有占位符则返回原始 SQL
     * @throws IllegalArgumentException 如果参数非法、日期格式错误或 format 不支持
     */
    public static String renderUnionSqlForDate(JSONArray arr, String sql, String format, String minDate, String maxDate) {
        return renderUnionSqlForDateInternal(arr, sql, format, minDate, maxDate, true);
    }

    /**
     * 根据日期范围生成 UNION ALL 查询语句，支持日期范围限制和控制包装
     *
     * @param arr       日期字符串数组，支持格式 yyyy-MM-dd 或 yyyy-MM-dd HH:mm:ss；如果为空则使用 maxDate
     * @param sql       SQL 模板，如果包含 [ROW] 占位符则进行替换，否则返回原始 SQL
     * @param format    日期格式化规则，仅支持 yyyy / yyyyMM / yyyyMMdd
     * @param minDate   最小日期限制（格式 yyyy-MM-dd），null 表示不限制
     * @param maxDate   最大日期限制（格式 yyyy-MM-dd），null 时使用当前日期
     * @param wrapOuter 是否包装外层 select * from (...)，true-包装（默认），false-不包装
     * @return 生成的 UNION ALL 查询语句；如果没有占位符则返回原始 SQL
     * @throws IllegalArgumentException 如果参数非法、日期格式错误或 format 不支持
     */
    public static String renderUnionSqlForDate(JSONArray arr, String sql, String format, String minDate, String maxDate, Boolean wrapOuter) {
        return renderUnionSqlForDateInternal(arr, sql, format, minDate, maxDate, wrapOuter);
    }

    /**
     * 核心日期 SQL 生成逻辑（私有方法）
     */
    private static String renderUnionSqlForDateInternal(JSONArray arr, String sql, String format, String minDate, String maxDate, Boolean wrapOuter) {
        // 第一步：参数校验和预处理
        validateSqlParam(sql);

        // 如果没有占位符，直接返回原始 SQL
        if (!sql.contains(PLACEHOLDER)) {
            return sql;
        }

        validateDateFormat(format);

        // 处理 maxDate 默认值（使用当前日期）
        if (maxDate == null || maxDate.trim().isEmpty()) {
            maxDate = DateUtil.format(new Date(), "yyyy-MM-dd");
        }

        // 处理 arr 为空的情况（使用 maxDate 创建单元素数组）
        if (arr == null || arr.isEmpty()) {
            arr = new JSONArray().put(maxDate);
        }

        // 第二步：日期解析与范围生成
        List<Date> dates = new ArrayList<>();

        // 解析所有日期
        for (int i = 0; i < arr.length(); i++) {
            String dateStr = arr.getString(i);
            try {
                Date date = DateUtil.parse(dateStr);
                dates.add(date);
            } catch (Exception e) {
                throw new IllegalArgumentException("日期格式错误: " + dateStr + ", 支持格式 yyyy-MM-dd 或 yyyy-MM-dd HH:mm:ss", e);
            }
        }

        // 找出 arr 中的最小和最大日期
        Date arrMinDate = Collections.min(dates);
        Date arrMaxDate = Collections.max(dates);

        // 第三步：日期范围裁剪
        Date actualMinDate = arrMinDate;
        Date actualMaxDate = arrMaxDate;
        Date minBoundary = null;
        Date maxBoundary = null;

        // 解析最小日期边界
        if (minDate != null && !minDate.trim().isEmpty()) {
            try {
                minBoundary = DateUtil.parse(minDate);
            } catch (Exception e) {
                throw new IllegalArgumentException("最小日期格式错误: " + minDate + ", 支持格式 yyyy-MM-dd", e);
            }
        }

        // 解析最大日期边界
        if (maxDate != null && !maxDate.trim().isEmpty()) {
            try {
                maxBoundary = DateUtil.parse(maxDate);
            } catch (Exception e) {
                throw new IllegalArgumentException("最大日期格式错误: " + maxDate + ", 支持格式 yyyy-MM-dd", e);
            }
        }

        // 应用最小日期限制
        if (minBoundary != null && arrMinDate.before(minBoundary)) {
            actualMinDate = minBoundary;
        }

        // 应用最大日期限制
        if (maxBoundary != null && arrMaxDate.after(maxBoundary)) {
            actualMaxDate = maxBoundary;
        }

        // 特殊情况处理：如果裁剪后 actualMinDate > actualMaxDate
        // 说明 arr 的所有日期都超出了边界范围，需要判断是哪种情况
        if (actualMinDate.after(actualMaxDate)) {
            // 如果 arr 的最大日期都小于最小边界，使用最小边界
            if (minBoundary != null && arrMaxDate.before(minBoundary)) {
                actualMinDate = minBoundary;
                actualMaxDate = minBoundary;
            }
            // 否则 arr 的最小日期都大于最大边界，使用最大边界
            else if (maxBoundary != null && arrMinDate.after(maxBoundary)) {
                actualMinDate = maxBoundary;
                actualMaxDate = maxBoundary;
            }
        }

        // 根据 format 生成日期序列
        LinkedHashSet<String> dateSequence = generateDateSequence(actualMinDate, actualMaxDate, format);

        // 第四步：SQL 生成
        StringBuilder result = new StringBuilder();

        if (wrapOuter) {
            result.append("select * from (\n");
        }

        int index = 0;
        for (String formattedDate : dateSequence) {
            // 替换占位符生成子查询
            String subQuery = sql.replace(PLACEHOLDER, formattedDate);
            result.append(subQuery);

            // 添加 union all（最后一个不需要）
            if (index < dateSequence.size() - 1) {
                result.append("\nunion all\n");
            }
            index++;
        }

        if (wrapOuter) {
            result.append("\n) t");  // 添加别名 t，解决 SQL Server 兼容性问题
        }

        return result.toString();
    }

    /**
     * 根据日期范围生成 UNION ALL 查询语句（List 重载版本）
     *
     * <p>此方法是 {@link #renderUnionSqlForDate(JSONArray, String, String)} 的重载版本，支持 List 参数</p>
     *
     * @param arr    List 日期字符串数组，支持格式 yyyy-MM-dd 或 yyyy-MM-dd HH:mm:ss
     * @param sql    SQL 模板，如果包含 [ROW] 占位符则进行替换，否则返回原始 SQL
     * @param format 日期格式化规则，仅支持 yyyy / yyyyMM / yyyyMMdd
     * @return 生成的 UNION ALL 查询语句，外层包裹 select * from (...)；如果没有占位符则返回原始 SQL
     * @throws IllegalArgumentException 如果参数非法、日期格式错误或 format 不支持
     */
    public static String renderUnionSqlForDate(List<String> arr, String sql, String format) {
        return renderUnionSqlForDate(arr, sql, format, true);
    }

    /**
     * 根据日期范围生成 UNION ALL 查询语句（List 重载版本，支持控制包装）
     *
     * @param arr       List 日期字符串数组，支持格式 yyyy-MM-dd 或 yyyy-MM-dd HH:mm:ss
     * @param sql       SQL 模板，如果包含 [ROW] 占位符则进行替换，否则返回原始 SQL
     * @param format    日期格式化规则，仅支持 yyyy / yyyyMM / yyyyMMdd
     * @param wrapOuter 是否包装外层 select * from (...)，true-包装（默认），false-不包装
     * @return 生成的 UNION ALL 查询语句；如果没有占位符则返回原始 SQL
     * @throws IllegalArgumentException 如果参数非法、日期格式错误或 format 不支持
     */
    public static String renderUnionSqlForDate(List<String> arr, String sql, String format, Boolean wrapOuter) {
        // 将 List 转换为 JSONArray（允许为空）
        JSONArray jsonArray = (arr == null) ? null : new JSONArray(arr);

        // 调用核心方法
        return renderUnionSqlForDateInternal(jsonArray, sql, format, null, null, wrapOuter);
    }

    /**
     * 根据日期范围生成 UNION ALL 查询语句（List 重载版本，支持日期范围限制）
     *
     * <p>此方法支持日期范围限制，可以通过 minDate 和 maxDate 参数控制生成的日期序列范围</p>
     *
     * @param arr     List 日期字符串数组，支持格式 yyyy-MM-dd 或 yyyy-MM-dd HH:mm:ss；如果为空则使用 maxDate
     * @param sql     SQL 模板，如果包含 [ROW] 占位符则进行替换，否则返回原始 SQL
     * @param format  日期格式化规则，仅支持 yyyy / yyyyMM / yyyyMMdd
     * @param minDate 最小日期限制（格式 yyyy-MM-dd），null 表示不限制
     * @param maxDate 最大日期限制（格式 yyyy-MM-dd），null 时使用当前日期
     * @return 生成的 UNION ALL 查询语句，外层包裹 select * from (...)；如果没有占位符则返回原始 SQL
     * @throws IllegalArgumentException 如果参数非法、日期格式错误或 format 不支持
     */
    public static String renderUnionSqlForDate(List<String> arr, String sql, String format, String minDate, String maxDate) {
        // 将 List 转换为 JSONArray（允许为空）
        JSONArray jsonArray = (arr == null) ? null : new JSONArray(arr);

        // 调用核心方法
        return renderUnionSqlForDateInternal(jsonArray, sql, format, minDate, maxDate, true);
    }

    /**
     * 根据日期范围生成 UNION ALL 查询语句（List 重载版本，支持日期范围限制和控制包装）
     *
     * @param arr       List 日期字符串数组，支持格式 yyyy-MM-dd 或 yyyy-MM-dd HH:mm:ss；如果为空则使用 maxDate
     * @param sql       SQL 模板，如果包含 [ROW] 占位符则进行替换，否则返回原始 SQL
     * @param format    日期格式化规则，仅支持 yyyy / yyyyMM / yyyyMMdd
     * @param minDate   最小日期限制（格式 yyyy-MM-dd），null 表示不限制
     * @param maxDate   最大日期限制（格式 yyyy-MM-dd），null 时使用当前日期
     * @param wrapOuter 是否包装外层 select * from (...)，true-包装（默认），false-不包装
     * @return 生成的 UNION ALL 查询语句；如果没有占位符则返回原始 SQL
     * @throws IllegalArgumentException 如果参数非法、日期格式错误或 format 不支持
     */
    public static String renderUnionSqlForDate(List<String> arr, String sql, String format, String minDate, String maxDate, Boolean wrapOuter) {
        // 将 List 转换为 JSONArray（允许为空）
        JSONArray jsonArray = (arr == null) ? null : new JSONArray(arr);

        // 调用核心方法
        return renderUnionSqlForDateInternal(jsonArray, sql, format, minDate, maxDate, wrapOuter);
    }

    /**
     * 根据日期范围和格式生成日期序列
     *
     * @param minDate 起始日期
     * @param maxDate 结束日期
     * @param format  格式化规则（yyyy/yyyyMM/yyyyMMdd）
     * @return 格式化后的日期字符串集合（有序且不重复）
     */
    private static LinkedHashSet<String> generateDateSequence(Date minDate, Date maxDate, String format) {
        LinkedHashSet<String> sequence = new LinkedHashSet<>();
        SimpleDateFormat sdf = new SimpleDateFormat(format);

        // 格式化最大日期作为终止条件
        String maxFormatted = sdf.format(maxDate);

        // 根据格式确定日期增量单位
        DateField incrementUnit = switch (format) {
            case FORMAT_YEAR -> DateField.YEAR;
            case FORMAT_MONTH -> DateField.MONTH;
            case FORMAT_DAY -> DateField.DAY_OF_MONTH;
            default -> throw new IllegalArgumentException("不支持的日期格式: " + format);
        };

        // 统一的日期遍历逻辑
        Date current = minDate;
        while (true) {
            String formatted = sdf.format(current);
            sequence.add(formatted);

            // 如果已经到达或超过最大日期，停止
            if (formatted.compareTo(maxFormatted) >= 0) {
                break;
            }

            // 根据增量单位递增日期
            current = DateUtil.offset(current, incrementUnit, 1);
        }

        return sequence;
    }
}

