package com.af.v4.system.common.plugins.concurrent;

import com.af.v4.system.common.core.context.GlobalVisualThreadContext;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.BiConsumer;
import java.util.function.Function;

/**
 * LogicBatchTaskProcessor
 * <p>
 * 一个适用于Logic环境的批量任务处理器，支持将大规模任务列表分块并发处理。
 *
 * <p>该类提供以下功能：
 * <ul>
 *   <li>通过动态计算分块大小，将任务列表拆分为多个子列表</li>
 *   <li>通过虚拟线程池实现高效并发处理</li>
 *   <li>使用者只需专注于单个任务的处理逻辑</li>
 *   <li>自动统计成功和失败的任务数量</li>
 *   <li>支持任务预处理及处理完成后的回调</li>
 *   <li>确保所有任务完成后再返回，适合对结果一致性有要求的场景</li>
 * </ul>
 * <p>
 * Consumer和Function参数通过Logic的lambda表达式传入
 */
public class LogicBatchTaskProcessor extends BatchTaskProcessor<JSONObject> {

    private static final Logger LOGGER = LoggerFactory.getLogger(LogicBatchTaskProcessor.class);

    /**
     * 按分块并发处理任务列表。
     *
     * <p>该方法将任务列表分割为多个任务块，使用虚拟线程池并发执行，
     * 并通过传入的处理逻辑对分块中的每个任务执行操作。任务完成后会打印成功和失败的任务统计信息。
     *
     * @param items           待处理的任务列表，不可为空
     * @param taskType        任务类型（IO 密集型或 CPU 密集型） @class TaskType
     * @param taskHandler     单个任务的处理逻辑，接受一个任务作为输入
     * @param preProcessor    （可选）在分块之前对任务列表进行预处理，返回处理后的任务列表
     * @param successCallback （可选）每个成功处理的任务的回调
     * @param failureCallback （可选）每个失败处理的任务的回调
     * @return 统计结果，包含成功和失败的任务数量、分块数量等
     */
    public JSONObject process(
            JSONArray items,
            String taskType,
            Function<JSONObject, Object> taskHandler,
            Function<JSONObject, Object> preProcessor,
            BiConsumer<JSONObject, Object> successCallback,
            BiConsumer<JSONObject, Exception> failureCallback
    ) {
        TaskType taskTypeEnum = TaskType.toType(taskType);

        // 依据任务列表大小和 CPU 核心数动态计算 chunkSize
        int chunkSize = calculateChunkSize(items.length(), taskTypeEnum);

        // 可选的任务预处理
        JSONArray processedItems;
        if (preProcessor != null) {
            JSONObject params = new JSONObject();
            params.put("items", items);
            processedItems = (JSONArray) preProcessor.apply(params);
        } else {
            processedItems = items;
        }

        // 分割任务列表为多个子列表
        JSONArray chunks = splitList(processedItems, chunkSize);
        List<CompletableFuture<Void>> futures = new ArrayList<>();

        JSONObject result = new JSONObject();
        AtomicInteger successCount = new AtomicInteger(0);
        AtomicInteger failureCount = new AtomicInteger(0);
        // 为每个分块创建异步任务
        for (Object obj : chunks) {
            JSONArray chunk = (JSONArray) obj;
            CompletableFuture<Void> future = GlobalVisualThreadContext.runAsync(() -> {
                for (Object chunkObj : chunk) {
                    JSONObject item = (JSONObject) chunkObj;
                    try {
                        Object resultValue = taskHandler.apply(item);
                        successCount.incrementAndGet(); // 成功计数自增
                        if (successCallback != null) {
                            successCallback.accept(item, resultValue);
                        }
                    } catch (Exception e) {
                        failureCount.incrementAndGet(); // 失败计数自增
                        if (failureCallback != null) {
                            failureCallback.accept(item, e);
                        }
                        LOGGER.error("Error processing task: {}", item, e);
                    }
                }
            });
            futures.add(future);
        }

        // 等待所有任务完成
        CompletableFuture.allOf(futures.toArray(new CompletableFuture[0])).join();

        // 统计结果
        result.put("totalItems", processedItems.length());
        result.put("chunkSize", chunkSize);
        result.put("totalChunks", chunks.length());
        result.put("successCount", successCount.get());
        result.put("failureCount", failureCount.get());

        LOGGER.info("Batch task processing completed: {}", result);
        return result;
    }

    public JSONObject process(
            JSONArray items,
            Function<JSONObject, Object> taskHandler
    ) {
        return process(items, TaskType.IO_INTENSIVE.getValue(), taskHandler, null, null, null);
    }

    /**
     * 将列表按指定大小分割为多个子列表。
     *
     * <p>如果列表长度不能被分块大小整除，则最后一个子列表的大小可能小于分块大小。
     *
     * @param list      待分割的列表，不可为空
     * @param chunkSize 每个分块的大小，必须大于 0
     * @return 按分块大小拆分后的子列表集合
     */
    protected JSONArray splitList(JSONArray list, int chunkSize) {
        JSONArray chunks = new JSONArray();
        int length = list.length();
        for (int i = 0; i < length; i += chunkSize) {
            int end = Math.min(i + chunkSize, length);
            chunks.put(list.subList(i, end));
        }
        return chunks;
    }
}
