package com.af.v4.system.common.plugins.auth;

import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.plugins.date.DateTools;
import com.af.v4.system.common.redis.RedisService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 第三方系统通用鉴权类
 * <p>
 * 带分布式锁实现，避免重复获取token
 *
 * @param <T> 鉴权所属参数类型
 * @param <R> 鉴权结果类型
 */
public abstract class AuthTools<T, R> {

    private static final Logger LOGGER = LoggerFactory.getLogger(AuthTools.class);
    private static final String PUBLIC_ORG_NAME = "Public";
    private final RedisService redisService;

    protected AuthTools(RedisService redisService) {
        this.redisService = redisService;
    }

    /**
     * 鉴权入口
     *
     * @param orgId 组织机构ID，为null时代表公共组织
     * @param authParams 鉴权所需参数
     * @return 鉴权结果
     */
    public final R authorization(String orgId, T authParams) {
        if (!supportsParameterlessAuth() && authParams == null) {
            throw new ServiceException("不支持无参鉴权");
        }
        if (orgId == null) {
            orgId = PUBLIC_ORG_NAME;
        }
        String key = getTokenCacheKey(orgId);
        R tokenObj = redisService.get(key);
        LOGGER.info("{}:获取现有token[{}]的结果:{}", getName(), key, tokenObj);
        if (tokenObj != null) {
            return tokenObj;
        } else {
            String finalOrgId = orgId;
            return redisService.syncLock(key, () -> {
                // 再次尝试从缓存获取，避免集群多实例重复请求获取auth
                R newTokenObj = redisService.get(key);
                if (newTokenObj != null) {
                    LOGGER.info("{}:获取现有token[{}]的结果:{}", getName(), key, newTokenObj);
                    return newTokenObj;
                }
                LOGGER.info("{}平台：{}:进行鉴权操作", getName(), key);
                AuthResult result = getToken(finalOrgId, authParams);
                // 缓存鉴权结果
                redisService.set(key, result.value, result.expiresIn);
                return result.value;
            });
        }
    }

    public final R authorization(String orgId) {
        return authorization(orgId, null);
    }

    public final R authorization() {
        return authorization(null);
    }

    /**
     * 强制删除token
     *
     * @param orgId 组织机构ID，为null时代表公共组织
     */
    public final void forceRefreshToken(String orgId) {
        if (orgId == null) {
            orgId = PUBLIC_ORG_NAME;
        }
        String key = getTokenCacheKey(orgId);
        String finalOrgId = orgId;
        redisService.syncLock(key, () -> {
            // 获取最后删除token时间
            String timeKey = getTokenLastDeleteTimeCacheKey(finalOrgId);
            Long lastDeleteTimeValue = Long.parseLong(redisService.get(timeKey));
            Long now = System.currentTimeMillis();
            if (now - lastDeleteTimeValue < 2 * 60_000) {
                // 2分钟内不重复删除
                return null;
            }
            // 删除鉴权标记，等待下一次重新鉴权
            redisService.delete(key);
            // 缓存最后删除token时间
            redisService.set(timeKey, DateTools.getNow2(), 300);
            return null;
        });
    }

    /**
     * 鉴权实现
     *
     * @param orgId 组织机构ID，为null时代表公共组织
     * @param authParams 鉴权所需参数
     * @return 鉴权结果
     */
    protected abstract AuthResult getToken(String orgId, T authParams);

    /**
     * 鉴权缓存的redisKey
     *
     * @param orgId 组织机构ID，为null时代表公共组织
     * @return key值
     */
    private String getTokenCacheKey(String orgId) {
        return STR."\{getName()}Token@\{orgId}";
    }

    /**
     * 最后删除token时间的redisKey
     *
     * @param orgId 组织机构ID，为null时代表公共组织
     * @return key值
     */
    private String getTokenLastDeleteTimeCacheKey(String orgId) {
        return STR."\{getName()}TokenLastDeleteTime@\{orgId}";
    }

    /**
     * 区分鉴权模块的唯一标识名
     *
     * @return 标识名
     */
    protected abstract String getName();

    /**
     * 鉴权结果对象
     */
    protected final class AuthResult {
        /**
         * 鉴权结果
         */
        private final R value;
        /**
         * 过期时间
         */
        private final Integer expiresIn;

        public AuthResult(R value, Integer expiresIn) {
            this.value = value;
            this.expiresIn = expiresIn;
        }
    }

    /**
     * 是否支持无参鉴权
     */
    protected boolean supportsParameterlessAuth(){
        return true;
    }
}
