package com.af.v4.system.common.plugins.concurrent;

import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import java.util.function.Function;

/**
 * LogicBatchTaskProcessor
 * <p>
 * 一个适用于Logic环境的批量任务处理器，支持将大规模任务列表分块并发处理。
 *
 * <p>该类提供以下功能：
 * <ul>
 *   <li>通过动态计算分块大小，将任务列表拆分为多个子列表</li>
 *   <li>通过虚拟线程池实现高效并发处理</li>
 *   <li>使用者只需专注于单个任务的处理逻辑</li>
 *   <li>自动统计成功和失败的任务数量</li>
 *   <li>支持任务预处理及处理完成后的回调</li>
 *   <li>确保所有任务完成后再返回，适合对结果一致性有要求的场景</li>
 * </ul>
 *
 * Consumer和Function参数通过Logic的lambda表达式传入
 */
public class LogicBatchTaskProcessor extends BatchTaskProcessor<JSONObject> {

    private static final Logger LOGGER = LoggerFactory.getLogger(LogicBatchTaskProcessor.class);

    public JSONObject process(
            JSONArray items,
            String taskType,
            Consumer<JSONObject> taskHandler,
            Function<JSONObject, Object> preProcessor,
            Consumer<JSONObject> successCallback,
            Consumer<JSONObject> failureCallback
    ) {
        TaskType taskTypeEnum = TaskType.toType(taskType);

        // 依据任务列表大小和 CPU 核心数动态计算 chunkSize
        int chunkSize = calculateChunkSize(items.length(), taskTypeEnum);

        // 可选的任务预处理
        JSONArray processedItems;
        if (preProcessor != null) {
            JSONObject params = new JSONObject();
            params.put("items", items);
            processedItems = (JSONArray) preProcessor.apply(params);
        } else {
            processedItems = items;
        }

        // 分割任务列表为多个子列表
        JSONArray chunks = splitList(processedItems, chunkSize);
        List<CompletableFuture<Void>> futures = new ArrayList<>();

        JSONObject result = new JSONObject();
        AtomicInteger successCount = new AtomicInteger(0);
        AtomicInteger failureCount = new AtomicInteger(0);

        try (ExecutorService executor = Executors.newVirtualThreadPerTaskExecutor()) {
            // 为每个分块创建异步任务
            for (Object obj : chunks) {
                JSONArray chunk = (JSONArray) obj;
                CompletableFuture<Void> future = CompletableFuture.runAsync(() -> {
                    for (Object chunkObj : chunk) {
                        JSONObject item = (JSONObject) chunkObj;
                        try {
                            taskHandler.accept(item);
                            successCount.incrementAndGet(); // 成功计数自增
                            if (successCallback != null) {
                                successCallback.accept(item);
                            }
                        } catch (Exception e) {
                            failureCount.incrementAndGet(); // 失败计数自增
                            if (failureCallback != null) {
                                failureCallback.accept(item);
                            }
                            LOGGER.error("Error processing task: {}", item, e);
                        }
                    }
                }, executor);
                futures.add(future);
            }
        }

        // 等待所有任务完成
        CompletableFuture.allOf(futures.toArray(new CompletableFuture[0])).join();

        // 统计结果
        result.put("totalItems", processedItems.length());
        result.put("chunkSize", chunkSize);
        result.put("totalChunks", chunks.length());
        result.put("successCount", successCount.get());
        result.put("failureCount", failureCount.get());

        LOGGER.info("Batch task processing completed: {}", result);
        return result;
    }

    /**
     * 将列表按指定大小分割为多个子列表。
     *
     * <p>如果列表长度不能被分块大小整除，则最后一个子列表的大小可能小于分块大小。
     *
     * @param list      待分割的列表，不可为空
     * @param chunkSize 每个分块的大小，必须大于 0
     * @return 按分块大小拆分后的子列表集合
     */
    protected JSONArray splitList(JSONArray list, int chunkSize) {
        JSONArray chunks = new JSONArray();
        int length = list.length();
        for (int i = 0; i < length; i += chunkSize) {
            int end = Math.min(i + chunkSize, length);
            chunks.put(list.subList(i, end));
        }
        return chunks;
    }
}
