package com.af.v4.system.common.plugins.http.core;

import com.af.v4.system.common.plugins.http.config.HttpClientConfig;
import com.af.v4.system.common.plugins.http.config.HttpRuntimeSupport;
import org.apache.hc.client5.http.async.methods.SimpleHttpRequest;
import org.apache.hc.client5.http.async.methods.SimpleHttpResponse;
import org.apache.hc.client5.http.classic.methods.HttpUriRequestBase;
import org.apache.hc.client5.http.config.RequestConfig;
import org.apache.hc.client5.http.entity.UrlEncodedFormEntity;
import org.apache.hc.core5.http.*;
import org.apache.hc.core5.http.io.entity.EntityUtils;
import org.apache.hc.core5.http.io.entity.StringEntity;
import org.apache.hc.core5.http.message.BasicClassicHttpRequest;
import org.apache.hc.core5.http.message.BasicNameValuePair;
import org.apache.hc.core5.util.Timeout;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

/**
 * 连接池抽象实现
 *
 * @author Mr.river
 */
public abstract class BaseHttpPoolUtil {

    private static final Logger LOGGER = LoggerFactory.getLogger(HttpConnectionPoolUtil.class);

    private static final Long FINAL_SLOW_TIME_VALUE = 2000L;

    /**
     * 对http请求进行基本设置
     *
     * @param httpRequestBase http请求
     */
    protected static void setRequestConfig(HttpClientConfig config, HttpUriRequestBase httpRequestBase) {
        if (config == null) {
            config = new HttpClientConfig.Builder().build();
        }
        RequestConfig requestConfig = RequestConfig.custom()
                .setConnectionRequestTimeout(Timeout.ofMilliseconds(config.getConnectTimeout()))
                .setConnectTimeout(Timeout.ofMilliseconds(config.getConnectTimeout()))
                .setResponseTimeout(Timeout.ofMilliseconds(config.getSocketTimeout())).build();
        httpRequestBase.setConfig(requestConfig);
    }

    /**
     * 设置请求头
     *
     * @param base       基础客户端对象
     * @param headersStr 请求头字符串
     */
    protected static void setHeaders(HttpRequest base, String headersStr) {
        // 设置请求头
        if (headersStr != null && !headersStr.isEmpty()) {
            JSONObject headers = new JSONObject(headersStr);
            Iterator<String> keys = headers.keys();
            while (keys.hasNext()) {
                String key = keys.next();
                String val = String.valueOf(headers.get(key));
                base.setHeader(key, val);
            }
        }
    }

    /**
     * 请求头json转NameValuePair集合
     *
     * @param params 请求头json
     * @return NameValuePair集合
     */
    protected static List<NameValuePair> paramsConverter(JSONObject params) {
        List<NameValuePair> nameValuePairs = new ArrayList<>();
        Set<String> paramsSet = params.keySet();
        for (String key : paramsSet) {
            nameValuePairs.add(new BasicNameValuePair(key, String.valueOf(params.get(key))));
        }
        return nameValuePairs;
    }

    /**
     * 设置请求体
     *
     * @param base 客户端对象
     * @param body 请求体字符串
     */
    protected static void setBody(HttpRequest base, String body) {
        //设置请求体
        if (body != null && !body.isEmpty()) {
            if (base instanceof SimpleHttpRequest request) {
                request.setBody(body, ContentType.TEXT_PLAIN.withCharset(StandardCharsets.UTF_8));
            } else if (base instanceof BasicClassicHttpRequest request) {
                StringEntity se = new StringEntity(body, StandardCharsets.UTF_8);
                request.setEntity(se);
            }
        }
    }

    protected static void setUrlEncodedBody(ClassicHttpRequest base, String body) {
        //设置请求体
        if (!body.isEmpty()) {
            base.setEntity(new UrlEncodedFormEntity(paramsConverter(new JSONObject(body)), StandardCharsets.UTF_8));
        }
    }

    /**
     * 获取响应体数据
     *
     * @param begin    计数器-开始时间
     * @param response 响应体数据
     * @return 响应数据
     */
    public static String getResponseData(long begin, HttpResponse response) throws IOException, ParseException {
        String result = null;
        try {
            String code = String.valueOf(response.getCode());
            if (response instanceof ClassicHttpResponse classicHttpResponse) {
                HttpEntity entity = classicHttpResponse.getEntity();
                if (entity != null) {
                    result = EntityUtils.toString(entity, StandardCharsets.UTF_8);
                }
            } else if (response instanceof SimpleHttpResponse simpleHttpResponse) {
                result = new String(simpleHttpResponse.getBodyBytes(), StandardCharsets.UTF_8);
            }
            // 请求失败时的处理
            if (!code.startsWith(String.valueOf(HttpStatus.SC_OK).substring(0, 1))) {
                String reasonPhrase = response.getReasonPhrase();
                // 返回错误信息
                JSONObject errorResult = new JSONObject();
                errorResult.put("code", code).put("data", result).put("msg", reasonPhrase);
                result = errorResult.toString();
            }
        } finally {
            boolean disableLogPrint = HttpRuntimeSupport.getDisableLogPrintValue();
            if (!disableLogPrint) {
                long end = System.currentTimeMillis();
                long time = end - begin;
                String text = "请求接口耗时：" + time + "ms";
                if (time >= FINAL_SLOW_TIME_VALUE) {
                    LOGGER.warn(text);
                } else {
                    LOGGER.info(text);
                }
            }
        }
        return result;
    }

    public static JSONObject buildHeader(String headers) {
        JSONObject headerJson;
        if (headers == null || headers.isEmpty()) {
            headerJson = new JSONObject();
        } else {
            headerJson = new JSONObject(headers);
        }
        if (!headerJson.has(HttpHeaders.CONTENT_TYPE)) {
            //指定请求参数的数据格式是JSON。
            headerJson.put(HttpHeaders.CONTENT_TYPE, ContentType.APPLICATION_JSON.getMimeType());
        }
        return headerJson;
    }
}
