package com.af.v4.system.common.plugins.json;

import com.af.v4.system.common.plugins.core.CommonTools;
import com.af.v4.system.common.plugins.io.IOTools;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.json.JSONArray;
import org.json.JSONObject;

import java.math.BigDecimal;
import java.util.List;
import java.util.Set;

/**
 * json工具类
 *
 * @author Mr.river
 */
public class JsonTools {

    private static final ObjectMapper objectMapper;

    static {
        objectMapper = new ObjectMapper();
        objectMapper.setSerializationInclusion(JsonInclude.Include.NON_EMPTY);
        objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
    }

    /**
     * 给一个JSONObject追加一个JSONObject中的所有值(deep clone)
     *
     * @param obj    被追加的JSONObject
     * @param addObj 追加的JSONObject
     * @return 追加完的JSONObject
     */
    public static JSONObject addJSON(JSONObject obj, JSONObject addObj) {
        if (addObj == null) {
            throw new NullPointerException("追加的JSONObject不存在！");
        }
        Set<String> keys = addObj.keySet();
        for (String key : keys) {
            Object item = addObj.get(key);
            if (item instanceof JSONObject json) {
                JSONObject cloneItem = new JSONObject(json.toMap());
                obj.put(key, cloneItem);
            } else if (item instanceof JSONArray array) {
                JSONArray cloneItem = new JSONArray(array.toList());
                obj.put(key, cloneItem);
            } else {
                obj.put(key, item);
            }
        }
        return obj;
    }

    /**
     * 将字符串转化为json
     *
     * @param str 要转为json的字符串
     * @return 原生json
     */
    public static JSONObject convertToJson(String str) {
        if (str == null || "".equals(str)) {
            return new JSONObject();
        }
        return new JSONObject(str);
    }

    public static JSONObject convertToJson(JSONObject json) {
        return json;
    }

    /**
     * 把字符串转换成JSON数组
     *
     * @param str 需要转换的String
     * @return 转换完成后的json数组
     */
    public static JSONArray parseArray(String str) {
        return new JSONArray(str);
    }

    public static JSONArray parseArray(JSONArray json) {
        return json;
    }

    /**
     * 返回一个JSON数组实例
     *
     * @return JSON数组实例
     */
    public static JSONArray getArray() {
        return new JSONArray();
    }

    /**
     * 返回一个制定长度的JSON数组实例
     *
     * @param val 长度
     * @return JSONArray
     */
    public static JSONArray getArray(Object val) {
        int len;
        try {
            len = new BigDecimal(val.toString()).intValue();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
        return new JSONArray(new String[len]);
    }

    /**
     * 返回一个JSON实例
     *
     * @return JSON数组实例
     */
    public static JSONObject getJson() {
        return new JSONObject();
    }

    /**
     * JSON键值对替换（保留原JSON中的键值对）
     *
     * @param oldObj 被替换的JSON
     * @param newObj 替换的JSON
     * @return 处理后的JSON
     */
    public static JSONObject replaceJSON(JSONObject oldObj, JSONObject newObj) {
        for (String key : newObj.keySet()) {
            //如果新JSON中有原JSON的键值对
            oldObj.put(key, newObj.get(key));
        }
        return oldObj;
    }

    /**
     * 读取JSON文件，返回JSON对象
     *
     * @param fileName 文件路径
     * @return JSONObject
     */
    public static JSONObject readJsonFile(String fileName) {
        return new JSONObject(IOTools.readText(fileName));
    }

    /**
     * 读取JSON文件，返回JSON数组
     *
     * @param fileName 文件路径
     * @return JSONArray
     */
    public static JSONArray readJsonArrayFile(String fileName) {
        return new JSONArray(IOTools.readText(fileName));
    }

    /**
     * 根据参数字典格式化Json
     *
     * @param object Json
     * @param params 参数字典
     * @return 格式化后的Json
     */
    public static JSONObject formatJsonByParams(JSONObject object, JSONObject params) {
        if (object.length() > 0) {
            return new JSONObject(CommonTools.formatStrByParams(object.toString(), params));
        }
        return object;
    }

    /**
     * 根据参数字典格式化Json集合
     *
     * @param array  Json集合
     * @param params 参数字典
     * @return 格式化后的Json集合
     */
    public static JSONArray formatJsonArrayByParams(JSONArray array, JSONObject params) {
        if (array.length() > 0) {
            return new JSONArray(CommonTools.formatStrByParams(array.toString(), params));
        }
        return array;
    }

    /**
     * 将json转化为实体POJO
     *
     * @param json org.json
     * @param obj  转换的实体类
     * @return 实体POJO
     */
    public static <T> T toParse(JSONObject json, Class<T> obj) {
        return toParse(json.toString(), obj);
    }

    public static <T> T toParse(String json, Class<T> obj) {
        T t = null;
        try {
            t = objectMapper.readValue(json, obj);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return t;
    }

    public static <T> List<T> toParseList(JSONArray json, Class<T> obj) {
        List<T> t = null;
        try {
            t = objectMapper.readValue(json.toString(), getCollectionType(List.class, obj));
        } catch (Exception e) {
            e.printStackTrace();
        }
        return t;
    }

    /**
     * 将实体POJO转换为json
     *
     * @param entity 被转换的实体类
     * @return org.json
     */
    public static JSONObject toJSON(Object entity) {
        try {
            return new JSONObject(objectMapper.writeValueAsString(entity));
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    public static <T> JSONArray toJsonArray(List<T> entity) {
        try {
            return new JSONArray(objectMapper.writeValueAsString(entity));
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * 获取泛型的Collection Type
     *
     * @param collectionClass 泛型的Collection
     * @param elementClasses  实体bean
     * @return JavaType Java类型
     */
    private static JavaType getCollectionType(Class<?> collectionClass, Class<?>... elementClasses) {
        return objectMapper.getTypeFactory().constructParametricType(collectionClass, elementClasses);
    }

    /**
     * 对象转json
     *
     * @param obj 任意类型对象(需要 getter 和 setter 方法)
     * @return json对象
     */
    public static <T> JSONObject objToJson(T obj) {
        try {
            return new JSONObject(objectMapper.writeValueAsString(obj));
        } catch (JsonProcessingException e) {
            e.printStackTrace();
            return new JSONObject();
        }
    }
}
