package com.af.v4.system.common.plugins.http;

import org.apache.hc.client5.http.async.methods.SimpleHttpRequest;
import org.apache.hc.client5.http.async.methods.SimpleHttpResponse;
import org.apache.hc.client5.http.impl.async.CloseableHttpAsyncClient;
import org.apache.hc.client5.http.impl.async.HttpAsyncClients;
import org.apache.hc.client5.http.impl.nio.PoolingAsyncClientConnectionManager;
import org.apache.hc.client5.http.impl.nio.PoolingAsyncClientConnectionManagerBuilder;
import org.apache.hc.client5.http.ssl.DefaultClientTlsStrategy;
import org.apache.hc.core5.concurrent.FutureCallback;
import org.apache.hc.core5.http.config.Registry;
import org.apache.hc.core5.http.config.RegistryBuilder;
import org.apache.hc.core5.http.nio.ssl.TlsStrategy;
import org.apache.hc.core5.reactor.IOReactorConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * HttpAsyncClient连接池实现
 *
 * @author Mr.river
 * @see com.af.v4.system.common.plugins.http.RestAsyncTools
 */
public class HttpAsyncConnectionPoolUtil extends BaseHttpPoolUtil {

    private static final Logger logger = LoggerFactory.getLogger(HttpAsyncConnectionPoolUtil.class);
    /**
     * 相当于线程锁,用于线程安全
     */
    private final static Object SYNC_LOCK = new Object();
    /**
     * 发送HTTP异步请求的客户端单例
     */
    private static volatile CloseableHttpAsyncClient httpClient;

    /**
     * 获取httpclient的异步https实例（需要SSL证书）
     *
     * @param sslSocketFactory SSL构建器
     * @return httpclient实例
     */
    public static CloseableHttpAsyncClient getHttpClient(DefaultClientTlsStrategy sslSocketFactory) {
        return createHttpClient(sslSocketFactory);
    }

    /**
     * 构建httpclient实例
     *
     * @return httpclient实例
     */
    private static CloseableHttpAsyncClient createHttpClient() {
        return createHttpClient(null);
    }

    /**
     * 构建httpclient实例
     *
     * @param sslSocketFactory SSL构建器
     * @return httpclient实例
     */
    private static CloseableHttpAsyncClient createHttpClient(DefaultClientTlsStrategy sslSocketFactory) {
        //设置https相关信息
        if (sslSocketFactory == null) {
            logger.debug("创建HTTP异步客户端会话");
            sslSocketFactory = (DefaultClientTlsStrategy) DefaultClientTlsStrategy.getDefault();
        } else {
            logger.debug("创建https异步客户端会话");
        }
        Registry<TlsStrategy> registry = RegistryBuilder.<TlsStrategy>create()
                .register("https", sslSocketFactory)
                .build();
        // 配置io线程
        IOReactorConfig ioReactorConfig = IOReactorConfig.custom()
                .setIoThreadCount(Runtime.getRuntime().availableProcessors() * 2)
                .setSoKeepAlive(true)
                .build();
        PoolingAsyncClientConnectionManager manager =
                PoolingAsyncClientConnectionManagerBuilder.create()
                        .setTlsStrategy(registry.lookup("https"))
                        .build();

        /* 设置连接参数 **/
        // 最大连接数
        manager.setMaxTotal(HttpClientConfig.HTTP_MAX_POOL_SIZE);
        // 路由最大连接数
        manager.setDefaultMaxPerRoute(HttpClientConfig.HTTP_MAX_POOL_SIZE);

        return HttpAsyncClients.custom()
                .setIOReactorConfig(ioReactorConfig)
                .setConnectionManager(manager)
                .build();
    }

    /**
     * 获取HTTP异步请求 httpclient实例
     *
     * @return httpclient实例
     */
    public static CloseableHttpAsyncClient getHttpClient() {
        if (httpClient == null) {
            //多线程下同时调用getHttpClient容易导致重复创建httpClient对象的问题,所以加上了同步锁
            synchronized (SYNC_LOCK) {
                if (httpClient == null) {
                    httpClient = createHttpClient();
                }
            }
        }
        httpClient.start();
        return httpClient;
    }

    /**
     * 发送通用HTTP异步请求
     *
     * @param value          请求参数
     * @param headersStr     请求头
     * @param config         HttpClient配置文件
     * @param base           请求类型
     * @param httpClient     client对象
     * @param futureCallback 异步回调处理函数
     */
    public static void request(String value,
                               String headersStr,
                               HttpClientConfig config,
                               SimpleHttpRequest base,
                               CloseableHttpAsyncClient httpClient,
                               FutureCallback<SimpleHttpResponse> futureCallback) {
        logger.info("异步请求地址：" + base.getScheme() + "://" + base.getAuthority() + base.getPath());
        if (httpClient == null) {
            httpClient = getHttpClient();
        }
        if (futureCallback == null) {
            futureCallback = new InnerAsyncResponse();
        }
        //配置请求参数
        setRequestConfig(config, base);
        //设置请求头
        setHeaders(base, headersStr);
        //设置请求体
        if (base.getBody() == null) {
            setBody(base, value);
        }
        httpClient.execute(base, futureCallback);
    }

    public static class InnerAsyncResponse implements FutureCallback<SimpleHttpResponse> {

        final long begin = System.currentTimeMillis();

        @Override
        public void completed(SimpleHttpResponse response) {
            String result = null;
            try {
                result = getResponseData(begin, response);
            } catch (Exception e) {
                this.failed(e);
            }
            logger.info("异步请求结果：" + result);
        }

        @Override
        public void failed(Exception ex) {
            logger.error("异步请求时发生异常：", ex);
        }

        @Override
        public void cancelled() {

        }
    }
}
