package com.af.v4.system.common.plugins.wechat;

import com.af.v4.system.common.plugins.http.RestTools;
import com.af.v4.system.common.redis.RedisService;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;

@Component
public class WechatApiUtils {

    public static final Integer ERROR_AUTH_CODE = 40001;
    static final Logger LOGGER = LoggerFactory.getLogger(WechatApiUtils.class);
    /**
     * 微信apiToken缓存key名
     */
    private static final String WECHAT_API_TOKEN_CACHE_KEY = "wechatApiToken@";
    private final RedisService redisService;

    public WechatApiUtils(RedisService redisService) {
        this.redisService = redisService;
    }

    private static String getWechatApiTokenRedisKey(String appid) {
        return WECHAT_API_TOKEN_CACHE_KEY + appid;
    }

    /**
     * 获取TOKEN
     *
     * @param appid        appid
     * @param secret       secret
     * @param forceRefresh 是否强制刷新
     * @return TOKEN值
     */
    public String getAuthToken(String appid, String secret, Boolean forceRefresh) {
        String key = getWechatApiTokenRedisKey(appid);
        String token = redisService.get(key);
        if (token != null) {
            LOGGER.debug("{}:获取现有token", appid);
            return token;
        } else {
            synchronized (WechatApiUtils.class) {
                return redisService.syncLock(key, () -> {
                    // 再次尝试从缓存获取，避免集群多实例重复请求获取auth
                    String newToken = redisService.get(key);
                    if (newToken != null) {
                        LOGGER.debug("{}:获取现有token", appid);
                        return newToken;
                    }
                    if (forceRefresh) {
                        LOGGER.info("微信：{}:进行强制刷新鉴权操作", appid);
                    } else {
                        LOGGER.info("微信：{}:进行鉴权操作", appid);
                    }
                    //组织请求报文
                    JSONObject content = new JSONObject();
                    content.put("grant_type", "client_credential");
                    content.put("appid", appid);
                    content.put("secret", secret);
                    content.put("force_refresh", forceRefresh);
                    JSONObject result = new JSONObject(RestTools.post(WechatApi.getAuthUrl(), content));
                    if (result.has("errcode")) {
                        throw new RuntimeException(result.toString());
                    } else {
                        String accessToken = result.getString("access_token");
                        int expiresIn = result.getInt("expires_in");
                        redisService.set(key, accessToken, expiresIn - 180);
                        return accessToken;
                    }
                });
            }
        }
    }

    public String getAuthToken(String appid, String secret) {
        return getAuthToken(appid, secret, false);
    }

    /**
     * 获取所有关注了公众号的所有用户OpenId
     *
     * @param openIdArray 接受openId结果的列表
     * @param accessToken accessToken，调用接口凭证
     * @param nextOpenIds 第一个拉取的OPENID，不填默认从头开始拉取
     * @return OpenId列表
     */
    private List<String> getAllOpenId(List<String> openIdArray, String accessToken, String nextOpenIds) {
        String url = WechatApi.getGetOpenidUrl() + "?access_token=" + accessToken;
        if (nextOpenIds != null) {
            url += "&next_openid=" + nextOpenIds;
        }
        JSONObject result = new JSONObject(RestTools.get(url));
        if (result.has("data")) {
            String nextOpenId = result.getString("next_openid");
            JSONObject data = result.getJSONObject("data");
            int count = result.getInt("count");
            JSONArray openIds = data.getJSONArray("openid");
            for (Object openId : openIds) {
                openIdArray.add((String) openId);
            }
            if (count < 10000) {
                return openIdArray;
            } else {
                return getAllOpenId(openIdArray, accessToken, nextOpenId);
            }
        } else {
            throw new RuntimeException("获取所有用户的OpenId失败，原因：" + result);
        }
    }

    public List<String> getAllOpenId(JSONObject authInfo) {
        String accessToken = getAuthToken(authInfo.getString("appid"), authInfo.getString("secret"));
        List<String> allOpenIds = new ArrayList<>(10000);
        return getAllOpenId(allOpenIds, accessToken, null);
    }

    /**
     * 发送模板消息
     */
    public JSONObject sendTemplateMessage(JSONObject authInfo, String openId, String templateId, String url, String miniprogram, String appid, String pagepath, JSONObject data, String orgName) {
        JSONObject body = new JSONObject();
        body.put("touser", openId);
        body.put("template_id", templateId);
        body.put("url", url);
        body.put("miniprogram", miniprogram);
        body.put("appid", appid);
        body.put("pagepath", pagepath);
        body.put("data", data);
        body.put("orgname", orgName);
        String accessToken = getAuthToken(authInfo.getString("appid"), authInfo.getString("secret"));
        String requestUrl = WechatApi.getSendTemplateMessageUrl() + "?access_token=" + accessToken;
        return new JSONObject(RestTools.post(requestUrl, body));
    }

    /**
     * 清空api的调用quota
     *
     * @apiNote 本接口用于清空公众号/小程序/第三方平台等接口的每日调用接口次数
     */
    public JSONObject clearQuota(JSONObject authInfo) {
        JSONObject body = new JSONObject();
        body.put("appid", authInfo.getString("appid"));
        String accessToken = getAuthToken(authInfo.getString("appid"), authInfo.getString("secret"));
        String requestUrl = WechatApi.getClearQuotaUrl() + "?access_token=" + accessToken;
        return new JSONObject(RestTools.post(requestUrl, body));
    }
}
