package com.af.v4.system.common.plugins.core;

import cn.hutool.core.convert.Convert;
import cn.hutool.core.util.HexUtil;
import cn.hutool.core.util.StrUtil;
import org.json.JSONArray;

import java.io.UnsupportedEncodingException;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.nio.charset.StandardCharsets;
import java.text.NumberFormat;
import java.util.Base64;

/**
 * 数据转换工具类
 *
 * @author Mr.river
 */
public class ConvertTools {
    static final NumberFormat NF;

    static {
        NF = NumberFormat.getInstance();
        NF.setGroupingUsed(false);
        NF.setMinimumFractionDigits(2);
        NF.setMaximumFractionDigits(4);
    }

    /**
     * 将数字按标准格式转换
     *
     * @param value 数字
     * @return 标准格式字符串
     */
    public static String formatNumber(Object value) {
        return PriceFormatTools.formatNumberBase(value);
    }

    /**
     * 字符串转整数
     *
     * @param str 字符串
     * @return 整数
     */
    public static int stringToInt(String str) {
        return Convert.toInt(str);
    }

    /**
     * 字符串转双精度浮点数
     *
     * @param str 字符串
     * @return 双精度浮点数
     */
    public static double stringToDouble(String str) {
        return Convert.toDouble(str);
    }

    /**
     * String转换为bigDecimal
     *
     * @param num 数字字符串
     * @return bigDecimal类型
     */
    public static BigDecimal stringToBigDecimal(String num) {
        return Convert.toBigDecimal(num);
    }

    /**
     * bigDecimal转换为String
     *
     * @param bigDecimal bigDecimal类型
     * @return String 数字字符串
     */
    public static String bigDecimalToString(BigDecimal bigDecimal) {
        return Convert.toStr(bigDecimal);
    }

    /**
     * 字节数组转UTF-8编码字符串
     *
     * @param bytes 字节数组
     * @return 字符串
     * @throws UnsupportedEncodingException 异常
     */
    public static String byteArrayToString(byte[] bytes) throws UnsupportedEncodingException {
        return byteArrayToCharsetString(bytes, "UTF-8");
    }

    /**
     * 字节数组转为指定编码格式的字符串
     *
     * @param bytes   字节数组
     * @param charset 编码格式
     * @return 字符串
     * @throws UnsupportedEncodingException 异常
     */
    public static String byteArrayToCharsetString(byte[] bytes, String charset) throws UnsupportedEncodingException {
        return new String(bytes, charset);
    }

    /**
     * 将byte数组进行Base64加密
     *
     * @param bytes 待加密的byte数组
     * @return base64加密字符串
     */
    public static String base64Encode(byte[] bytes) {
        Base64.Encoder encoder = Base64.getEncoder();
        return new String(encoder.encode(bytes), StandardCharsets.UTF_8);
    }

    /**
     * 将base64加密的byte数组进行解密
     *
     * @param bytes 待解密的byte数组
     * @return base64解密后的byte数组
     */
    public static byte[] base64Decode(byte[] bytes) {
        Base64.Decoder decoder = Base64.getDecoder();
        return decoder.decode(bytes);
    }

    /**
     * byte数组转16进制字符串
     *
     * @param bytes byte数组
     * @return 16进制字符串
     */
    public static String byteToHexStr(byte[] bytes) {
        return HexUtil.encodeHexStr(bytes);
    }

    /**
     * 16进制字符串转byte数组
     *
     * @param hexStr 16进制字符串
     * @return byte数组
     */
    public static byte[] hexStrToByte(String hexStr) {
        return HexUtil.decodeHex(hexStr);
    }


    /**
     * 字符串转byte[]数组
     *
     * @param str 字符串
     * @return byte[]数组
     */
    public static byte[] strToBytes(String str) {
        return str.getBytes(StandardCharsets.UTF_8);
    }

    /**
     * byte[]转字符串功能
     *
     * @param bytes byte[]数组
     * @return 字符串
     */
    public static String bytesToStr(byte[] bytes) {
        return new String(bytes, StandardCharsets.UTF_8);
    }

    /**
     * 指定编码格式转换字符串
     *
     * @param str        字符串
     * @param oldCharset 原编码格式
     * @param newCharset 新编码格式
     * @return 字符串
     */
    public static String stringToCharset(String str, String oldCharset, String newCharset) {
        if (null != str) {
            try {
                str = new String(str.getBytes(oldCharset), newCharset);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return str;
    }

    /**
     * 十六进制转二进制字符串(反转)
     *
     * @param hexString 十六进制字符串
     * @return 二进制字符串
     */
    public static String hexStrToBinaryStr(String hexString) {
        StringBuilder bString = new StringBuilder();
        String tmp;
        for (int i = 0; i < hexString.length(); i++) {
            tmp = "0000" + Integer.toBinaryString(Integer.parseInt(hexString.substring(i, i + 1), 16));
            bString.append(tmp.substring(tmp.length() - 4));
        }
        if (bString.length() == 4) {
            //补足0
            bString = new StringBuilder("0000").append(bString);
        }
        //反转
        bString.reverse();
        return bString.toString();
    }

    /**
     * 二进制转十六进制字符串
     *
     * @param bString 二进制字符串
     * @return 十六进制字符串
     */
    public static String binaryStrToHexStr(String bString) {
        StringBuilder tmp = new StringBuilder();
        int iTmp;
        for (int i = 0; i < bString.length(); i += 4) {
            iTmp = 0;
            for (int j = 0; j < 4; j++) {
                iTmp += Integer.parseInt(bString.substring(i + j, i + j + 1)) << (4 - j - 1);
            }
            tmp.append(Integer.toHexString(iTmp));
        }
        return tmp.toString();
    }

    /**
     * 二进制字符串转数组
     *
     * @param binaryString 二进制字符串
     * @return 二进制数组
     */
    public static byte[] binaryStrToBinaryArray(String binaryString) {
        byte[] result = new byte[binaryString.length()];
        for (int i = 0; i < binaryString.length(); ++i) {
            result[i] = (byte) "0123456789ABCDEF".indexOf(binaryString.charAt(i));
        }
        return result;
    }

    /**
     * 向上取整
     *
     * @param numObj double类型参数
     * @return 向上取整后的整数
     */
    public static int ceil(Object numObj) {
        BigDecimal b = new BigDecimal(numObj.toString());
        return b.setScale(0, RoundingMode.UP).intValue();
    }

    public static double ceil(Object numObj, Object scale) {
        BigDecimal b = new BigDecimal(numObj.toString());
        return b.setScale(Integer.parseInt(String.valueOf(scale)), RoundingMode.UP).doubleValue();
    }

    /**
     * 向下取整
     *
     * @param numObj double类型参数
     * @return 向下取整后的整数
     */
    public static int floor(Object numObj) {
        BigDecimal b = new BigDecimal(numObj.toString());
        return b.setScale(0, RoundingMode.DOWN).intValue();
    }

    public static double floor(Object numObj, Object scale) {
        BigDecimal b = new BigDecimal(numObj.toString());
        return b.setScale(Integer.parseInt(String.valueOf(scale)), RoundingMode.DOWN).doubleValue();
    }

    /**
     * 四舍五入
     *
     * @param numObj double类型参数
     * @return 四舍五入后的参数
     */
    public static int round(Object numObj) {
        BigDecimal b = new BigDecimal(numObj.toString());
        return b.setScale(0, RoundingMode.HALF_UP).intValue();
    }

    public static double round(Object numObj, Object scale) {
        BigDecimal b = new BigDecimal(numObj.toString());
        return b.setScale(Integer.parseInt(String.valueOf(scale)), RoundingMode.HALF_UP).doubleValue();
    }

    /**
     * 去除字符串所有空格
     *
     * @param str 待处理字符串
     * @return 处理后的字符串
     */
    public static String replaceAll(String str) {
        return str.replace(" ", "");
    }

    public static JSONArray splitString(String str, String flag) {
        if ("yAxis".equals(flag)) {
            flag = "\\|";
        }
        return new JSONArray(StrUtil.split(str, flag));
    }
}
