package com.af.v4.system.common.plugins.calculate;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.math.BigDecimal;
import java.math.RoundingMode;

/**
 * 数字类工具, 主要用于数据判断
 *
 * @author 张琪
 * @date 2022/9/14 18:21
 */
public class NumberUtil {
    /**
     * 精确到四位小数
     */
    static final int SCALE_NUM = 4;
    private static final Logger log = LoggerFactory.getLogger(NumberUtil.class);
    /**
     * 小数进行取整操作
     */
    private static final RoundingMode DOWN = RoundingMode.DOWN;


    /**
     * 判断传入字符串是否为数字
     *
     * @param str 待判断字符串
     * @return 是否为数字
     */
    public static boolean isNumeric(String str) {
        try {
            new BigDecimal(str);
        } catch (Exception e) {
            //异常 说明包含非数字。
            return false;
        }
        return true;
    }


    /**
     * 获取 BigDecimal 对象
     *
     * @param bigDecimal 待转化的字符串
     * @return 转化后对象
     */
    public static BigDecimal getBigDecimal(Object bigDecimal) {
        return getBigDecimal(bigDecimal, SCALE_NUM, DOWN);
    }

    /**
     * 自定义转换对象
     *
     * @param bigDecimal 待转化对象
     * @param scaleNum   有效位数
     * @param mode       约分格式
     * @return 转化后对象
     */
    public static BigDecimal getBigDecimal(Object bigDecimal, int scaleNum, RoundingMode mode) {
        return commonParse(parseBigDecimal(bigDecimal), scaleNum, mode);
    }


    /**
     * 对象转换为 BigDecimal
     *
     * @param o 待转换对象
     * @return 转换后数字
     */
    public static BigDecimal parseBigDecimal(Object o) {
        try {
            return new BigDecimal(String.valueOf(o));
        } catch (Exception e) {
            log.error("数据转换异常, 请检查参数");
            return null;
        }
    }

    /**
     * 奥枫 - 通用格式化
     *
     * @param bigDecimal 待格式化数据
     * @return 四舍五入, 保留 4 为小数的数据
     */
    public static BigDecimal commonParse(Object bigDecimal) {
        return commonParse(bigDecimal, SCALE_NUM, DOWN);
    }


    /**
     * 奥枫 - 通用格式化
     *
     * @param bigDecimal 待格式化数据
     * @return 四舍五入, 保留 4 为小数的数据
     */
    public static BigDecimal commonParse(BigDecimal bigDecimal) {
        return commonParse(bigDecimal, SCALE_NUM, DOWN);
    }


    /**
     * 数据格式化
     *
     * @param bigDecimal 待格式化数据
     * @param scaleNum   有效位数
     * @param mode       约分格式
     * @return 格式化后的数据
     */
    public static BigDecimal commonParse(Object bigDecimal, int scaleNum, RoundingMode mode) {
        return commonParse(getBigDecimal(bigDecimal), scaleNum, mode);
    }


    /**
     * 数据格式化
     *
     * @param bigDecimal 待格式化数据
     * @param scaleNum   有效位数
     * @param mode       约分格式
     * @return 格式化后的数据
     */
    public static BigDecimal commonParse(BigDecimal bigDecimal, int scaleNum, RoundingMode mode) {
        if (scaleNum < 0) {
            throw new IllegalArgumentException("有效位数不能小于 0 ");
        }
        return bigDecimal.setScale(scaleNum, mode);
    }


    /**
     * 正数向上约分, 负数向下约分
     * 例: 5.1 --> 6   -1.1 --> -1
     *
     * @param bigDecimal 待约分数字
     * @return 约分后数字
     */
    public static BigDecimal ceil(BigDecimal bigDecimal) {
        return ceil(bigDecimal, SCALE_NUM);
    }

    public static BigDecimal ceil(BigDecimal bigDecimal, Integer scale) {
        return commonParse(bigDecimal, scale, RoundingMode.CEILING);
    }

    /**
     * 四舍五入 例: 4.4 --> 4     4.5-->5 -1.2 --> -1  -1.5 --> -2
     *
     * @param bigDecimal 待约分数字
     * @return 约分后数字
     */
    public static BigDecimal round(BigDecimal bigDecimal) {
        return round(bigDecimal, SCALE_NUM);
    }

    public static BigDecimal round(BigDecimal bigDecimal, Integer scale) {
        return commonParse(bigDecimal, scale, RoundingMode.HALF_UP);
    }
}
