package com.af.v4.system.common.plugins.calculate;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.math.BigDecimal;
import java.math.RoundingMode;

import static com.af.v4.system.common.plugins.calculate.NumberUtil.*;
import static java.math.RoundingMode.DOWN;

/**
 * 计算工具类, 主要用于数值计算
 *
 * @author 张琪
 * @date 2022/9/14 18:14
 */
public final class CalculateUtil {
    private static final Logger log = LoggerFactory.getLogger(CalculateUtil.class);

    /**
     * 奥枫 - logic通用加法 - 除logic外不建议使用
     *
     * @param addOne     加数
     * @param addAnother 另一个加数
     * @return {@code addOne + addAnother}
     */
    public static BigDecimal add(Object addOne, Object addAnother) {
        return add(addOne, addAnother, SCALE_NUM);
    }


    /**
     * 奥枫 - logic通用加法 - 除logic外不建议使用
     *
     * @param addOne     加数
     * @param addAnother 另一个加数
     * @param scaleNum   有效位数
     * @return {@code addOne + addAnother}
     */
    public static BigDecimal add(Object addOne, Object addAnother, Integer scaleNum) {
        return add(getBigDecimal(addOne), getBigDecimal(addAnother), scaleNum, DOWN);
    }

    /**
     * 奥枫 - 通用加法
     *
     * @param addOne     加数
     * @param addAnother 另一个加数
     * @return {@code addOne + addAnother}
     */
    public static BigDecimal add(BigDecimal addOne, BigDecimal addAnother) {
        return add(addOne, addAnother, SCALE_NUM, DOWN);
    }

    /**
     * 自定义加法
     *
     * @param addOne     加数
     * @param addAnother 另一个加数
     * @param scaleNum   有效位数
     * @param mode       约分格式
     * @return {@code addOne + addAnother}
     */
    public static BigDecimal add(BigDecimal addOne, BigDecimal addAnother, int scaleNum, RoundingMode mode) {
        return commonParse(addOne.add(addAnother), scaleNum, mode);
    }

    /**
     * 奥枫 - logic通用减法 - 除logic外不建议使用
     *
     * @param minuend    被减数
     * @param subtrahend 减数
     * @return {@code minuend - subtrahend}
     */
    public static BigDecimal sub(Object minuend, Object subtrahend) {
        return sub(minuend, subtrahend, SCALE_NUM);
    }


    /**
     * 奥枫 - logic通用减法 - 除logic外不建议使用
     *
     * @param minuend    被减数
     * @param subtrahend 减数
     * @param scaleNum   有效位数
     * @return {@code minuend - subtrahend}
     */
    public static BigDecimal sub(Object minuend, Object subtrahend, Integer scaleNum) {
        return sub(getBigDecimal(minuend), getBigDecimal(subtrahend), scaleNum, DOWN);
    }

    /**
     * 奥枫 - 通用减法
     *
     * @param minuend    被减数
     * @param subtrahend 减数
     * @return {@code minuend - subtrahend}
     */
    public static BigDecimal sub(BigDecimal minuend, BigDecimal subtrahend) {
        return sub(minuend, subtrahend, SCALE_NUM, DOWN);
    }

    /**
     * 自定义减法
     *
     * @param minuend    被减数
     * @param subtrahend 减数
     * @param scaleNum   有效位数
     * @param mode       约分格式
     * @return {@code minuend - subtrahend}
     */
    public static BigDecimal sub(BigDecimal minuend, BigDecimal subtrahend, int scaleNum, RoundingMode mode) {
        return commonParse(minuend.subtract(subtrahend), scaleNum, mode);
    }

    /**
     * 奥枫 - logic通用乘法 - 除logic外不建议使用
     *
     * @param multiplierOne     乘数
     * @param multiplierAnother 另一个乘数
     * @return {@code multiplierOne * multiplierAnother}
     */
    public static BigDecimal mul(Object multiplierOne, Object multiplierAnother) {
        return mul(multiplierOne, multiplierAnother, SCALE_NUM);
    }

    /**
     * 奥枫 - logic通用乘法 - 除logic外不建议使用
     *
     * @param multiplierOne     乘数
     * @param multiplierAnother 另一个乘数
     * @param scaleNum          有效位数
     * @return {@code multiplierOne * multiplierAnother}
     */
    public static BigDecimal mul(Object multiplierOne, Object multiplierAnother, Integer scaleNum) {
        return mul(getBigDecimal(multiplierOne), getBigDecimal(multiplierAnother), scaleNum, DOWN);
    }

    /**
     * 奥枫 - 通用乘数
     *
     * @param multiplierOne     乘数
     * @param multiplierAnother 另一个乘数
     * @return {@code multiplierOne * multiplierAnother}
     */
    public static BigDecimal mul(BigDecimal multiplierOne, BigDecimal multiplierAnother) {
        return mul(multiplierOne, multiplierAnother, SCALE_NUM, DOWN);
    }

    /**
     * 自定义乘法
     *
     * @param multiplierOne     乘数
     * @param multiplierAnother 另一个乘数
     * @param scaleNum          有效位数
     * @param mode              约分格式
     * @return {@code multiplierOne * multiplierAnother}
     */
    public static BigDecimal mul(BigDecimal multiplierOne, BigDecimal multiplierAnother, int scaleNum, RoundingMode mode) {
        return commonParse(multiplierOne.multiply(multiplierAnother), scaleNum, mode);
    }


    /**
     * 奥枫 - 通用除法 -除logic外不建议使用
     *
     * @param dividend 被除数
     * @param divisor  除数
     * @return {@code dividend / divisor}
     */
    public static BigDecimal div(Object dividend, Object divisor) {
        return div(dividend, divisor, SCALE_NUM);
    }

    /**
     * 奥枫 - 通用除法 -除logic外不建议使用
     *
     * @param dividend 被除数
     * @param divisor  除数
     * @param scaleNum 有效位数
     * @return {@code dividend / divisor}
     */
    public static BigDecimal div(Object dividend, Object divisor, Integer scaleNum) {
        return div(getBigDecimal(dividend), getBigDecimal(divisor), scaleNum, DOWN);
    }

    /**
     * 奥枫 - 通用除法
     *
     * @param dividend 被除数
     * @param divisor  除数
     * @return {@code dividend / divisor}
     */
    public static BigDecimal div(BigDecimal dividend, BigDecimal divisor) {
        return div(dividend, divisor, SCALE_NUM, DOWN);
    }

    /**
     * 自定义除法
     *
     * @param dividend 被除数
     * @param divisor  除数
     * @param scaleNum 有效位数
     * @param mode     约分格式
     * @return {@code dividend / divisor}
     */
    public static BigDecimal div(BigDecimal dividend, BigDecimal divisor, int scaleNum, RoundingMode mode) {
        return dividend.divide(divisor, scaleNum, mode);
    }

}
