package com.af.v4.system.common.plugins.http.core;

import com.af.v4.system.common.plugins.http.config.HttpClientConfig;
import org.apache.hc.client5.http.ConnectionKeepAliveStrategy;
import org.apache.hc.client5.http.config.RequestConfig;
import org.apache.hc.client5.http.impl.DefaultHttpRequestRetryStrategy;
import org.apache.hc.client5.http.impl.classic.CloseableHttpClient;
import org.apache.hc.client5.http.impl.classic.CloseableHttpResponse;
import org.apache.hc.client5.http.impl.classic.HttpClients;
import org.apache.hc.client5.http.impl.io.PoolingHttpClientConnectionManager;
import org.apache.hc.client5.http.protocol.HttpClientContext;
import org.apache.hc.client5.http.socket.ConnectionSocketFactory;
import org.apache.hc.client5.http.socket.LayeredConnectionSocketFactory;
import org.apache.hc.client5.http.socket.PlainConnectionSocketFactory;
import org.apache.hc.client5.http.ssl.NoopHostnameVerifier;
import org.apache.hc.client5.http.ssl.SSLConnectionSocketFactory;
import org.apache.hc.client5.http.ssl.TrustSelfSignedStrategy;
import org.apache.hc.core5.http.ClassicHttpRequest;
import org.apache.hc.core5.http.HeaderElement;
import org.apache.hc.core5.http.HttpHeaders;
import org.apache.hc.core5.http.config.Registry;
import org.apache.hc.core5.http.config.RegistryBuilder;
import org.apache.hc.core5.http.message.BasicHeaderElementIterator;
import org.apache.hc.core5.ssl.SSLContexts;
import org.apache.hc.core5.util.TimeValue;
import org.apache.hc.core5.util.Timeout;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.security.KeyManagementException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;

/**
 * HttpClient连接池实现
 *
 * @author Mr.river
 * @see com.af.v4.system.common.plugins.http.RestTools
 */
public class HttpConnectionPoolUtil extends BaseHttpPoolUtil {

    private static final Logger LOGGER = LoggerFactory.getLogger(HttpConnectionPoolUtil.class);
    /**
     * 相当于线程锁,用于线程安全
     */
    private final static Object SYNC_LOCK = new Object();
    /**
     * 发送HTTP请求的客户端单例
     */
    private static volatile CloseableHttpClient httpClient;

    /**
     * 获取httpclient的https实例（需要SSL证书）
     *
     * @param sslSocketFactory SSL构建器
     * @return httpclient实例
     */
    public static CloseableHttpClient getHttpClient(LayeredConnectionSocketFactory sslSocketFactory) {
        return createHttpClient(sslSocketFactory);
    }

    /**
     * 构建httpclient实例
     *
     * @return httpclient实例
     */
    private static CloseableHttpClient createHttpClient() {
        return createHttpClient(null);
    }

    /**
     * 构建httpclient实例
     *
     * @param sslSocketFactory SSL构建器
     * @return httpclient实例
     */
    private static CloseableHttpClient createHttpClient(LayeredConnectionSocketFactory sslSocketFactory) {
        //设置https相关信息
        if (sslSocketFactory == null) {
            try {
                sslSocketFactory = new SSLConnectionSocketFactory(
                        SSLContexts.custom().loadTrustMaterial(null, new TrustSelfSignedStrategy()).build(),
                        NoopHostnameVerifier.INSTANCE
                );
            } catch (NoSuchAlgorithmException | KeyManagementException | KeyStoreException e) {
                throw new RuntimeException(e);
            }
        }
        Registry<ConnectionSocketFactory> registry = RegistryBuilder.<ConnectionSocketFactory>create()
                .register("http", PlainConnectionSocketFactory.INSTANCE)
                .register("https", sslSocketFactory)
                .build();

        /* 设置连接参数 **/
        PoolingHttpClientConnectionManager manager = new PoolingHttpClientConnectionManager(registry);
        // 最大连接数
        manager.setMaxTotal(HttpClientConfig.HTTP_MAX_POOL_SIZE);
        // 路由最大连接数
        manager.setDefaultMaxPerRoute(HttpClientConfig.HTTP_MAX_POOL_SIZE);

        //设置keepalive
        ConnectionKeepAliveStrategy myStrategy = (response, context) -> {
            // Honor 'keep-alive' header
            BasicHeaderElementIterator it = new BasicHeaderElementIterator(
                    response.headerIterator(HttpHeaders.KEEP_ALIVE));
            while (it.hasNext()) {
                HeaderElement he = it.next();
                String param = he.getName();
                String value = he.getValue();
                if (value != null && "timeout".equalsIgnoreCase(param)) {
                    try {
                        return TimeValue.ofSeconds(Long.parseLong(value) * 1000);
                    } catch (NumberFormatException ignore) {
                    }
                }
            }
            // Keep alive for 5 seconds only
            return TimeValue.ofSeconds(5 * 1000);
        };

        HttpClientConfig config = new HttpClientConfig.Builder().build();

        RequestConfig requestConfig = RequestConfig.custom()
                .setConnectionRequestTimeout(Timeout.ofSeconds(config.getConnectTimeout()))
                .setConnectTimeout(Timeout.ofSeconds(config.getConnectTimeout()))
                .setResponseTimeout(Timeout.ofSeconds(config.getSocketTimeout())).build();

        return HttpClients.custom()
                .setKeepAliveStrategy(myStrategy)
                .setConnectionManagerShared(true)
                .setConnectionManager(manager)
                .setRetryStrategy(new DefaultHttpRequestRetryStrategy(0, TimeValue.ofSeconds(1L)))
                .evictExpiredConnections()
                .evictIdleConnections(TimeValue.ofMilliseconds(HttpClientConfig.HTTP_IDEL_TIME_OUT))
                .setDefaultRequestConfig(requestConfig)
                .build();
    }

    /**
     * 获取HTTP请求 httpclient实例
     *
     * @return httpclient实例
     */
    public static CloseableHttpClient getHttpClient() {
        if (httpClient == null) {
            //多线程下同时调用getHttpClient容易导致重复创建httpClient对象的问题,所以加上了同步锁
            synchronized (SYNC_LOCK) {
                if (httpClient == null) {
                    httpClient = createHttpClient();
                }
            }
        }
        return httpClient;
    }

    /**
     * 发送通用HTTP请求
     *
     * @param value      请求参数
     * @param headersStr 请求头
     * @param base       请求类型
     * @return 请求结果
     */
    public static String request(String value, String headersStr, HttpClientConfig config, ClassicHttpRequest base) {
        return request(value, headersStr, config, base, null);
    }

    public static String requestFormUrlEncoded(String value, String headersStr, HttpClientConfig config, ClassicHttpRequest base) {
        return requestFormUrlEncoded(value, headersStr, config, base, null);
    }

    /**
     * 发送通用HTTP请求
     *
     * @param value      请求参数
     * @param headersStr 请求头
     * @param base       请求类型
     * @param httpClient client对象
     * @return 请求结果
     */
    public static String request(String value,
                                 String headersStr,
                                 HttpClientConfig config,
                                 ClassicHttpRequest base,
                                 CloseableHttpClient httpClient) {
        LOGGER.info("请求地址：" + base.getScheme() + "://" + base.getAuthority() + base.getPath());
        if (httpClient == null) {
            httpClient = getHttpClient();
        }
        //配置请求参数
        setRequestConfig(config, base);
        //设置请求头
        setHeaders(base, headersStr);
        //设置请求体
        if (base.getEntity() == null) {
            setBody(base, value);
        }
        String result;
        long begin = System.currentTimeMillis();
        try (CloseableHttpResponse response = httpClient.execute(base, HttpClientContext.create())) {
            result = getResponseData(begin, response);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        return result;
    }

    public static String requestFormUrlEncoded(String value,
                                               String headersStr,
                                               HttpClientConfig config,
                                               ClassicHttpRequest base,
                                               CloseableHttpClient httpClient) {
        setUrlEncodedBody(base, value);
        return request(value, headersStr, config, base, httpClient);
    }
}
