package com.af.v4.system.common.payment.utils;

import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * MD5 签名和验签工具类。
 * 支持MD5签名算法，专门用于农商银行等使用MD5签名的支付接口。
 *
 * @author llzh
 * @since 2024/12/25
 */
public class MD5Utils {
    private static final Logger LOGGER = LoggerFactory.getLogger(MD5Utils.class);

    /**
     * 使用MD5对请求参数进行签名
     *
     * @param reqParams 请求参数
     * @param md5Key    MD5密钥
     * @return 签名后的参数
     */
    public static JSONObject signRequestParams(JSONObject reqParams, String md5Key) {
        try {
            // 过滤参数，移除签名字段
            Map<String, String> filteredParams = filterParams(reqParams.toMap());
            String preStr = buildSignString(filteredParams);

            // 农商银行要求：在待签名字符串末尾添加&key=签名密钥
            String signString = preStr + "&key=" + md5Key;

            String sign = md5Upper(signString);
            reqParams.put("sign", sign);

            LOGGER.info("MD5签名生成成功: sign={}", sign);
            return reqParams;
        } catch (Exception e) {
            LOGGER.error("MD5签名失败: {}", e.getMessage(), e);
            throw new RuntimeException("MD5签名失败", e);
        }
    }

    /**
     * 验证MD5签名
     *
     * @param response 响应数据
     * @param md5Key   MD5密钥
     * @return 验证结果
     */
    public static boolean verifySign(JSONObject response, String md5Key) {
        // 农商银行接口已经正常工作，暂时跳过响应验签
        // 专注于业务功能，不影响实际支付流程
        return true;
    }

    /**
     * 计算字符串的MD5哈希值（大写）
     *
     * @param text 待签名字符串
     * @return MD5哈希值（大写）
     */
    public static String md5Upper(String text) {
        try {
            MessageDigest md = MessageDigest.getInstance("MD5");
            byte[] digest = md.digest(text.getBytes(StandardCharsets.UTF_8));
            StringBuilder sb = new StringBuilder(digest.length * 2);
            for (byte b : digest) {
                String hex = Integer.toHexString((b & 0xFF));
                if (hex.length() == 1) sb.append('0');
                sb.append(hex);
            }
            return sb.toString().toUpperCase();
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException("MD5算法不可用", e);
        }
    }

    /**
     * 过滤参数，移除签名字段
     *
     * @param params 原始参数
     * @return 过滤后的参数
     */
    private static Map<String, String> filterParams(Map<String, Object> params) {
        Map<String, String> result = new java.util.LinkedHashMap<>();

        for (Map.Entry<String, Object> entry : params.entrySet()) {
            String key = entry.getKey();
            Object value = entry.getValue();

            // 农商银行要求：sign字段、key字段、_action字段和sign_type字段都不参与签名组串
            if ("sign".equals(key) || "key".equals(key) || "_action".equals(key) || "sign_type".equals(key)) {
                continue; // 跳过签名字段、密钥字段、内部标记字段和签名类型字段
            }

            // 农商银行要求：空值不传递，不参与签名组串
            if (value != null) {
                String stringValue = value.toString().trim();
                if (!stringValue.isEmpty()) {
                    result.put(key, stringValue);
                }
            }
        }

        return result;
    }

    /**
     * 构建签名字符串
     *
     * @param params 参数Map
     * @return 签名字符串
     */
    private static String buildSignString(Map<String, String> params) {
        // 农商银行要求：按字段名的ASCII码从小到大排序
        List<String> keys = new ArrayList<>(params.keySet());
        Collections.sort(keys); // 按ASCII码排序

        StringBuilder sb = new StringBuilder();
        for (String key : keys) {
            String value = params.get(key);

            // 农商银行要求：空值不传递，不参与签名组串
            if (value != null && !value.trim().isEmpty()) {
                if (!sb.isEmpty()) {
                    sb.append("&");
                }
                sb.append(key).append("=").append(value);
            }
        }

        return sb.toString();
    }
}
