package com.af.v4.system.common.payment.handler.impl.sxxh;

import com.af.v4.system.common.payment.annotation.PaymentHandlerConfig;
import com.af.v4.system.common.payment.dto.*;
import com.af.v4.system.common.payment.enums.BankName;
import com.af.v4.system.common.payment.enums.IntegrationType;
import com.af.v4.system.common.payment.enums.PaymentStatus;
import com.af.v4.system.common.payment.exceptions.PaymentException;
import com.af.v4.system.common.payment.handler.impl.AbstractPaymentHandler;
import com.af.v4.system.common.plugins.core.ConvertTools;
import com.af.v4.system.common.plugins.date.DateTools;
import com.af.v4.system.common.plugins.http.RestTools;
import com.af.v4.system.common.plugins.json.JsonTools;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.io.FileInputStream;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.nio.charset.StandardCharsets;
import java.security.KeyStore;
import java.security.PrivateKey;
import java.security.Signature;
import java.util.Enumeration;

/**
 * 陕西信合支付处理器
 */
@Component
@PaymentHandlerConfig(bankName = BankName.SXXH, integrationType = IntegrationType.AGGREGATE)
public class ShanXiXinHeAbstractPaymentHandler extends AbstractPaymentHandler {

    private static final Logger LOGGER = LoggerFactory.getLogger(ShanXiXinHeAbstractPaymentHandler.class);
    private static final String SUCCESS_CODE = "000000";

    /**
     * 构建支付订单请求参数
     *
     * @param request       支付订单请求对象，包含业务相关的请求信息
     * @param paymentConfig 支付配置信息，包含支付渠道相关配置参数
     * @return 返回构建好的请求参数JSON对象
     */
    @Override
    protected JSONObject buildPayOrderRequestParams(PaymentOrderRequest request, JSONObject paymentConfig) {
        LOGGER.info(">>> 构建陕西信合支付订单请求参数 > ");
        try {
            JSONObject reqParams = new JSONObject();

            // 基础参数
            reqParams.put("versionNo", "1.0.0");
            reqParams.put("orderTime", DateTools.getNow("yyyy-MM-dd HH:mm:ss.SSS"));
            reqParams.put("merchDate", DateTools.getNow("yyyyMMdd"));
            reqParams.put("frontURL", paymentConfig.optString("backUrl"));
            reqParams.put("ccy", "CNY"); // 币种
            reqParams.put("qrpCodeFlag", "02"); // 01:不生成二维码，02:生成二维码

            // 商户信息
            reqParams.put("merchId", paymentConfig.optString("merchId"));
            reqParams.put("merchOrderId", request.getOrderNo());
            reqParams.put("orderName", paymentConfig.optString("body", request.getDescription()));

            // 金额转换（分->元）
            String amountYuan = BigDecimal.valueOf(request.getAmount())
                    .divide(BigDecimal.valueOf(100), 2, RoundingMode.DOWN)
                    .toPlainString();
            reqParams.put("txnAmt", amountYuan);

            return reqParams;
        } catch (Exception e) {
            LOGGER.error("构建陕西信合支付订单请求参数异常", e);
            throw new PaymentException(PaymentStatus.PAY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getCode(),
                    request.toString(), PaymentStatus.PAY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getMessage());
        }
    }

    /**
     * 处理支付订单响应结果
     *
     * @param response      支付响应
     * @param request       支付请求
     * @param resultJson    响应数据
     * @param paymentConfig 配置
     * @return 结果
     */
    @Override
    protected PaymentOrderResponse processPayOrderResponse(PaymentOrderResponse response, PaymentOrderRequest request,
                                                           JSONObject resultJson, JSONObject paymentConfig) {
        LOGGER.info(">>> 处理陕西信合支付订单响应结果 > ");
        try {
            // 检查是否包含URL字段（表示下单成功）
            if (JsonTools.isContains(resultJson, "url")) {
                response.setState(PaymentStatus.PAY_SUCCESS);
                response.setOrderNo(request.getOrderNo());
                response.setPayUrl(resultJson.optString("url"));
                response.setNeedQuery(true); // 需要主动查询支付结果
                response.setPayResult("下单成功，等待支付");
                response.setQrCodeType("02"); // 生成二维码
                response.setAllowRefunds("1"); // 允许退款
                response.setAllowCancel("0"); // 不允许撤销
            } else {
                LOGGER.info("陕西信合支付订单下单失败：{}", resultJson);
                response.setState(PaymentStatus.PAY_FAIL);
                response.setCode(PaymentStatus.PAY_FAIL.getCode());
                response.setMessage(resultJson.optString("msg", "下单失败"));
            }
        } catch (Exception e) {
            LOGGER.error("处理陕西信合支付订单返回参数异常", e);
            throw new PaymentException(PaymentStatus.PAY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getCode(),
                    request.toString(), PaymentStatus.PAY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getMessage());
        }
        return response;
    }

    /**
     * 构建查询支付状态请求参数
     *
     * @param request       查询支付状态请求对象
     * @param paymentConfig 支付配置信息
     * @return 返回构建好的查询支付状态请求参数JSON对象
     */
    @Override
    protected JSONObject buildQueryPaymentStatusRequestParams(QueryPaymentStatusRequest request, JSONObject paymentConfig) {
        LOGGER.info(">>> 构建陕西信合查询支付状态请求参数 > ");
        try {
            JSONObject reqParams = new JSONObject();

            reqParams.put("versionNo", "1.0.0");
            reqParams.put("merchDate", DateTools.getNow("yyyyMMdd"));
            reqParams.put("merchId", paymentConfig.optString("merchId"));
            reqParams.put("merchOrderId", request.getOrderNo());
            reqParams.put("orderType", "01"); // 交易类型，01-支付02-退款

            return reqParams;
        } catch (Exception e) {
            LOGGER.error("构建陕西信合查询订单支付状态请求参数异常", e);
            throw new PaymentException(PaymentStatus.QUERY_ORDER_BUILD_REQUEST_PARAM_ERROR.getCode(),
                    request.toString(), PaymentStatus.QUERY_ORDER_BUILD_REQUEST_PARAM_ERROR.getMessage());
        }
    }

    /**
     * 处理查询支付状态响应结果
     *
     * @param response   查询支付状态响应对象
     * @param request    查询支付状态请求对象
     * @param resultJson 支付状态查询接口返回的原始响应JSON数据
     * @return 返回处理后的查询支付状态响应对象
     */
    @Override
    protected QueryPaymentStatusResponse processQueryPaymentStatusResponse(QueryPaymentStatusResponse response,
                                                                           QueryPaymentStatusRequest request, JSONObject resultJson) {
        LOGGER.info("处理陕西信合查询支付状态响应结果 > ");
        try {
            if (SUCCESS_CODE.equals(resultJson.optString("retCode"))) {
                String txnStat = resultJson.optString("txnStat");
                response.setOrderNo(request.getOrderNo());
                response.setMchId(resultJson.optString("merchId"));

                PaymentStatus status = switch (txnStat) {
                    case "01" -> { // 交易处理中
                        response.setPaymentStatus("交易处理中");
                        yield PaymentStatus.PAYMENT_IN_PROGRESS;
                    }
                    case "02" -> { // 交易成功
                        response.setPaymentStatus("支付成功");
                        // 设置第三方订单号
                        response.setTransactionId(resultJson.optString("txnSeq"));
                        String payTime = resultJson.optString("txnTime");
                        if (StringUtils.hasText(payTime)) {
                            response.setPaySuccessDate(DateTools.formatDateTime(payTime));
                        }
                        yield PaymentStatus.SUCCESS_FOR_PAYMENT;
                    }
                    case "03" -> { // 交易失败
                        response.setPaymentStatus("支付失败");
                        yield PaymentStatus.FAIL_FOR_PAYMENT;
                    }
                    case "04" -> { // 订单不存在
                        response.setPaymentStatus("订单不存在");
                        yield PaymentStatus.CANCEL_FOR_PAYMENT;
                    }
                    default -> {
                        response.setPaymentStatus("未知状态");
                        yield PaymentStatus.QUERY_ORDER_FAIL;
                    }
                };

                response.setState(status);
            } else {
                LOGGER.info("陕西信合查询支付状态失败：{}", resultJson);
                response.setState(PaymentStatus.QUERY_ORDER_FAIL);
                response.setCode(PaymentStatus.QUERY_ORDER_FAIL.getCode());
                response.setMessage(resultJson.optString("retMsg", "查询失败"));
            }
        } catch (Exception e) {
            LOGGER.error("处理陕西信合查询支付状态返回参数异常", e);
            throw new PaymentException(PaymentStatus.QUERY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getCode(),
                    request.toString(), PaymentStatus.QUERY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getMessage());
        }
        return response;
    }

    /**
     * 构建取消订单请求参数
     *
     * @param request       取消支付请求对象
     * @param paymentConfig 支付配置信息
     * @return 返回构建好的取消订单请求参数JSON对象
     */
    @Override
    protected JSONObject buildCancelOrderRequestParams(CancelPaymentRequest request, JSONObject paymentConfig) {
        // 陕西信合不支持取消订单操作
        LOGGER.warn("陕西信合不支持取消订单操作");
        return null;
    }

    /**
     * 处理取消订单响应结果
     *
     * @param response   取消支付响应对象
     * @param request    取消支付请求对象
     * @param resultJson 取消订单接口返回的原始响应JSON数据
     * @return 返回处理后的取消支付响应对象
     */
    @Override
    protected CancelPaymentResponse processCancelOrderResponse(CancelPaymentResponse response, CancelPaymentRequest request,
                                                               JSONObject resultJson) {
        // 陕西信合不支持取消订单操作
        response.setCode(PaymentStatus.REFUND_NOT_SUPPORTED.getCode());
        response.setMessage("陕西信合不支持取消订单操作");
        return response;
    }

    /**
     * 构建退款订单请求参数
     *
     * @param request       退款支付请求对象
     * @param paymentConfig 支付配置信息
     * @return 返回构建好的退款订单请求参数JSON对象
     */
    @Override
    protected JSONObject buildRefundOrderRequestParams(RefundPaymentRequest request, JSONObject paymentConfig) {
        LOGGER.info("构建陕西信合退款订单请求参数 > ");
        try {
            JSONObject reqParams = new JSONObject();

            reqParams.put("versionNo", "1.0.0");
            reqParams.put("merchTime", DateTools.getNow("yyyyMMddHHmmss"));
            reqParams.put("merchDate", DateTools.getNow("yyyyMMdd"));
            reqParams.put("ccy", "CNY"); // 币种
            reqParams.put("merchId", paymentConfig.optString("merchId"));
            reqParams.put("backURL", paymentConfig.optString("refundBackUrl"));
            reqParams.put("primMerchOrderId", request.getOrderNo()); // 原商户支付订单号
            reqParams.put("primMerchOrderDate", DateTools.getNow("yyyyMMdd")); // 原支付订单日期

            // 退款金额转换（分->元）
            String refundAmountYuan = BigDecimal.valueOf(request.getRefundAmount())
                    .divide(BigDecimal.valueOf(100), 2, RoundingMode.DOWN)
                    .toPlainString();
            reqParams.put("txnAmt", refundAmountYuan); // 支付金额
            reqParams.put("refundAmt", refundAmountYuan); // 退款金额
            reqParams.put("refundMerchOrderId", request.getRefundOrderNo()); // 退款订单号

            return reqParams;
        } catch (Exception e) {
            LOGGER.error("构建陕西信合退款订单请求参数异常", e);
            throw new PaymentException(PaymentStatus.REFUND_BUILD_REQUEST_PARAM_ERROR.getCode(),
                    request.toString(), PaymentStatus.REFUND_BUILD_REQUEST_PARAM_ERROR.getMessage());
        }
    }

    /**
     * 处理退款订单响应结果
     *
     * @param response   退款支付响应对象
     * @param request    退款支付请求对象
     * @param resultJson 退款接口返回的原始响应JSON数据
     * @return 返回处理后的退款支付响应对象
     */
    @Override
    protected RefundPaymentResponse processRefundOrderResponse(RefundPaymentResponse response, RefundPaymentRequest request,
                                                               JSONObject resultJson) {
        LOGGER.info("处理陕西信合订单退款响应结果 > ");
        try {
            if (SUCCESS_CODE.equals(resultJson.optString("retCode"))) {
                response.setRefundId(request.getRefundOrderNo());
                response.setOrderNo(request.getOrderNo());
                response.setRefundAmount(request.getRefundAmount());
                response.setRefundResult("退款申请成功");
                response.setState(PaymentStatus.REFUND_SUCCESS);
            } else {
                LOGGER.info("陕西信合订单退款失败：{}", resultJson);
                response.setState(PaymentStatus.REFUND_FAIL);
                response.setCode(PaymentStatus.REFUND_FAIL.getCode());
                response.setMessage(resultJson.optString("retMsg", "退款失败"));
            }
        } catch (Exception e) {
            LOGGER.error("处理陕西信合订单退款返回参数异常", e);
            throw new PaymentException(PaymentStatus.REFUND_BUILD_RESPONSE_PARAM_ERROR.getCode(),
                    request.toString(), PaymentStatus.REFUND_BUILD_RESPONSE_PARAM_ERROR.getMessage());
        }
        return response;
    }

    /**
     * 构建查询退款状态请求参数
     *
     * @param request       查询退款状态请求对象
     * @param paymentConfig 支付配置信息
     * @return 返回构建好的查询退款状态请求参数JSON对象
     */
    @Override
    protected JSONObject buildQueryRefundStatusRequestParams(QueryRefundStatusRequest request, JSONObject paymentConfig) {
        LOGGER.info("构建陕西信合查询退款状态请求参数 > ");
        try {
            JSONObject reqParams = new JSONObject();

            reqParams.put("versionNo", "1.0.0");
            reqParams.put("merchDate", DateTools.getNow("yyyyMMdd"));
            reqParams.put("merchId", paymentConfig.optString("merchId"));
            reqParams.put("merchOrderId", request.getRefundId()); // 退款订单号
            reqParams.put("orderType", "02"); // 交易类型，01-支付02-退款

            return reqParams;
        } catch (Exception e) {
            LOGGER.error("构建陕西信合查询退款请求参数异常", e);
            throw new PaymentException(PaymentStatus.QUERY_REFUND_BUILD_REQUEST_PARAM_ERROR.getCode(),
                    request.toString(), PaymentStatus.QUERY_REFUND_BUILD_REQUEST_PARAM_ERROR.getMessage());
        }
    }

    /**
     * 处理查询退款状态响应结果
     *
     * @param response   查询退款状态响应对象
     * @param request    查询退款状态请求对象
     * @param resultJson 退款状态查询接口返回的原始响应JSON数据
     * @return 返回处理后的查询退款状态响应对象
     */
    @Override
    protected QueryRefundStatusResponse processQueryRefundStatusResponse(QueryRefundStatusResponse response,
                                                                         QueryRefundStatusRequest request, JSONObject resultJson) {
        LOGGER.info("处理陕西信合查询退款状态响应结果 > ");
        try {
            if (SUCCESS_CODE.equals(resultJson.optString("retCode"))) {
                String txnStat = resultJson.optString("txnStat");

                response.setRefundId(request.getRefundId());
                response.setOrderNo(resultJson.optString("primMerchOrderId"));
                response.setTransactionId(resultJson.optString("txnSeq"));

                PaymentStatus status = switch (txnStat) {
                    case "01" -> { // 退款处理中
                        response.setRefundResult("退款处理中");
                        yield PaymentStatus.REFUND_PROCESSING;
                    }
                    case "02" -> { // 退款成功
                        response.setRefundResult("退款成功");
                        String refundTime = resultJson.optString("txnTime");
                        if (StringUtils.hasText(refundTime)) {
                            response.setRefundDateTime(DateTools.formatDateTime(refundTime));
                        }
                        yield PaymentStatus.REFUND_SUCCESS;
                    }
                    case "03" -> { // 退款失败
                        response.setRefundResult("退款失败");
                        yield PaymentStatus.REFUND_FAIL;
                    }
                    default -> {
                        response.setRefundResult("未知状态");
                        yield PaymentStatus.QUERY_REFUND_FAIL;
                    }
                };

                response.setState(status);
            } else {
                LOGGER.info("陕西信合查询退款状态失败：{}", resultJson);
                response.setState(PaymentStatus.QUERY_REFUND_FAIL);
                response.setCode(PaymentStatus.QUERY_REFUND_FAIL.getCode());
                response.setMessage(resultJson.optString("retMsg", "查询退款状态失败"));
            }
        } catch (Exception e) {
            LOGGER.error("处理陕西信合查询退款状态返回参数异常", e);
            throw new PaymentException(PaymentStatus.QUERY_REFUND_BUILD_RESPONSE_PARAM_ERROR.getCode(),
                    request.toString(), PaymentStatus.QUERY_REFUND_BUILD_RESPONSE_PARAM_ERROR.getMessage());
        }
        return response;
    }

    /**
     * 对请求参数进行签名处理
     *
     * @param reqParams     请求参数JSON对象
     * @param paymentConfig 支付配置信息，包含签名所需密钥等参数
     */
    @Override
    protected void signRequest(JSONObject reqParams, JSONObject paymentConfig) {
        try {
            String pfxUrl = paymentConfig.optString("pfxUrl");
            String pfxPwd = paymentConfig.optString("pfxPwd");

            if (ObjectUtils.isEmpty(pfxUrl) || ObjectUtils.isEmpty(pfxPwd)) {
                throw new PaymentException(
                        PaymentStatus.PAY_ORDER_ERROR.getCode(),
                        reqParams.toString(),
                        "陕西信合签名证书未配置"
                );
            }

            // 转换参数为字符串格式（模拟BankUtils.appendParam方法）
            String transData = appendParam(reqParams);

            // 调用SxxhUtil.sign进行签名（这里需要实现或调用现有的签名方法）
            String signStr = sxxhSign(transData, pfxUrl, pfxPwd);

            // 清空原参数，设置最终请求参数
            reqParams.clear();
            reqParams.put("transData", transData);
            reqParams.put("signMsg", signStr);

            LOGGER.info("陕西信合签名后的请求参数: {}", reqParams);
        } catch (Exception e) {
            LOGGER.error("陕西信合请求签名失败", e);
            throw new PaymentException(
                    PaymentStatus.PAY_ORDER_ERROR.getCode(),
                    reqParams.toString(),
                    "陕西信合请求签名失败: " + e.getMessage()
            );
        }
    }

    /**
     * 执行支付请求
     *
     * @param reqParams     请求参数JSON对象
     * @param paymentConfig 支付配置信息
     * @return 返回支付接口响应的JSON数据
     */
    @Override
    protected JSONObject executePaymentRequest(JSONObject reqParams, JSONObject paymentConfig) {
        try {
            // 根据不同操作选择不同的URL
            String url;
            if (reqParams.has("orderType")) {
                String orderType = reqParams.getJSONObject("transData") != null ?
                        new JSONObject(reqParams.getString("transData")).optString("orderType") : "01";
                if ("02".equals(orderType)) {
                    // 退款查询
                    url = paymentConfig.optString("queryUrl");
                } else {
                    // 支付查询
                    url = paymentConfig.optString("queryUrl");
                }
            } else if (reqParams.toString().contains("refundAmt")) {
                // 退款
                url = paymentConfig.optString("refundUrl");
            } else {
                // 支付下单
                url = paymentConfig.optString("payUrl");
            }

            JSONObject headers = new JSONObject();
            headers.put("Content-Type", "application/x-www-form-urlencoded");

            // 陕西信合接口要求POST form格式
            String requestBody = "transData=" + reqParams.getString("transData") + "&signMsg=" + reqParams.getString("signMsg");
            LOGGER.info(">>> 陕西信合请求URL={}, 请求参数：{}", url, requestBody);

            String response = RestTools.post(url, requestBody, headers.toString());

            // 验证响应签名
            if (!sxxhVerifyData(response)) {
                throw new PaymentException(
                        PaymentStatus.VERIFY_SIGN_FAIL.getCode(),
                        response,
                        "陕西信合响应验签失败"
                );
            }

            JSONObject jsonObject = JsonTools.convertToJson(response);
            LOGGER.info(">>> 陕西信合响应结果：{}", jsonObject);
            return jsonObject;
        } catch (Exception e) {
            LOGGER.error("请求陕西信合业务接口异常", e);
            throw new PaymentException(
                    PaymentStatus.PAY_ORDER_ERROR.getCode(),
                    reqParams.toString(),
                    PaymentStatus.PAY_ORDER_ERROR.getMessage()
            );
        }
    }

    /**
     * 验证响应数据签名
     *
     * @param response      支付接口响应JSON数据
     * @param paymentConfig 支付配置信息，包含验签所需密钥等参数
     * @return 验证通过返回true，否则返回false
     */
    @Override
    protected boolean verifyResponseSign(JSONObject response, JSONObject paymentConfig) {
        // 在executePaymentRequest中已经完成验签，这里直接返回true
        return true;
    }

    @Override
    protected JSONObject buildDownloadReconciliationFileRequestParams(DownloadReconciliationFileRequest request, JSONObject paymentConfig) {
        return null;
    }

    @Override
    protected DownloadReconciliationFileResponse processDownloadReconciliationFileResponse(DownloadReconciliationFileResponse response, DownloadReconciliationFileRequest request, JSONObject resultJson) {
        return null;
    }

    /**
     * 将JSON参数转换为字符串格式（模拟BankUtils.appendParam方法）
     *
     * @param params 参数JSON对象
     * @return 参数字符串
     */
    private String appendParam(JSONObject params) {
        StringBuilder sb = new StringBuilder();
        for (String key : params.keySet()) {
            if (!sb.isEmpty()) {
                sb.append("&");
            }
            sb.append(key).append("=").append(params.opt(key));
        }
        return sb.toString();
    }

    /**
     * 陕西信合签名方法
     *
     * @param transData 交易数据
     * @param pfxUrl    证书路径
     * @param pfxPwd    证书密码
     * @return 签名字符串
     */
    private String sxxhSign(String transData, String pfxUrl, String pfxPwd) {
        try {

            // 获取私钥（需要实现或调用CertUtil.getSignCertPrivateKeyByStoreMap）
            PrivateKey privateKey = getSignCertPrivateKeyByStoreMap(pfxUrl, pfxPwd);

            // 签名实现
            byte[] result = signBySoft(privateKey, transData.getBytes(StandardCharsets.UTF_8));
            String signStr = ConvertTools.base64Encode(result);

            LOGGER.debug("陕西信合签名成功，数据长度: {}", signStr.length());
            return signStr;
        } catch (Exception e) {
            LOGGER.error("陕西信合签名失败", e);
            throw new PaymentException(
                    PaymentStatus.SIGN_FAIL.getCode(),
                    transData,
                    "陕西信合签名失败: " + e.getMessage()
            );
        }
    }

    /**
     * 软签名实现
     *
     * @param privateKey 私钥
     * @param data       待签名数据
     * @return 签名结果
     * @throws Exception 签名异常
     */
    private byte[] signBySoft(PrivateKey privateKey, byte[] data) throws Exception {
        // 使用SHA256WithRSA算法进行签名
        Signature signature = Signature.getInstance("SHA256withRSA");
        signature.initSign(privateKey);
        signature.update(data);
        byte[] result = signature.sign();

        // 输出签名结果用于调试
        String s = ConvertTools.base64Encode(result);
        LOGGER.debug("签名结果: {}", s);

        return result;
    }

    /**
     * 从证书存储中获取签名证书私钥
     *
     * @param certPath 证书路径
     * @param password 证书密码
     * @return 私钥
     * @throws Exception 获取私钥异常
     */
    private PrivateKey getSignCertPrivateKeyByStoreMap(String certPath, String password) throws Exception {
        try {
            // 加载PKCS12证书
            KeyStore keyStore = KeyStore.getInstance("PKCS12");
            try (FileInputStream fis = new FileInputStream(certPath)) {
                keyStore.load(fis, password.toCharArray());
            }

            // 获取别名
            Enumeration<String> aliases = keyStore.aliases();
            String alias = null;
            while (aliases.hasMoreElements()) {
                String currentAlias = aliases.nextElement();
                if (keyStore.isKeyEntry(currentAlias)) {
                    alias = currentAlias;
                    break;
                }
            }

            if (alias == null) {
                throw new Exception("证书中未找到私钥条目");
            }

            // 获取私钥
            PrivateKey privateKey = (PrivateKey) keyStore.getKey(alias, password.toCharArray());
            if (privateKey == null) {
                throw new Exception("无法从证书中获取私钥");
            }

            LOGGER.debug("成功从证书获取私钥，算法: {}", privateKey.getAlgorithm());
            return privateKey;
        } catch (Exception e) {
            LOGGER.error("获取证书私钥失败，证书路径: {}", certPath, e);
            throw new Exception("获取证书私钥失败: " + e.getMessage(), e);
        }
    }

    /**
     * 陕西信合验签方法（基于SxxhUtil.verifyData实现）
     *
     * @param response 响应数据
     * @return 验签结果
     */
    private boolean sxxhVerifyData(String response) {
        // 陕西信合的验签逻辑：检查响应数据是否不为空且以"{"开头（JSON格式）
        boolean isValid = response != null && response.startsWith("{");

        LOGGER.debug("陕西信合验签结果: {}, 响应数据长度: {}", isValid,
                response != null ? response.length() : 0);

        if (!isValid) {
            LOGGER.warn("陕西信合验签失败，响应数据: {}", response);
        }

        return isValid;
    }
}
