package com.af.v4.system.common.payment.utils;

import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.charset.StandardCharsets;
import org.apache.commons.codec.binary.Base64;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * 用于生成签名、验证签名及参数处理等
 *
 * @author llzh
 * @DateTime 2024/12/26 15:10
 */
public class SignUtils {

    private static final Logger LOGGER = LoggerFactory.getLogger(SignUtils.class);

    private static final String SIGN_TYPE_RSA = "RSA_1_256";
    private static final String SIGN_KEY = "sign";


    public static void main(String[] args) throws Exception {
        JSONObject jsonObject = new JSONObject("{\n" +
                "    \"nonce_str\": \"1736936317382\",\n" +
                "    \"out_trade_no\": \"SN20250115181837377\",\n" +
                "    \"service\": \"unified.trade.micropay\",\n" +
                "    \"total_fee\": \"1\",\n" +
                "    \"terminal_info\": \"{\\\"terminal_type\\\":\\\"11\\\",\\\"app_version\\\":\\\"1.000000\\\",\\\"terminal_id\\\":\\\"44440101\\\"}\",\n" +
                "    \"mch_id\": \"105570132005\",\n" +
                "    \"body\": \"测试缴费\",\n" +
                "    \"sign_type\": \"RSA_1_256\",\n" +
                "    \"mch_create_ip\": \"61.150.11.227\",\n" +
                "    \"auth_code\": \"133967315945743602\"\n" +
                "}");
        System.out.println("看看请求参数--------->" + jsonObject);
        String privateKey = "MIIEvAIBADANBgkqhkiG9w0BAQEFAASCBKYwggSiAgEAAoIBAQCWrQkUux1oKOd/uzKmT6aR3j5dhvKQmSNGNFx6rtANdftnBhPAisXRhG5k+AJoT/Olnt8NSgK63G24NYQYNazBWd4f75m21pqFg8XNa3hsvu0oRj6TExIWuIqW2QAgo7Dd6sWNkGCVI0YEe0YjBBCYcLnY4dY56LSSA/65lWWkqjAt6UmYcEPW7a1l+BmdCCmsZf/2c1UxsyrBX7MoNSv9wpV9Z20wSHt+v9zvdHDKauUZUDmr68bzSUThUkpwEQEDmgnPMmfjapux9Lb5jfqutR7SDpesItvo+TnenmoKCDkC7iVkF/W2YkBVLsAoRZnUqBt7iVo8DsaJSs+SBYNNAgMBAAECggEAd6uYocl3MqoCK1WvhZ2PSx07ZSbv8l5eQL0HkjzTa02ATbPq7iqJsL1AvJqpttWBStJij3hvsJ+v2PVQI7ZFrg9FNxIigK/zbAVbjapErAH789fOjRtBa6BIGkThMAsyRx7Wpne77ddlHdwWuWraD2jAgnvjnHyxJaNcpWL9poh1+td0aGyV0fi+DZl/U7chY61jmlJjcbRWkA53TLK9k2RSA+1y9AOWJV+vtJYBKpj4TTkAU/FE4VCTKMmntiXdOtfAzOvwaHngkQ24KcKtyjnxQFtFujolUmNHoQwmrig5dwMOMIWOIcGThNyjHt2O++kgtRyzYiAn2GeNiYigIQKBgQDhCTZNE4jucJN2oa6fA783GspjjSq3yJmfrcUpwddmolmG2YdYfnHj+lsf29MUaVxCTRCzD7ZjfqHtpjSEfZ8ERz7Lp7XD2aD0OdbrJhyn1VnzxFBVo77YrC8ozWSbwidtXxumQPwgUWZOCILwpKdfJZ7eDmeTmSPmxoLylJsNOQKBgQCraIZQC+Pa7XqIGmcmrXUWnZhAHKC3ehGeX7Vkv4x88eoTYpGG7v+9+mSgBK9nWAgQzbANgN61x4nsHpJZHzBOmf4fUYJ8qSC8jxuaAbCVnCFOuCm7L8MM3qQjE4nYxln5JvwODIU7wu8fjNlBXoVyZ72UVKWY8RcofT6XFNh6tQKBgDOx2Q6FsUwNAmMbKzEOPEvdVKbf0/NaGqhK0+MOy3uhSe2nZOcYTPNxHKPJ7Mhgl7gHWhG4R0691VbxelkBPplOAIfqI+A0yzuvWca0/5oycbKE42t47St7Cm5WGImLZWFDmudrQD/wUw8T93huaqB51O/v4S1P3aZy7bvqFWBJAoGAWWatFzb8YwWV8UMEJhI2jdMow9eqo8fRRK9apu+ZUNnEYtP4X05sqSPeM+/9pIMvwcPmXUSn5FvsvfC8GCvCQFzlRGR5EJgN8QsbNFpJRd8nKC255l3k9+Nv7LOWJDBgYSlKMfGp1B+6JNSLxCKOq7scEC+OJ5UnpQCeEqLWvjkCgYB4+VNb1jl3Y8NoZSQjZGocrb30cNgEZK8k3Bz1653b/NxCfsg2sgjBOo22IG5D6iTdjrMseT/1QFfbLRZuSj9+kTJjZY/DEF/SMFDvBnfjj+xut/JW3GePKx1gWBGzNNGYSsOyMxXxv5tgrfqOJoVReu7vTVkIp9xn3s/4qnTA6A==";
        signRequestParams(jsonObject, "RSA_1_256", "sign", privateKey);
        System.out.println("看看签名后请求参数--------->" + jsonObject);
        String publicKey = "MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAlq0JFLsdaCjnf7sypk+mkd4+XYbykJkjRjRceq7QDXX7ZwYTwIrF0YRuZPgCaE/zpZ7fDUoCutxtuDWEGDWswVneH++ZttaahYPFzWt4bL7tKEY+kxMSFriKltkAIKOw3erFjZBglSNGBHtGIwQQmHC52OHWOei0kgP+uZVlpKowLelJmHBD1u2tZfgZnQgprGX/9nNVMbMqwV+zKDUr/cKVfWdtMEh7fr/c73RwymrlGVA5q+vG80lE4VJKcBEBA5oJzzJn42qbsfS2+Y36rrUe0g6XrCLb6Pk53p5qCgg5Au4lZBf1tmJAVS7AKEWZ1Kgbe4laPA7GiUrPkgWDTQIDAQAB";
        System.out.println("验签结果：" + verifySign(jsonObject, "RSA_1_256", jsonObject.getString("sign"), publicKey));
    }


    /**
     * 生成签名并添加到请求参数中
     *
     * @param reqParams  请求参数集合
     * @param signType   签名类型 可选  不传的默认为 RSA_1_256
     * @param signKey    签名字段 可选  不传的默认为 sign
     * @param privateKey 密钥
     * @throws Exception 如果签名生成失败
     */
    public static void signRequestParams(JSONObject reqParams, String signType, String signKey, String privateKey) throws Exception {
        try {
            Map<String, String> filteredParams = filterParams(reqParams.toMap());
            String preStr = buildSignString(filteredParams);
            LOGGER.info("待签名字符串: {}", preStr);
            // 使用传入的签名类型或默认签名类型
            signType = Optional.ofNullable(signType).orElse(SIGN_TYPE_RSA);
            String sign = generateSign(signType, preStr, privateKey);
            reqParams.put(signKey, sign);
            LOGGER.info("签名生成成功: signType={}, sign={}", signType, sign);
            LOGGER.info("加签后的请求参数: reqParams={}", reqParams);
        } catch (Exception e) {
            LOGGER.error("签名生成失败: {}", e.getMessage(), e);
            throw new Exception("签名生成失败", e);
        }
    }

    /**
     * 生成签名并添加到请求参数中，签名类型默认为RSA_1_256
     *
     * @param reqParams  请求参数集合
     * @param privateKey 密钥
     * @throws Exception 如果签名生成失败
     */
    public static void signRequestParams(JSONObject reqParams, String privateKey) throws Exception {
        signRequestParams(reqParams, "RSA_1_256", "sign", privateKey);
    }

    /**
     * 生成签名
     *
     * @param signType   签名类型（目前仅支持 RSA）
     * @param preStr     待签名字符串
     * @param privateKey 私钥字符串（Base64 编码）
     * @return 签名结果（Base64 编码）
     * @throws Exception 如果签名失败或签名类型不支持
     */
    private static String generateSign(String signType, String preStr, String privateKey) {
        if (SIGN_TYPE_RSA.equalsIgnoreCase(signType)) {
            try {
                // 使用 RSA 签名并返回 Base64 编码结果
                byte[] signature = RSAUtils.sign(RSAUtils.SignatureSuite.SHA256, preStr.getBytes(StandardCharsets.UTF_8), privateKey);
                return new String(Base64.encodeBase64(signature), StandardCharsets.UTF_8);
            } catch (Exception e) {
                throw new RuntimeException("签名生成失败: " + e.getMessage(), e);
            }
        } else {
            // 对于不支持的签名类型抛出明确异常
            throw new UnsupportedOperationException("不支持的签名方式: " + signType);
        }
    }
    public static boolean verifySign(JSONObject response, String signType, String sign, String publicKey) {
        if("RSA_1_256".equals(signType)) {
            Map<String, String> Reparams = filterParams(response.toMap());
            String repreStr = buildSignString(Reparams);
            LOGGER.info("待验签字符串: {}", repreStr);
            return RSAUtils.verifySign(RSAUtils.SignatureSuite.SHA256, repreStr, sign, publicKey);
        }
        return false;
    }


    /**
     * 过滤无效字段（移除 sign 字段和空值字段）
     *
     * @param params 参数 Map
     * @return 过滤后的 Map
     */
    public static Map<String, String> filterParams(Map<String, Object> params) {
        return params.entrySet().stream()
                .filter(entry -> !"sign".equals(entry.getKey()) && entry.getValue() != null && !entry.getValue().toString().isEmpty())
                .collect(Collectors.toMap(Map.Entry::getKey, entry -> entry.getValue().toString()));
    }


    /**
     * 按字段名排序并拼接成签名原始串
     *
     * @param params 过滤后的参数 Map
     * @return 签名原始串
     */
    public static String buildSignString(Map<String, String> params) {
        return params.entrySet().stream().sorted(Map.Entry.comparingByKey())
                .map(entry -> entry.getKey() + "=" + entry.getValue())
                .collect(Collectors.joining("&"));
    }




}
