package com.af.v4.system.common.payment.service;

import com.af.v4.system.common.payment.config.PaymentConfig;
import com.af.v4.system.common.payment.dto.*;
import com.af.v4.system.common.payment.enums.IntegrationType;
import com.af.v4.system.common.payment.enums.BankName;
import com.af.v4.system.common.payment.enums.PaymentStatus;
import com.af.v4.system.common.payment.exceptions.PaymentException;
import com.af.v4.system.common.payment.factory.PaymentHandlerFactory;
import com.af.v4.system.common.payment.handler.PaymentHandler;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.util.UUID;
import java.util.concurrent.*;

/**
 * @author llzh
 * @ClassName PaymentService
 * @Description 支付服务
 * @DateTime 2024/12/24 17:24
 */
@Service
public class PaymentService {

    private static final Logger LOGGER = LoggerFactory.getLogger(PaymentService.class);
    private static final int TIMEOUT_SECONDS = 5;
    private final ExecutorService executorService = Executors.newCachedThreadPool();
    private final JSONObject paymentConfig;
    private final IntegrationType integrationType;
    private final BankName bankName;
    private PaymentService(PaymentConfig paymentConfig) {
        this.paymentConfig = paymentConfig.paymentConfig;
        this.integrationType = paymentConfig.integrationType;
        this.bankName = paymentConfig.bankName;
    }


    /**
     * 处理支付请求
     *
     * 该方法通过提交支付请求到对应的支付处理器进行异步处理，并等待支付结果
     * 如果支付处理超过预设时间未完成，则自动撤销当前订单，抛出支付异常, 以异常形式返回给调用方,
     *
     * @param request 支付请求对象，包含支付所需的信息
     * @return 返回支付响应对象，包含支付结果
     * @throws PaymentException 当支付处理失败或超时
     */
    public PaymentOrderResponse paymentOrder(PaymentOrderRequest request) {
        try {
            PaymentHandler handler = getPaymentHandler(bankName, integrationType);
            Future<PaymentOrderResponse> future = (Future<PaymentOrderResponse>)
                    executorService.submit(() -> handler.paymentOrder(request,paymentConfig));
            return future.get(TIMEOUT_SECONDS, TimeUnit.SECONDS);
        } catch (TimeoutException e) {
            throw new PaymentException(PaymentStatus.CONNECT_TIME_OUT.getCode(),request.toString(), PaymentStatus.CONNECT_TIME_OUT.getMessage());
        } catch (Exception e) {
            throw new PaymentException(PaymentStatus.PAY_ORDER_ERROR.getCode(),request.toString() ,e.getMessage());
        }
    }

    public QueryPaymentStatusResponse queryPaymentStatus(QueryPaymentStatusRequest request) {
        try {
            PaymentHandler handler = getPaymentHandler(bankName, integrationType);
            Future<QueryPaymentStatusResponse> future = executorService.submit(() -> handler.queryPaymentStatus(request,paymentConfig));
            return future.get(TIMEOUT_SECONDS, TimeUnit.SECONDS);
        } catch (TimeoutException e) {
            throw new PaymentException(PaymentStatus.CONNECT_TIME_OUT.getCode(),request.toString(), PaymentStatus.CONNECT_TIME_OUT.getMessage());
        } catch (Exception e) {
            throw new PaymentException(PaymentStatus.PAY_ORDER_ERROR.getCode(), request.toString(), e.getMessage());
        }
    }

    public CancelPaymentResponse cancelPayment(CancelPaymentRequest request) {
        try {
            PaymentHandler handler = getPaymentHandler(bankName, integrationType);
            Future<CancelPaymentResponse> future = executorService.submit(() -> handler.cancelOrder(request,paymentConfig));
            return future.get(TIMEOUT_SECONDS, TimeUnit.SECONDS);
        } catch (TimeoutException e) {
            throw new PaymentException(PaymentStatus.CONNECT_TIME_OUT.getCode(),request.toString(), PaymentStatus.CONNECT_TIME_OUT.getMessage());
        } catch (Exception e) {
            throw new PaymentException(PaymentStatus.PAY_ORDER_ERROR.getCode(), request.toString(), e.getMessage());
        }
    }
    public RefundPaymentResponse refundPayment(RefundPaymentRequest request) {
        try {
            PaymentHandler handler = getPaymentHandler(bankName, integrationType);
            Future<RefundPaymentResponse> future = executorService.submit(() -> handler.refundPayment(request,paymentConfig));
            return future.get(TIMEOUT_SECONDS, TimeUnit.SECONDS);
        } catch (TimeoutException e) {
            throw new PaymentException(PaymentStatus.CONNECT_TIME_OUT.getCode(),request.toString(), PaymentStatus.CONNECT_TIME_OUT.getMessage());
        } catch (Exception e) {
            throw new PaymentException(PaymentStatus.PAY_ORDER_ERROR.getCode(), request.toString(), e.getMessage());
        }
    }
    public QueryRefundStatusResponse queryRefundStatus(QueryRefundStatusRequest request) {
        try {
            PaymentHandler handler = getPaymentHandler(bankName, integrationType);
            Future<QueryRefundStatusResponse> future = executorService.submit(() -> handler.queryRefundStatus(request,paymentConfig));
            return future.get(TIMEOUT_SECONDS, TimeUnit.SECONDS);
        } catch (TimeoutException e) {
            throw new PaymentException(PaymentStatus.CONNECT_TIME_OUT.getCode(),request.toString(), PaymentStatus.CONNECT_TIME_OUT.getMessage());
        } catch (Exception e) {
            throw new PaymentException(PaymentStatus.PAY_ORDER_ERROR.getCode(), request.toString(), e.getMessage());
        }
    }
    /**
     * 获取支付处理器
     */
    private PaymentHandler getPaymentHandler(BankName bankName, IntegrationType integrationType) {
        try {
            return PaymentHandlerFactory.getHandler(bankName, integrationType);
        } catch (PaymentException e) {
            throw new PaymentException(PaymentStatus.INVALID_CONNECTION_METHOD_OR_BANK_NAME.getCode(), "{}", PaymentStatus.INVALID_CONNECTION_METHOD_OR_BANK_NAME.getMessage());
        }
    }

    public String generateOrderNumber() {
        return "ORD-" + UUID.randomUUID().toString().replaceAll("-", "").substring(0, 16).toUpperCase();
    }

}
