package com.af.v4.system.common.payment.handler.impl.abc;

import com.af.v4.system.common.payment.annotation.PaymentHandlerConfig;
import com.af.v4.system.common.payment.dto.*;
import com.af.v4.system.common.payment.enums.BankName;
import com.af.v4.system.common.payment.enums.IntegrationType;
import com.af.v4.system.common.payment.enums.PaymentStatus;
import com.af.v4.system.common.payment.exceptions.PaymentException;
import com.af.v4.system.common.payment.handler.impl.AbstractPaymentHandler;
import com.af.v4.system.common.payment.utils.MD5Utils;
import com.af.v4.system.common.payment.utils.RSAUtils;
import com.af.v4.system.common.plugins.http.RestTools;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

/**
 * @author ly
 * @ClassName ABCAbstractPaymentHandler
 * @Description 中国农业银行主扫支付处理类
 * @DateTime 2024/12/23 11:27
 */
@Component
@PaymentHandlerConfig(bankName = BankName.ABC, integrationType = IntegrationType.AGGREGATE)
public class ABCAbstractPaymentHandler extends AbstractPaymentHandler {

    private static final Logger LOGGER = LoggerFactory.getLogger(ABCAbstractPaymentHandler.class);


    @Override
    protected JSONObject buildPayOrderRequestParams(PaymentOrderRequest request, JSONObject paymentConfig) {

        try {

            String posNo = null;
            if (request.getPosNo() != null ) {
                posNo = request.getPosNo();
            } else if (paymentConfig.has("posNo")) {
                posNo = paymentConfig.getString("posNo");
            }

            JSONObject requestJson = new JSONObject();
            requestJson.put("mid", paymentConfig.getString("merchantId"));
            requestJson.put("posNo", posNo);
            requestJson.put("txnAmt", request.getAmount());
            requestJson.put("tranType", "F");
            requestJson.put("merTradeNo", request.getOrderNo());
            return requestJson;
        }catch (Exception e){
            LOGGER.error("构建支付订单请求参数异常", e);
            throw new PaymentException(PaymentStatus.PAY_ORDER_BUILD_REQUEST_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.PAY_ORDER_BUILD_REQUEST_PARAM_ERROR.getMessage());
        }

    }

    @Override
    protected PaymentOrderResponse processPayOrderResponse(PaymentOrderResponse response, PaymentOrderRequest request, JSONObject resultJson, JSONObject paymentConfig) {

        if (!"00".equals(resultJson.getString("respCode"))){
            throw new PaymentException(PaymentStatus.PAY_FAIL.getCode(),
                    resultJson.optString("respMsg", "下单失败"),
                    PaymentStatus.PAY_FAIL.getMessage());
        }

        response.setState(PaymentStatus.PAY_SUCCESS);
        response.setMchId(resultJson.optString("mid"));
        response.setOrderNo(resultJson.optString("merTradeNo"));
        response.setPayUrl(resultJson.optString("scanCode"));
        response.setAmount(resultJson.optInt("txnAmt"));
        // 聚合码下单成功，等待用户扫码支付
        response.setMessage(resultJson.optString("respMsg", "生成动态聚合码成功"));
        return response;
    }

    @Override
    protected JSONObject buildQueryPaymentStatusRequestParams(QueryPaymentStatusRequest request, JSONObject paymentConfig) {
        try {
            JSONObject requestJson = new JSONObject();
            requestJson.put("mid", paymentConfig.getString("merchantId"));
            // posNo 从 customParams 中获取，如果没有则从 paymentConfig 中获取
            String posNo = null;
            if (request.getCustomParams() != null && request.getCustomParams().has("posNo")) {
                posNo = request.getCustomParams().getString("posNo");
            } else if (paymentConfig.has("posNo")) {
                posNo = paymentConfig.getString("posNo");
            }
            requestJson.put("posNo", posNo);
            requestJson.put("merTradeNo", request.getOrderNo());
            requestJson.put("tranType", "G");
            return requestJson;
        } catch (Exception e) {
            LOGGER.error("构建查询支付状态请求参数异常", e);
            throw new PaymentException(PaymentStatus.QUERY_ORDER_BUILD_REQUEST_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.QUERY_ORDER_BUILD_REQUEST_PARAM_ERROR.getMessage());
        }
    }

    @Override
    protected QueryPaymentStatusResponse processQueryPaymentStatusResponse(QueryPaymentStatusResponse response, QueryPaymentStatusRequest request, JSONObject resultJson) {
        try {
            if (!"00".equals(resultJson.getString("respCode"))) {
                response.setState(PaymentStatus.QUERY_ORDER_FAIL);
                response.setMessage(resultJson.optString("respMsg", "查询失败"));
                return response;
            }

            // 设置响应数据
            response.setMchId(resultJson.optString("mid"));
            response.setOrderNo(resultJson.optString("merTradeNo"));
            response.setTransactionId(resultJson.optString("transNo"));
            response.setAmount(resultJson.optInt("txnAmt"));

            // 设置支付完成时间，格式：bankDate + bankTime
            String bankDate = resultJson.optString("bankDate", "");
            String bankTime = resultJson.optString("bankTime", "");
            if (!bankDate.isEmpty() && !bankTime.isEmpty()) {
                response.setPaySuccessDate(bankDate + bankTime);
            }

            // 根据 respCode 设置支付状态
            response.setPaymentStatus(PaymentStatus.SUCCESS_FOR_PAYMENT.getMessage());
            response.setState(PaymentStatus.SUCCESS_FOR_PAYMENT);

        } catch (PaymentException e) {
            LOGGER.error("处理查询支付状态响应参数异常", e);
            throw e;
        }
        catch (Exception e) {
            LOGGER.error("处理查询支付状态响应参数异常", e);
            throw new PaymentException(PaymentStatus.QUERY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.QUERY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getMessage());
        }
        return response;
    }

    @Override
    protected JSONObject buildCancelOrderRequestParams(CancelPaymentRequest request, JSONObject paymentConfig) {
        return null;
    }

    @Override
    protected CancelPaymentResponse processCancelOrderResponse(CancelPaymentResponse response, CancelPaymentRequest request, JSONObject resultJson) {
        return null;
    }

    @Override
    protected JSONObject buildRefundOrderRequestParams(RefundPaymentRequest request, JSONObject paymentConfig) {
        try {
            JSONObject requestJson = new JSONObject();
            requestJson.put("mid", paymentConfig.getString("merchantId"));
            requestJson.put("vfTradeNo", request.getOrderNo());

             // posNo 从 customParams 中获取，如果没有则从 paymentConfig 中获取
            String posNo = null;
            if (request.getCustomParams() != null && request.getCustomParams().has("posNo")) {
                posNo = request.getCustomParams().getString("posNo");
            } else if (paymentConfig.has("posNo")) {
                posNo = paymentConfig.getString("posNo");
            }
            requestJson.put("posNo", posNo);
            requestJson.put("txnAmt", request.getRefundAmount());
            requestJson.put("merTradeNo", request.getRefundOrderNo());
            requestJson.put("tranType", "R");
            return requestJson;
        } catch (Exception e) {
            LOGGER.error("构建退款请求参数异常", e);
            throw new PaymentException(PaymentStatus.REFUND_BUILD_REQUEST_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.REFUND_BUILD_REQUEST_PARAM_ERROR.getMessage());
        }
    }

    @Override
    protected RefundPaymentResponse processRefundOrderResponse(RefundPaymentResponse response, RefundPaymentRequest request, JSONObject resultJson) {
        try {
            if (!"00".equals(resultJson.getString("respCode"))) {
                response.setCode(PaymentStatus.REFUND_REQUEST_FAIL.getCode());
                response.setMessage(resultJson.optString("respMsg", "退款申请失败"));
                return response;
            }

            // 设置响应数据
            response.setMchId(resultJson.optString("mid"));
            response.setOrderNo(resultJson.optString("merTradeNo"));
            response.setRefundOrderNo(resultJson.optString("vfTradeNo"));
            response.setTransactionId(resultJson.optString("transNo"));
            response.setRefundAmount(resultJson.optInt("txnAmt"));

            // 设置退款结果
            response.setRefundResult(PaymentStatus.REFUND_REQUEST_SUCCESS.getMessage());
            response.setState(PaymentStatus.REFUND_REQUEST_SUCCESS);

        } catch (Exception e) {
            LOGGER.error("处理退款响应参数异常", e);
            throw new PaymentException(PaymentStatus.REFUND_BUILD_RESPONSE_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.REFUND_BUILD_RESPONSE_PARAM_ERROR.getMessage());
        }
        return response;
    }

    @Override
    protected JSONObject buildQueryRefundStatusRequestParams(QueryRefundStatusRequest request, JSONObject paymentConfig) {
        return null;
    }

    @Override
    protected QueryRefundStatusResponse processQueryRefundStatusResponse(QueryRefundStatusResponse response, QueryRefundStatusRequest request, JSONObject resultJson) {
        return null;
    }

    @Override
    protected void signRequest(JSONObject reqParams, JSONObject paymentConfig) {
        try {
            String signed = MD5Utils.md5Upper(STR."\{reqParams.toString()}&\{paymentConfig.getString("md5Key")}");
            reqParams.put("sign", signed);
        }catch (Exception e){
            LOGGER.error("获取签名失败", e);
            throw new PaymentException(PaymentStatus.SIGN_FAIL.getCode(), reqParams.toString(), PaymentStatus.SIGN_FAIL.getMessage());
        }
    }

    @Override
    protected JSONObject executePaymentRequest(JSONObject reqParams, JSONObject paymentConfig) {

        try {
            String url = paymentConfig.getString("url");

            JSONObject header = new JSONObject();
            header.put("Content-Type", "application/json");
            header.put("sign", reqParams.getString("sign"));
            reqParams.remove("sign");

            // 获取请求报文
            String requestBody = reqParams.toString();

            // 检查是否需要RSA加密
            if (paymentConfig.has("rsaEncId") && paymentConfig.has("publicKey")) {
                String rsaEncId = paymentConfig.getString("rsaEncId");
                String publicKey = paymentConfig.getString("publicKey");

                LOGGER.info("检测到rsaEncId，开始RSA加密处理");

                // 使用公钥加密请求报文
                String encryptedBody = RSAUtils.encryptByPublicKey(requestBody, publicKey);
                if (encryptedBody == null) {
                    throw new PaymentException(PaymentStatus.PAY_FAIL.getCode(), "RSA加密失败", PaymentStatus.PAY_FAIL.getMessage());
                }

                // 将rsaEncId添加到header
                header.put("rsaEncId", rsaEncId);

                LOGGER.info("RSA加密成功，发送加密报文");
                requestBody = encryptedBody;
            }

            // 发送请求
            String result = RestTools.post(url, requestBody, header.toString());

            // 检查响应是否需要解密
            if (paymentConfig.has("rsaEncId") && paymentConfig.has("privateKey")) {
                String privateKey = paymentConfig.getString("privateKey");

                LOGGER.info("检测到rsaEncId，开始RSA解密响应");

                // 使用私钥解密响应
                String decryptedResult = RSAUtils.decryptByPrivateKey(result, privateKey);
                if (decryptedResult == null) {
                    throw new PaymentException(PaymentStatus.PAY_FAIL.getCode(), "RSA解密失败", PaymentStatus.PAY_FAIL.getMessage());
                }

                LOGGER.info("RSA解密成功");
                result = decryptedResult;
            }

            return new JSONObject(result);

        }catch (PaymentException e){
            throw e;
        } catch (Exception e){
            LOGGER.error("调用支付接口异常", e);
            throw new PaymentException(PaymentStatus.PAY_FAIL.getCode(), reqParams.toString(), PaymentStatus.PAY_FAIL.getMessage());
        }
    }

    @Override
    protected boolean verifyResponseSign(JSONObject response, JSONObject paymentConfig) {
        return true;
    }

    @Override
    protected JSONObject buildDownloadReconciliationFileRequestParams(DownloadReconciliationFileRequest request, JSONObject paymentConfig) {
        return null;
    }

    @Override
    protected DownloadReconciliationFileResponse processDownloadReconciliationFileResponse(DownloadReconciliationFileResponse response, DownloadReconciliationFileRequest request, JSONObject resultJson) {
        return null;
    }
}
