package com.af.v4.system.common.payment.enums;

/**
 * 邮政储蓄银行支付状态枚举
 * 
 * @author system
 * @since 2024/12/25
 */
public enum PsbcStatus {
    
    // ========== 订单支付 ==========
    /** 订单支付请求成功 */
    ORDER_PAYMENT_SUCCESS("000000", "订单支付请求成功"),
    
    /** 下单时间为空 */
    ORDER_TIME_EMPTY("PA000001", "下单时间为空"),
    
    /** 微信小程序appId或openId字段为空 */
    WECHAT_APPID_OPENID_EMPTY("PA000002", "微信小程序appId或openId字段为空"),
    
    /** 支付宝小程序openId字段为空 */
    ALIPAY_OPENID_EMPTY("PA000003", "支付宝小程序openId字段为空"),
    
    /** D模式下单金额与预定金额不匹配 */
    AMOUNT_MISMATCH("PA000004", "D模式下单金额与预定金额不匹配"),
    
    /** 订单已关闭 */
    ORDER_CLOSED("PA000005", "订单已关闭"),
    
    /** 没有开通渠道或者大商户模式 */
    CHANNEL_NOT_OPEN("PA000006", "没有开通渠道或者大商户模式"),
    
    /** 传入的收费单位编码不匹配 */
    UNIT_CODE_MISMATCH("PA000007", "传入的收费单位编码不匹配"),
    
    /** 暂不支持该客户缴费 */
    CUSTOMER_NOT_SUPPORTED("PA000008", "暂不支持该客户缴费"),
    
    /** 大商户模式入驻信息为空 */
    MERCHANT_INFO_EMPTY("PA000009", "大商户模式入驻信息为空"),
    
    /** 没有找到入驻商户数据 */
    MERCHANT_NOT_FOUND("PA000010", "没有找到入驻商户数据"),
    
    /** 未开通微信产品 */
    WECHAT_PRODUCT_NOT_OPEN("PA000011", "未开通微信产品"),
    
    /** 产品开通位图为空或者长度小于三位 */
    PRODUCT_BITMAP_INVALID("PA000012", "产品开通位图为空或者长度小于三位"),
    
    /** 二级商户号微信支付模式值错误 */
    WECHAT_MODE_ERROR("PA000013", "二级商户号微信支付模式值错误"),
    
    /** 入驻时未开通支付宝产品 */
    ALIPAY_PRODUCT_NOT_OPEN("PA000014", "入驻时未开通支付宝产品"),
    
    /** 二级商户号支付宝支付模式值错误 */
    ALIPAY_MODE_ERROR("PA000015", "二级商户号支付宝支付模式值错误"),
    
    /** 订单已支付 */
    ORDER_ALREADY_PAID("PA000016", "订单已支付"),
    
    /** 支付失败 */
    PAYMENT_FAILED("PA000017", "支付失败"),
    
    // ========== 缴费项目相关 ==========
    /** 缴费项目编号为空 */
    PROJECT_CODE_EMPTY("CP000001", "缴费项目编号为空"),
    
    /** 未查询到缴费项目 */
    PROJECT_NOT_FOUND("CP000002", "未查询到缴费项目"),
    
    /** 缴费项目上线状态为空 */
    PROJECT_STATUS_EMPTY("CP000003", "缴费项目上线状态为空"),
    
    /** 缴费项目未设置业务开关 */
    PROJECT_SWITCH_NOT_SET("CP000004", "缴费项目未设置业务开关"),
    
    /** 缴费项目已停用 */
    PROJECT_DISABLED("CP000005", "缴费项目已停用"),
    
    /** 缴费项目未设置业务允许开办时间 */
    PROJECT_START_TIME_NOT_SET("CP000006", "缴费项目未设置业务允许开办时间"),
    
    /** 缴费项目未设置业务允许结束时间 */
    PROJECT_END_TIME_NOT_SET("CP000007", "缴费项目未设置业务允许结束时间"),
    
    /** 缴费项目未到业务办理时间 */
    PROJECT_NOT_STARTED("CP000008", "缴费项目未到业务办理时间"),
    
    /** 缴费项目未设置渠道开通位图 */
    PROJECT_CHANNEL_BITMAP_NOT_SET("CP000009", "缴费项目未设置渠道开通位图"),
    
    /** 缴费项目未开通该渠道业务 */
    PROJECT_CHANNEL_NOT_OPEN("CP000010", "缴费项目未开通该渠道业务"),
    
    /** 缴费项目未设置商户单笔限额 */
    PROJECT_LIMIT_NOT_SET("CP000011", "缴费项目未设置商户单笔限额"),
    
    /** 缴费项目已配置暂停 */
    PROJECT_PAUSED("CP000012", "缴费项目已配置暂停"),
    
    // ========== 二维码支付订单相关 ==========
    /** 二维码支付订单支付模式异常 */
    QR_PAYMENT_MODE_ERROR("DP000001", "二维码支付订单支付模式异常"),
    
    /** 二维码支付订单缴费记录链接标志值异常 */
    QR_PAYMENT_LINK_FLAG_ERROR("DP000002", "二维码支付订单缴费记录链接标志值异常"),
    
    /** 二维码支付订单商户是否开启通知标志值异常 */
    QR_PAYMENT_NOTIFY_FLAG_ERROR("DP000003", "二维码支付订单商户是否开启通知标志值异常"),
    
    /** 二维码支付订单接入方式标志值异常 */
    QR_PAYMENT_ACCESS_MODE_ERROR("DP000004", "二维码支付订单接入方式标志值异常"),
    
    /** 二维码支付订单请求头传入渠道商代号异常 */
    QR_PAYMENT_DEALER_CODE_ERROR("DP000005", "二维码支付订单请求头传入渠道商代号异常"),
    
    /** 二维码支付订单请求头传入版本号异常 */
    QR_PAYMENT_VERSION_ERROR("DP000006", "二维码支付订单请求头传入版本号异常"),
    
    /** 二维码支付订单请求头传入交易码异常 */
    QR_PAYMENT_TRANSACTION_CODE_ERROR("DP000008", "二维码支付订单请求头传入交易码异常"),
    
    // ========== 内部户相关 ==========
    /** 委托单位清算内部户未配置或未同步 */
    INTERNAL_ACCOUNT_NOT_CONFIG("IC000001", "委托单位清算内部户未配置或未同步"),
    
    /** 内部机构开户机构不合法 */
    INTERNAL_ORG_INVALID("IC000002", "内部机构开户机构不合法"),
    
    /** 内部户账号长度不能小于16位 */
    INTERNAL_ACCOUNT_LENGTH_INVALID("IC000003", "内部户账号长度不能小于16位"),
    
    // ========== 幂等校验 ==========
    /** 幂等校验未通过 */
    IDEMPOTENT_CHECK_FAILED("JP000001", "幂等校验未通过"),
    
    // ========== 订单支付结果通知 ==========
    /** 订单支付结果通知请求接收成功 */
    NOTIFICATION_SUCCESS("000000", "订单支付结果通知请求接收成功"),
    
    // ========== 订单状态确认/查询退款订单状态 ==========
    /** 订单状态确认请求成功 */
    ORDER_STATUS_CONFIRM_SUCCESS("000000", "订单状态确认请求成功"),
    
    /** 订单不存在 */
    ORDER_NOT_EXISTS("CF000001", "订单不存在"),
    
    /** 二维码订单已关单 */
    QR_ORDER_CLOSED("CF000002", "二维码订单已关单"),
    
    /** 流水不存在 */
    TRANSACTION_NOT_EXISTS("CF000003", "流水不存在"),
    
    /** 订单状态异常 */
    ORDER_STATUS_ABNORMAL("CF000004", "订单状态异常"),

    /** 订单时间超过360天，无法查询 */
    ORDER_TIME_OVER_360_DAYS("JP126", "订单时间超过360天，无法查询"),
    
    // ========== 申请退款 ==========
    /** 退款成功 */
    REFUND_SUCCESS("000000", "退款成功"),
    
    /** 订单记录不存在 */
    ORDER_RECORD_NOT_EXISTS("RF000001", "订单记录不存在"),
    
    /** 渠道商订单号不唯一，不允许退款 */
    ORDER_NOT_UNIQUE("RF000002", "渠道商订单号不唯一，不允许退款"),
    
    /** 幂等校验未通过 */
    REFUND_IDEMPOTENT_CHECK_FAILED("RF000003", "幂等校验未通过"),
    
    /** 退款金额大于支付金额 */
    REFUND_AMOUNT_EXCEED("RF000004", "退款金额大于支付金额"),
    
    /** 当前状态不能退款 */
    REFUND_NOT_ALLOWED("RF000005", "当前状态不能退款"),
    
    /** 该支付方式暂不支持 */
    PAYMENT_METHOD_NOT_SUPPORTED("RF000006", "该支付方式暂不支持"),
    
    /** 超过退款时间有效期 */
    REFUND_TIME_EXPIRED("RF000007", "超过退款时间有效期"),
    
    /** 未查询到缴费项目 */
    REFUND_PROJECT_NOT_FOUND("RF000008", "未查询到缴费项目"),
    
    /** 缴费项目不支持退款 */
    PROJECT_REFUND_NOT_SUPPORTED("RF000009", "缴费项目不支持退款"),
    
    /** 当缴费项目业务开办时间为空 */
    PROJECT_BUSINESS_TIME_EMPTY("RF000010", "当缴费项目业务开办时间为空"),
    
    /** 当前未到缴费项目业务开办时间 */
    PROJECT_BUSINESS_NOT_STARTED("RF000011", "当前未到缴费项目业务开办时间"),
    
    /** 订单扩展记录不存在 */
    ORDER_EXTEND_NOT_EXISTS("RF000012", "订单扩展记录不存在"),
    
    /** 缴费订单号参与了优惠活动，暂不支持退款 */
    ORDER_DISCOUNT_REFUND_NOT_SUPPORTED("RF000013", "缴费订单号参与了优惠活动，暂不支持退款"),
    
    /** 状态正常的三方入驻信息不存在 */
    MERCHANT_INFO_NOT_EXISTS("RF000014", "状态正常的三方入驻信息不存在"),
    
    /** 未查询到伞列钱包信息，不允许退款 */
    WALLET_INFO_NOT_FOUND("RF000015", "未查询到伞列钱包信息，不允许退款"),
    
    /** 钱包类型，不支持退款 */
    WALLET_TYPE_NOT_SUPPORTED("RF000016", "（0-本贷本,1-银联,2-网联,4-数币）钱包类型，不支持退款"),
    
    /** 退款金额小于0 */
    REFUND_AMOUNT_NEGATIVE("RF000017", "退款金额小于0"),
    
    /** 退款单号已退款，不予受理 */
    REFUND_ALREADY_PROCESSED("RF000018", "退款单号已退款，不予受理"),
    
    /** 该缴费项目没有开通对应渠道 */
    PROJECT_CHANNEL_NOT_OPEN_FOR_REFUND("RF000019", "该缴费项目没有开通对应渠道"),
    
    // ========== 下载对账文件 ==========
    /** 对账文件下载成功 */
    RECONCILE_FILE_SUCCESS("000000", "对账文件下载成功"),
    
    /** 对账文件不存在 */
    RECONCILE_FILE_NOT_EXISTS("RE000001", "对账文件不存在"),
    
    /** 对账文件压缩加密失败 */
    RECONCILE_FILE_COMPRESS_FAILED("RE000002", "对账文件压缩加密失败"),
    
    // ========== 主动关闭订单 ==========
    /** 主动关闭订单成功 */
    CLOSE_ORDER_SUCCESS("000000", "主动关闭订单成功");
    
    private final String code;
    private final String message;
    
    PsbcStatus(String code, String message) {
        this.code = code;
        this.message = message;
    }
    
    public String getCode() {
        return code;
    }
    
    public String getMessage() {
        return message;
    }
    
    /**
     * 根据响应码获取状态枚举
     * 
     * @param code 响应码
     * @return 对应的状态枚举，如果未找到则返回null
     */
    public static PsbcStatus fromCode(String code) {
        if (code == null) {
            return null;
        }
        
        for (PsbcStatus status : values()) {
            if (status.getCode().equals(code)) {
                return status;
            }
        }
        return null;
    }
    
    /**
     * 判断是否为成功状态
     * 
     * @return true表示成功，false表示失败
     */
    public boolean isSuccess() {
        return "000000".equals(this.code);
    }
    
    @Override
    public String toString() {
        return STR."PsbcStatus{code='\{code}', message='\{message}'}";
    }
}
