package com.af.v4.system.common.payment.dto;

import org.json.JSONObject;

/**
 * @author llzh
 * @ClassName RefundPaymentRequest
 * @Description 退款申请请求参数
 * @DateTime 2024/12/25 13:46
 */
public class RefundPaymentRequest {
    /**
     * 商户订单号（业务系统生成的唯一订单号）
     */
    private final String orderNo;
    /**
     * 退款订单号（业务系统生成的唯一退款订单号）
     */

    private final String transactionId;
    /**
     * 退款订单号（业务系统生成的唯一退款订单号）
     */
    private final String refundOrderNo;
    /**
     * 支付金额（单位：分）
     */
    private final int amount;
    /**
     * 退款金额（单位：分）(要和订单金额保持一致)
     * （PS:虽然支持部分退款，但不建议）
     */
    private final int refundAmount;
    /**
     * 银行名称
     */
    private final String bankName;
    /**
     * 分公司名称
     */
    private final String orgName;

    /**
     * 自定义扩充参数
     */
    private final JSONObject customParams;

    // 私有构造函数，防止直接实例化
    private RefundPaymentRequest(Builder builder) {
        this.orderNo = builder.orderNo;
        this.refundOrderNo = builder.refundOrderNo;
        this.amount = builder.amount;
        this.refundAmount = builder.refundAmount;
        this.bankName = builder.bankName;
        this.orgName = builder.orgName;
        this.transactionId = builder.transactionId;
        this.customParams = builder.customParams;
    }

    public String getBankName() {
        return bankName;
    }

    public String getOrgName() {
        return orgName;
    }

    public String getOrderNo() {
        return orderNo;
    }

    public String getRefundOrderNo() {
        return refundOrderNo;
    }

    public String getTransactionId() {
        return transactionId;
    }

    public int getAmount() {
        return amount;
    }

    public int getRefundAmount() {
        return refundAmount;
    }

    public JSONObject getCustomParams() {
        return customParams;
    }

    // Builder 静态内部类
    public static class Builder {
        private String orderNo;
        private String refundOrderNo;
        private int amount;
        private int refundAmount;
        private String bankName;
        private String orgName;
        private String transactionId;
        private JSONObject customParams;    
        public Builder() {
        }

        public static Builder fromJsonObject(JSONObject jsonObject) {
            Builder builder = new Builder();
            if (jsonObject == null) {
                return builder;
            }
            if (jsonObject.has("orderNo")) {
                builder.orderNo(jsonObject.getString("orderNo"));
            }
            if (jsonObject.has("refundOrderNo")) {
                builder.refundOrderNo(jsonObject.getString("refundOrderNo"));
            }
            if (jsonObject.has("amount")) {
                builder.amount(jsonObject.getInt("amount"));
            }
            if (jsonObject.has("refundAmount")) {
                builder.refundAmount(jsonObject.getInt("refundAmount"));
            }
            if (jsonObject.has("bankName")) {
                builder.bankName(jsonObject.getString("bankName"));
            }
            if (jsonObject.has("orgName")) {
                builder.orgName(jsonObject.getString("orgName"));
            }
            builder.transactionId(jsonObject.optString("transactionId"));
            builder.customParams(jsonObject.optJSONObject("customParams", new JSONObject()));
            return builder;

        }

        public Builder orderNo(String orderNo) {
            this.orderNo = orderNo;
            return this;
        }

        public Builder refundOrderNo(String refundOrderNo) {
            this.refundOrderNo = refundOrderNo;
            return this;
        }

        public Builder amount(int amount) {
            this.amount = amount;
            return this;
        }

        public Builder refundAmount(int refundAmount) {
            this.refundAmount = refundAmount;
            return this;
        }

        public Builder orgName(String orgName) {
            this.orgName = orgName;
            return this;
        }

        public Builder bankName(String bankName) {
            this.bankName = bankName;
            return this;
        }

        public Builder transactionId(String transactionId) {
            this.transactionId = transactionId;
            return this;
        }

        public Builder customParams(JSONObject customParams) {
            this.customParams = customParams;
            return this;
        }

        public RefundPaymentRequest build() {
            return new RefundPaymentRequest(this);
        }
    }


}


