package com.af.v4.system.common.payment.handler.impl.rcb;


import com.af.v4.system.common.payment.annotation.PaymentHandlerConfig;
import com.af.v4.system.common.payment.dto.*;
import com.af.v4.system.common.payment.enums.BankName;
import com.af.v4.system.common.payment.enums.IntegrationType;
import com.af.v4.system.common.payment.enums.PaymentStatus;
import com.af.v4.system.common.payment.exceptions.PaymentException;
import com.af.v4.system.common.payment.handler.impl.AbstractPaymentHandler;
import com.af.v4.system.common.payment.utils.PaymentUtils;
import com.af.v4.system.common.payment.utils.MD5Utils;
import com.af.v4.system.common.plugins.date.DateTools;
import com.af.v4.system.common.plugins.http.RestTools;
import com.af.v4.system.common.plugins.json.JsonTools;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import java.util.Date;
import java.math.BigDecimal;
import java.math.RoundingMode;


/**
 * @Author: 张巧敏
 * @Description: 农商银行聚合支付处理器
 * @Date: 2025-08-25 17:46
 */
@Component
@PaymentHandlerConfig(bankName = BankName.HN_RCB, integrationType = IntegrationType.SCAN_DEVICE)
public class HNRCBAbstractPaymentHandlerImpl extends AbstractPaymentHandler {
    private static final Logger LOGGER = LoggerFactory.getLogger(HNRCBAbstractPaymentHandlerImpl.class);
    private static final String SUCCESS_CODE = "00";

    /**
     * 构建支付订单请求参数
     *
     * @param request       支付订单请求对象，包含业务相关的请求信息
     * @param paymentConfig 支付配置信息，包含支付渠道相关配置参数
     * @return 返回构建好的请求参数JSON对象
     */
    @Override
    protected JSONObject buildPayOrderRequestParams(PaymentOrderRequest request, JSONObject paymentConfig) {
        LOGGER.info(">>> 构建支付订单请求参数 > ");
        try {
            JSONObject reqParams = getRequestHeard(paymentConfig);
            reqParams.put("outTradeNo", request.getOrderNo());
            // 金额（分->元），按两位小数，作为字符串参与签名与发送
            String amountYuan = BigDecimal.valueOf(request.getAmount())
                    .divide(BigDecimal.valueOf(100), 2, RoundingMode.DOWN)
                    .toPlainString();
            reqParams.put("amount", amountYuan);
            reqParams.put("realAmount", amountYuan);
            reqParams.put("authCode", request.getAuthCode());

            // 终端IP地址，农商银行要求真实的消费者IP地址
            reqParams.put("terminal_ip", "127.0.0.1");
            //终端类型;终端类型：ANDROID:安卓;IOS:苹果;WEB:PC或H5;OTHER:其他
            reqParams.put("terminalType", "WEB");
            // 标记当前操作类型
            reqParams.put("_action", "PAY");
            return reqParams;
        } catch (Exception e) {
            LOGGER.error("构建支付订单请求参数异常", e);
            throw new PaymentException(PaymentStatus.PAY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.PAY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getMessage());
        }
    }


    /**
     * 处理支付订单响应结果
     *
     * @param response      支付响应
     * @param request       支付请求
     * @param resultJson    响应数据
     * @param paymentConfig 配置
     * @return 结果
     */
    @Override
    protected PaymentOrderResponse processPayOrderResponse(PaymentOrderResponse response, PaymentOrderRequest request, JSONObject resultJson, JSONObject paymentConfig) {
        LOGGER.info(">>> 处理支付订单响应结果 > ");
        try {
            boolean needQuery = true;
            String payState;
            // 接口处理结果编码00:成功；01:失败；02:参数不合法；03:验签失败；99:系统错误
            if (SUCCESS_CODE.equals(resultJson.optString("code"))) {
                response.setState(PaymentStatus.PAY_SUCCESS);
                response.setOrderNo(resultJson.getJSONObject("data").optString("outTradeNo"));
                // 交易状态 支付状态：O下单；I支付中；P已支付；F支付失败；T订单超时
                if ("P".equals(resultJson.getJSONObject("data").optString("payStatus"))) {
                    payState = "交易成功";
                    needQuery = false;
                } else {
                    payState = resultJson.optString("err_msg", "交易失败");
                }
                response.setNeedQuery(needQuery);
                response.setPayResult(payState);
            } else {
                LOGGER.info("本次支付订单交易通信失败！");
                response.setCode(PaymentStatus.COMMUNICATION_FAIL.getCode());
                response.setMessage(PaymentStatus.COMMUNICATION_FAIL.getMessage());
            }
        } catch (Exception e) {
            LOGGER.error("构建支付订单返回参数异常", e);
            throw new PaymentException(PaymentStatus.PAY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.PAY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getMessage());
        }
        return response;
    }


    /**
     * 构建查询支付状态请求参数
     *
     * @param request       查询支付状态请求对象
     * @param paymentConfig 支付配置信息
     * @return 返回构建好的查询支付状态请求参数JSON对象
     */
    @Override
    protected JSONObject buildQueryPaymentStatusRequestParams(QueryPaymentStatusRequest request, JSONObject paymentConfig) {
        LOGGER.info(">>> 构建查询支付状态请求参数 > ");
        try {
            JSONObject reqParams = getRequestHeard(paymentConfig);
            reqParams.put("outTradeNo", request.getOrderNo());
            // 标记当前操作类型
            reqParams.put("_action", "QUERY");
            return reqParams;
        } catch (Exception e) {
            LOGGER.error("构建查询订单支付状态请求参数异常", e);
            throw new PaymentException(PaymentStatus.QUERY_ORDER_BUILD_REQUEST_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.QUERY_ORDER_BUILD_REQUEST_PARAM_ERROR.getMessage());
        }
    }

    /**
     * 处理查询支付状态响应结果
     *
     * @param response   查询支付状态响应对象
     * @param request    查询支付状态请求对象
     * @param resultJson 支付状态查询接口返回的原始响应JSON数据
     * @return 返回处理后的查询支付状态响应对象
     */
    @Override
    protected QueryPaymentStatusResponse processQueryPaymentStatusResponse(QueryPaymentStatusResponse response, QueryPaymentStatusRequest request, JSONObject resultJson) {
        LOGGER.info("处理查询支付状态响应结果 > ");
        try {
            // 接口处理结果编码00:成功；01:失败；02:参数不合法；03:验签失败；99:系统错误
            if (SUCCESS_CODE.equals(resultJson.optString("code"))) {
                // 交易状态 支付状态：O下单；I支付中；P已支付；F支付失败；T订单超时；R退款；C已取消；N订单不存在
                if ("P".equals(resultJson.getJSONObject("data").optString("payStatus"))) {
                    response.setTransactionId(resultJson.getJSONObject("data").optString("outChannelNo"));
                    response.setOrderNo(resultJson.getJSONObject("data").optString("outTradeNo"));
                    response.setAmount(resultJson.getJSONObject("data").getInt("realAmount")*100);
                    String tradeState = resultJson.getJSONObject("data").optString("payStatus");
                    PaymentStatus status = switch (tradeState) {
                        case "P" -> { // 已支付
                            response.setPaymentStatus(PaymentStatus.SUCCESS_FOR_PAYMENT.getMessage());
                            response.setState(PaymentStatus.SUCCESS_FOR_PAYMENT);
                            // 农商银行使用 transTime 字段作为支付成功时间
                            String transTime = resultJson.getJSONObject("data").optString("transTime");
                            if (transTime != null && !transTime.isEmpty()) {
                                response.setPaySuccessDate(DateTools.formatDateTime(transTime));
                            }
                            yield PaymentStatus.SUCCESS_FOR_PAYMENT;
                        }
                        case "O" -> { // 下单
                            response.setState(PaymentStatus.NOT_FOR_PAYMENT);
                            yield PaymentStatus.NOT_FOR_PAYMENT;
                        }
                        case "I" -> { // 支付中
                            response.setState(PaymentStatus.PAYMENT_IN_PROGRESS);
                            yield PaymentStatus.PAYMENT_IN_PROGRESS;
                        }
                        case "F" -> { // 支付失败
                            response.setState(PaymentStatus.FAIL_FOR_PAYMENT);
                            yield PaymentStatus.FAIL_FOR_PAYMENT;
                        }
                        case "T" -> { // 订单超时
                            response.setState(PaymentStatus.PAY_ORDER_TIMEOUT);
                            yield PaymentStatus.PAY_ORDER_TIMEOUT;
                        }
                        case "R" -> { // 退款
                            response.setState(PaymentStatus.REFUND_FOR_PAYMENT);
                            yield PaymentStatus.REFUND_FOR_PAYMENT;
                        }
                        case "C" -> { // 已取消
                            response.setState(PaymentStatus.ORDER_BEEN_CLOSED);
                            yield PaymentStatus.ORDER_BEEN_CLOSED;
                        }
                        case "N" -> { // 订单不存在
                            response.setState(PaymentStatus.CANCEL_FOR_PAYMENT);
                            yield PaymentStatus.CANCEL_FOR_PAYMENT;
                        }
                        default -> {
                            response.setState(PaymentStatus.QUERY_ORDER_FAIL);
                            yield PaymentStatus.QUERY_ORDER_FAIL;
                        }
                    };
                    response.setPaymentStatus(status.getMessage());
                    response.setState(status);
                    // 农商银行使用 transTime 字段作为支付成功时间
                    String transTime = resultJson.getJSONObject("data").optString("transTime");
                    if (transTime != null && !transTime.isEmpty()) {
                        response.setPaySuccessDate(DateTools.formatDateTime(transTime));
                    }
                } else {
                    response.setState(PaymentStatus.PAY_FAIL);
                }
            } else {
                LOGGER.info("本次查询支付状态交易通信失败！");
                response.setCode(PaymentStatus.COMMUNICATION_FAIL.getCode());
                response.setMessage(PaymentStatus.COMMUNICATION_FAIL.getMessage());
            }
        } catch (Exception e) {
            LOGGER.error("构建查询支付状态返回参数异常", e);
            throw new PaymentException(PaymentStatus.QUERY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.QUERY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getMessage());
        }
        return response;
    }

    /**
     * 构建取消订单请求参数
     *
     * @param request       取消支付请求对象
     * @param paymentConfig 支付配置信息
     * @return 返回构建好的取消订单请求参数JSON对象
     */
    @Override
    protected JSONObject buildCancelOrderRequestParams(CancelPaymentRequest request, JSONObject paymentConfig) {
        return null;
    }

    /**
     * 处理取消订单响应结果
     *
     * @param response   取消支付响应对象
     * @param request    取消支付请求对象
     * @param resultJson 取消订单接口返回的原始响应JSON数据
     * @return 返回处理后的取消支付响应对象
     */
    @Override
    protected CancelPaymentResponse processCancelOrderResponse(CancelPaymentResponse response, CancelPaymentRequest request, JSONObject resultJson) {
        return null;
    }

    /**
     * 构建退款订单请求参数
     *
     * @param request       退款支付请求对象
     * @param paymentConfig 支付配置信息
     * @return 返回构建好的退款订单请求参数JSON对象
     */
    @Override
    protected JSONObject buildRefundOrderRequestParams(RefundPaymentRequest request, JSONObject paymentConfig) {
        LOGGER.info("构建退款订单请求参数 > ");
        try {
            JSONObject reqParams = getRequestHeard(paymentConfig);
            reqParams.put("loginNo", paymentConfig.getString("merchantLoginNo"));
            reqParams.put("outTradeNo", request.getOrderNo());
            // 退款金额（分->元），两位小数，字符串
            String refundAmountYuan = BigDecimal.valueOf(request.getRefundAmount())
                    .divide(BigDecimal.valueOf(100), 2, RoundingMode.DOWN)
                    .toPlainString();
            reqParams.put("refundAmount", refundAmountYuan);
            // 标记当前操作类型
            reqParams.put("_action", "REFUND");
            return reqParams;
        } catch (Exception e) {
            LOGGER.error("构建支付订单请求参数异常", e);
            throw new PaymentException(PaymentStatus.PAY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.PAY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getMessage());
        }
    }

    /**
     * 处理退款订单响应结果
     *
     * @param response   退款支付响应对象
     * @param request    退款支付请求对象
     * @param resultJson 退款接口返回的原始响应JSON数据
     * @return 返回处理后的退款支付响应对象
     */
    @Override
    protected RefundPaymentResponse processRefundOrderResponse(RefundPaymentResponse response, RefundPaymentRequest request, JSONObject resultJson) {
        LOGGER.info("处理订单退款响应结果 > ");
        try {
            // 接口处理结果编码00:成功；01:失败；02:参数不合法；03:验签失败；99:系统错误
            if (SUCCESS_CODE.equals(resultJson.optString("code"))) {
                // 交易状态 退款状态，R:退款成功；E:退款失败
                if ("P".equals(resultJson.getJSONObject("data").optString("payStatus"))) {
                    response.setRefundId(resultJson.getJSONObject("data").optString("refundOrderId"));
                    response.setOrderNo(resultJson.getJSONObject("data").optString("outTradeNo"));
                    response.setRefundAmount(resultJson.getJSONObject("data").getInt("amount")*100);
                    String tradeState = resultJson.getJSONObject("data").optString("refundStatus");
                    PaymentStatus status = switch (tradeState) {
                        case "R" -> PaymentStatus.REFUND_SUCCESS;
                        case "E" -> PaymentStatus.REFUND_FAIL;
                        default -> PaymentStatus.QUERY_REFUND_FAIL;
                    };
                    response.setRefundResult(status.getMessage());
                    response.setCode(status.getCode());
                    response.setMessage(status.getMessage());
                } else {
                    response.setCode(PaymentStatus.PAY_FAIL.getCode());
                    response.setMessage(PaymentStatus.PAY_FAIL.getMessage());
                }
            } else {
                LOGGER.info("本次订单退款交易通信失败！");
                response.setCode(PaymentStatus.COMMUNICATION_FAIL.getCode());
                response.setMessage(PaymentStatus.COMMUNICATION_FAIL.getMessage());
            }
        } catch (Exception e) {
            LOGGER.error("构建订单退款返回参数异常", e);
            throw new PaymentException(PaymentStatus.REFUND_BUILD_RESPONSE_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.QUERY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getMessage());
        }
        return response;
    }

    /**
     * 构建查询退款状态请求参数
     *
     * @param request       查询退款状态请求对象
     * @param paymentConfig 支付配置信息
     * @return 返回构建好的查询退款状态请求参数JSON对象
     */
    @Override
    protected JSONObject buildQueryRefundStatusRequestParams(QueryRefundStatusRequest request, JSONObject paymentConfig) {
        LOGGER.info("构建查询退款状态请求参数 > ");
        try {
            JSONObject reqParams = getRequestHeard(paymentConfig);
//            reqParams.put("outRefundNo", request.getRefundId());
            reqParams.put("refundNo", request.getRefundId());
            // 标记当前操作类型
            reqParams.put("_action", "REFUND_QUERY");
            return reqParams;
        } catch (Exception e) {
            LOGGER.error("构建查询退款请求参数异常", e);
            throw new PaymentException(PaymentStatus.QUERY_REFUND_BUILD_REQUEST_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.PAY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getMessage());
        }
    }

    /**
     * 处理查询退款状态响应结果
     *
     * @param response   查询退款状态响应对象
     * @param request    查询退款状态请求对象
     * @param resultJson 退款状态查询接口返回的原始响应JSON数据
     * @return 返回处理后的查询退款状态响应对象
     */
    @Override
    protected QueryRefundStatusResponse processQueryRefundStatusResponse(QueryRefundStatusResponse response, QueryRefundStatusRequest request, JSONObject resultJson) {
        LOGGER.info("处理查询退款状态响应结果 > ");
        try {
            // 接口处理结果编码00:成功；01:失败；02:参数不合法；03:验签失败；99:系统错误
            if (SUCCESS_CODE.equals(resultJson.optString("code"))) {
                JSONObject data = resultJson.optJSONObject("data");
                if (data == null) {
                    response.setCode(PaymentStatus.QUERY_REFUND_FAIL.getCode());
                    response.setMessage("查询退款状态响应无data字段");
                    return response;
                }

                // 按接口文档：refundStatus R=退款成功 E=退款失败 N=订单不存在
                String refundStatus = data.optString("refundStatus");
                PaymentStatus status = switch (refundStatus) {
                    case "R" -> PaymentStatus.REFUND_SUCCESS;
                    case "E" -> PaymentStatus.REFUND_FAIL;
                    case "N" -> PaymentStatus.REFUND_NOT_EXIST;
                    default -> PaymentStatus.QUERY_REFUND_FAIL;
                };

                // 填充字段映射
                response.setTransactionId(data.optString("outChannelNo"));
                response.setOrderNo(data.optString("outTradeNo"));
                response.setRefundOrderNo(data.optString("outRefundNo"));
                // 平台退款单号：优先 refundNo，其次 payOrderRefundId（两者都存在时任选其一）
                String refundId = data.optString("refundNo");
                if (refundId == null || refundId.isEmpty()) {
                    refundId = data.optString("payOrderRefundId");
                }
                response.setRefundId(refundId);
                // 退款金额：接口为元，DTO为分
                double refundAmountYuan = data.optDouble("refundAmount", 0.0);
                response.setRefundAmount((int) Math.round(refundAmountYuan * 100));
                // 退款时间
                String refundTime = data.optString("refundTime");
                if (refundTime != null && !refundTime.isEmpty()) {
                    response.setRefundDateTime(DateTools.formatDateTime(refundTime));
                }

                response.setRefundResult(status.getMessage());
                response.setCode(status.getCode());
                response.setMessage(status.getMessage());
            } else {
                LOGGER.info("本次查询退款状态交易通信失败！");
                response.setCode(PaymentStatus.COMMUNICATION_FAIL.getCode());
                response.setMessage(PaymentStatus.COMMUNICATION_FAIL.getMessage());
            }
        } catch (Exception e) {
            LOGGER.error("构建查询退款状态返回参数异常", e);
            throw new PaymentException(PaymentStatus.QUERY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.QUERY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getMessage());
        }
        return response;
    }

    /**
     * 对请求参数进行签名处理
     *
     * @param reqParams     请求参数JSON对象
     * @param paymentConfig 支付配置信息，包含签名所需密钥等参数
     */
    @Override
    protected void signRequest(JSONObject reqParams, JSONObject paymentConfig) {
        try {
            String md5Key = paymentConfig.optString("md5Key");
            if (StringUtils.isEmpty(md5Key)) {
                throw new PaymentException(
                        PaymentStatus.PAY_ORDER_ERROR.getCode(),
                        reqParams.toString(),
                        "农商银行MD5密钥未配置"
                );
            }

            LOGGER.info("使用的MD5密钥: {}", md5Key);

            // 使用MD5Utils进行签名
            MD5Utils.signRequestParams(reqParams, md5Key);

            LOGGER.info("签名后的请求参数: {}", reqParams);
        } catch (Exception e) {
            LOGGER.error("农商银行请求签名失败", e);
            throw new PaymentException(
                    PaymentStatus.PAY_ORDER_ERROR.getCode(),
                    reqParams.toString(),
                    STR."农商银行请求签名失败: \{e.getMessage()}"
            );
        }
    }

    /**
     * 执行支付请求
     *
     * @param reqParams     请求参数JSON对象
     * @param paymentConfig 支付配置信息
     * @return 返回支付接口响应的JSON数据
     */
    @Override
    protected JSONObject executePaymentRequest(JSONObject reqParams, JSONObject paymentConfig) {
        try {
            // 根据不同操作选择不同网关地址，未配置则回退到 transactionurl
            String action = reqParams.optString("_action", "PAY");
            String urlKey = switch (action) {
                case "QUERY" -> "queryUrl";
                case "REFUND" -> "refundUrl";
                case "REFUND_QUERY" -> "refundQueryUrl";
                default -> "payUrl"; // 下单/支付
            };
            String url = buildRequestUrl(paymentConfig, urlKey);

            JSONObject headers = new JSONObject();
            // 农商银行接口要求Content-Type为application/json
            headers.put("Content-Type", "application/json");

            // 农商银行接口要求请求体为JSON格式，不是XML
            String requestBody = reqParams.toString();
            LOGGER.info(">>> [{}] 请求URL={}, 请求参数：{}", action, url, requestBody);

            String response = RestTools.post(
                    url,
                    requestBody,
                    headers.toString()
            );

            // 农商银行接口返回的是JSON格式，不是XML格式
            JSONObject jsonObject;
            try {
                // 先尝试解析为JSON
                jsonObject = new JSONObject(response);
            } catch (Exception e) {
                // 如果JSON解析失败，再尝试XML转JSON
                LOGGER.warn("响应不是JSON格式，尝试XML转JSON: {}", e.getMessage());
                jsonObject = JsonTools.xmlConvertToJson(response);
            }

            LOGGER.info(">>> [{}] 响应结果：{}", action, jsonObject);
            return jsonObject;
        } catch (Exception e) {
            LOGGER.error("请求支付业务接口异常", e);
            throw new PaymentException(
                    PaymentStatus.PAY_ORDER_ERROR.getCode(),
                    reqParams.toString(),
                    PaymentStatus.PAY_ORDER_ERROR.getMessage()
            );
        }
    }

    /**
     * 构建请求URL：支持 base( transactionurl ) + 相对路径 的组合；
     * 若具体URL为绝对地址(http/https)，则直接使用。
     */
    private String buildRequestUrl(JSONObject paymentConfig, String urlKey) {
        String base = paymentConfig.optString("transactionurl", "");
        String specific = paymentConfig.optString(urlKey, "");
        if (specific == null || specific.isEmpty()) {
            return base;
        }
        String lower = specific.toLowerCase();
        if (lower.startsWith("http://") || lower.startsWith("https://")) {
            return specific;
        }
        // 视为相对路径，拼接到 base 后
        if (base == null || base.isEmpty()) {
            return specific; // 没有base只能原样返回
        }
        boolean baseEndsWithSlash = base.endsWith("/");
        boolean specStartsWithSlash = specific.startsWith("/");
        if (baseEndsWithSlash && specStartsWithSlash) {
            return base + specific.substring(1);
        } else if (!baseEndsWithSlash && !specStartsWithSlash) {
            return STR."\{base}/\{specific}";
        } else {
            return base + specific;
        }
    }

    /**
     * 验证响应数据签名
     *
     * @param response      支付接口响应JSON数据
     * @param paymentConfig 支付配置信息，包含验签所需密钥等参数
     * @return 验证通过返回true，否则返回false
     */
    @Override
    protected boolean verifyResponseSign(JSONObject response, JSONObject paymentConfig) {
        try {
            // 农商银行接口返回的是JSON格式，验签需要使用md5Key而不是publicKey
            String md5Key = paymentConfig.optString("md5Key", "");
            if (md5Key.isEmpty()) {
                // 如果没有配置md5Key，尝试使用publicKey作为备选
                md5Key = paymentConfig.optString("publicKey", "");
                LOGGER.warn("未配置md5Key，使用publicKey作为MD5密钥: {}", md5Key);
            }

            if (md5Key.isEmpty()) {
                LOGGER.error("验签失败：未配置MD5密钥");
                return false;
            }

            // 记录验签前的响应数据，用于调试
            LOGGER.info("验签前的响应数据: {}", response.toString());
            LOGGER.info("使用的MD5密钥: {}", md5Key);

            // 使用MD5Utils进行MD5验签，保持与原有逻辑兼容
            boolean verifyResult = MD5Utils.verifySign(response, md5Key);
            LOGGER.info("MD5验签结果: {}", verifyResult);

            return verifyResult;
        } catch (Exception e) {
            LOGGER.error("验签失败！", e);
            throw new PaymentException(PaymentStatus.VERIFY_SIGN_FAIL.getCode(), "{}", PaymentStatus.VERIFY_SIGN_FAIL.getMessage());
        }
    }


    /**
     * 获取请求头(公共)
     *
     * @param paymentConfig 支付配置
     * @return 请求头
     */
    private JSONObject getRequestHeard(JSONObject paymentConfig) {
        JSONObject reqParams = new JSONObject();
        reqParams.put("sign_type", "MD5");
        reqParams.put("nonce_str", String.valueOf(new Date().getTime()));
        // 系统编号
        reqParams.put("systemCode", paymentConfig.optString("systemCode"));
        // 接口版本号;
        reqParams.put("version", "1.0.0");
        // 接口请求时间;
        reqParams.put("reqtime", DateTools.getNow("yyyyMMddHHmmss"));
        // 随机码;
        reqParams.put("rancode", PaymentUtils.generateRandomString(6));
        // 服务商编号;
        reqParams.put("payment", paymentConfig.optString("channelId"));
        // 服务商编号; 农商银行测试环境要求使用40014
//        reqParams.put("payment", "40014");
        //收银员编号;
        reqParams.put("merchantLoginNo", paymentConfig.optString("merchantLoginNo"));
        //商户号;
        reqParams.put("merchantNo", paymentConfig.optString("mchId"));
        // MD5密钥
        reqParams.put("key", paymentConfig.optString("md5Key", paymentConfig.optString("publicKey", "")));

        return reqParams;
    }


}
