package com.af.v4.system.common.payment.handler.impl;

import com.af.v4.system.common.payment.dto.*;
import com.af.v4.system.common.payment.exceptions.PaymentException;
import com.af.v4.system.common.payment.handler.PaymentHandler;
import com.af.v4.system.common.payment.enums.PaymentStatus;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 抽象支付处理器
 * <p>
 * 定义支付功能的通用逻辑和扩展点，提供支付请求、响应处理等流程的模板方法。
 * 各银行支付渠道通过继承此类实现其具体逻辑。
 *
 * @Author llz
 * @Date 2024/12/26 22:12
 */
public abstract class AbstractPaymentHandler implements PaymentHandler {
    private static final Logger LOGGER = LoggerFactory.getLogger(AbstractPaymentHandler.class);

    /**
     * 支付订单
     *
     * @param request       支付订单请求
     * @param paymentConfig 支付配置
     * @return 支付响应
     */
    @Override
    public PaymentOrderResponse paymentOrder(PaymentOrderRequest request, JSONObject paymentConfig) {
        LOGGER.info("开始执行支付请求！");
        PaymentOrderResponse response = new PaymentOrderResponse();
        try {
            // 构建请求参数
            JSONObject reqParams = buildPayOrderRequestParams(request, paymentConfig);
            // 签名
            signRequest(reqParams, paymentConfig);
            // 发起支付请求
            JSONObject payOrderRes = executePaymentRequest(reqParams, paymentConfig);
            // 验签
            if (!verifyResponseSign(payOrderRes, paymentConfig)) {
                throw new PaymentException(PaymentStatus.VERIFY_SIGN_FAIL.getCode(), PaymentStatus.VERIFY_SIGN_FAIL.getMessage(), request.toString());
            }
            // 处理响应结果
            processPayOrderResponse(response, request, payOrderRes, paymentConfig);
        } catch (Exception e) {
            LOGGER.error("支付失败: 请求报文={}, 异常信息={}", request, e.getMessage());
            response.setCode(PaymentStatus.PAY_FAIL.getCode());
            response.setMessage(PaymentStatus.PAY_FAIL.getMessage());
            response.setRequestJson(new JSONObject(request.toString()));
        }
        return response;
    }


    /**
     * 查询订单支付状态
     *
     * @param request       查询订单支付状态请求参数
     * @param paymentConfig 支付配置
     * @return 查询订单支付状态响应
     */
    @Override
    public QueryPaymentStatusResponse queryPaymentStatus(QueryPaymentStatusRequest request, JSONObject paymentConfig) {
        LOGGER.info("开始执行查询订单支付状态请求！");
        QueryPaymentStatusResponse response = new QueryPaymentStatusResponse();
        JSONObject queryPaymentRes = null;
        try {
            // 构建请求参数
            JSONObject reqParams = buildQueryPaymentStatusRequestParams(request, paymentConfig);
            // 签名
            signRequest(reqParams, paymentConfig);
            // 发起支付请求
            queryPaymentRes = executePaymentRequest(reqParams, paymentConfig);
            // 验签
            if (!verifyResponseSign(queryPaymentRes, paymentConfig)) {
                throw new PaymentException(PaymentStatus.VERIFY_SIGN_FAIL.getCode(), PaymentStatus.VERIFY_SIGN_FAIL.getMessage(), request.toString());
            }
            // 处理响应结果
            processQueryPaymentStatusResponse(response, request, queryPaymentRes);
        } catch (Exception e) {
            LOGGER.error("查询订单支付状态失败: 请求报文={}, 异常信息={}", request, e.getMessage());
            response.setCode(PaymentStatus.QUERY_ORDER_FAIL.getCode());
            response.setMessage(PaymentStatus.QUERY_ORDER_FAIL.getMessage());
            response.setRequestJson(new JSONObject(request.toString()));
        }
        return response;
    }

    /**
     * 订单撤销
     *
     * @param request       订单撤销请求参数
     * @param paymentConfig 支付配置
     * @return 订单撤销响应
     */
    @Override
    public final CancelPaymentResponse cancelOrder(CancelPaymentRequest request, JSONObject paymentConfig) {
        LOGGER.info("开始执行订单撤销请求！");
        CancelPaymentResponse response = new CancelPaymentResponse();
        try {
            // 构建请求参数
            JSONObject reqParams = buildCancelOrderRequestParams(request, paymentConfig);
            // 签名
            signRequest(reqParams, paymentConfig);
            // 发起支付请求
            JSONObject queryPaymentRes = executePaymentRequest(reqParams, paymentConfig);
            // 验签
            if (!verifyResponseSign(queryPaymentRes, paymentConfig)) {
                throw new PaymentException(PaymentStatus.VERIFY_SIGN_FAIL.getCode(), PaymentStatus.VERIFY_SIGN_FAIL.getMessage(), request.toString());
            }
            // 处理响应结果
            processCancelOrderResponse(response, request, queryPaymentRes);
        } catch (Exception e) {
            LOGGER.error("订单撤销失败: 请求报文={}, 异常信息={}", request, e.getMessage());
            response.setCode(PaymentStatus.QUERY_ORDER_FAIL.getCode());
            response.setMessage(PaymentStatus.QUERY_ORDER_FAIL.getMessage());
            response.setRequestJson(new JSONObject(request.toString()));
        }
        return response;
    }


    /**
     * 订单退款
     *
     * @param request       订单退款请求参数
     * @param paymentConfig 支付配置
     * @return 订单退款响应
     */
    @Override
    public RefundPaymentResponse refundPayment(RefundPaymentRequest request, JSONObject paymentConfig) {
        LOGGER.info("开始执行订单退款请求！");
        RefundPaymentResponse response = new RefundPaymentResponse();
        try {
            // 构建请求参数
            JSONObject reqParams = buildRefundOrderRequestParams(request, paymentConfig);
            // 签名
            signRequest(reqParams, paymentConfig);
            // 发起支付请求
            JSONObject queryPaymentRes = executePaymentRequest(reqParams, paymentConfig);
            // 验签
            if (!verifyResponseSign(queryPaymentRes, paymentConfig)) {
                throw new PaymentException(PaymentStatus.VERIFY_SIGN_FAIL.getCode(), PaymentStatus.VERIFY_SIGN_FAIL.getMessage(), request.toString());
            }
            // 处理响应结果
            processRefundOrderResponse(response, request, queryPaymentRes);
        } catch (Exception e) {
            LOGGER.error("订单退款失败: 请求报文={}, 异常信息={}", request, e.getMessage());
            response.setCode(PaymentStatus.QUERY_ORDER_FAIL.getCode());
            response.setMessage(PaymentStatus.QUERY_ORDER_FAIL.getMessage());
            response.setRequestJson(new JSONObject(request.toString()));
        }
        return response;
    }


    /**
     * 订单退款
     *
     * @param request       订单退款请求参数
     * @param paymentConfig 支付配置
     * @return 订单退款响应
     */
    @Override
    public final QueryRefundStatusResponse queryRefundStatus(QueryRefundStatusRequest request, JSONObject paymentConfig) {
        LOGGER.info("开始执行查询退款状态请求！");
        QueryRefundStatusResponse response = new QueryRefundStatusResponse();
        try {
            // 构建请求参数
            JSONObject reqParams = buildQueryRefundStatusRequestParams(request, paymentConfig);
            // 签名
            signRequest(reqParams, paymentConfig);
            // 发起支付请求
            JSONObject queryPaymentRes = executePaymentRequest(reqParams, paymentConfig);
            // 验签
            if (!verifyResponseSign(queryPaymentRes, paymentConfig)) {
                throw new PaymentException(PaymentStatus.VERIFY_SIGN_FAIL.getCode(), PaymentStatus.VERIFY_SIGN_FAIL.getMessage(), request.toString());
            }
            // 处理响应结果
            processQueryRefundStatusResponse(response, request, queryPaymentRes);
        } catch (Exception e) {
            LOGGER.warn("查询退款状态失败: 请求报文={},异常信息={}", request, e.getMessage());
            response.setCode(PaymentStatus.QUERY_ORDER_FAIL.getCode());
            response.setMessage(PaymentStatus.QUERY_ORDER_FAIL.getMessage());
            response.setRequestJson(new JSONObject(request.toString()));
        }
        return response;
    }


    /**
     * 构建支付订单请求参数
     * 由子类实现，提供具体的支付参数
     *
     * @param request       支付请求
     * @param paymentConfig 支付配置
     * @return 请求参数
     */
    protected abstract JSONObject buildPayOrderRequestParams(PaymentOrderRequest request, JSONObject paymentConfig);


    /**
     * 处理支付响应结果
     * 由子类实现
     *
     * @param response   支付响应
     * @param request    支付请求
     * @param resultJson 响应数据
     */
    protected abstract PaymentOrderResponse processPayOrderResponse(PaymentOrderResponse response, PaymentOrderRequest request, JSONObject resultJson, JSONObject paymentConfig);

    /**
     * 构建查询订单支付状态请求参数
     * 由子类实现，提供具体的支付参数
     *
     * @param request       支付请求
     * @param paymentConfig 支付配置
     * @return 请求参数
     */
    protected abstract JSONObject buildQueryPaymentStatusRequestParams(QueryPaymentStatusRequest request, JSONObject paymentConfig);

    /**
     * 处理查询支付状态响应结果
     * 由子类实现
     *
     * @param response   支付响应
     * @param resultJson 响应数据
     */
    protected abstract QueryPaymentStatusResponse processQueryPaymentStatusResponse(QueryPaymentStatusResponse response, QueryPaymentStatusRequest request, JSONObject resultJson);


    /**
     * 构建订单撤销请求参数
     * 由子类实现，提供具体的支付参数
     *
     * @param request       支付请求
     * @param paymentConfig 支付配置
     * @return 请求参数
     */
    protected abstract JSONObject buildCancelOrderRequestParams(CancelPaymentRequest request, JSONObject paymentConfig);


    /**
     * 处理订单撤销响应结果 由子类实现
     *
     * @param response   撤销响应
     * @param resultJson 响应数据
     */
    protected abstract CancelPaymentResponse processCancelOrderResponse(CancelPaymentResponse response, CancelPaymentRequest request, JSONObject resultJson);


    /**
     * 构建退款请求参数
     * 由子类实现，提供具体的支付参数
     *
     * @param request       支付请求
     * @param paymentConfig 支付配置
     * @return 请求参数
     */
    protected abstract JSONObject buildRefundOrderRequestParams(RefundPaymentRequest request, JSONObject paymentConfig);


    /**
     * 构建退款请求参数
     * 由子类实现，提供具体的支付参数
     *
     * @param response   支付请求
     * @param resultJson 响应参数
     * @return 退款返回参数
     */
    protected abstract RefundPaymentResponse processRefundOrderResponse(RefundPaymentResponse response, RefundPaymentRequest request, JSONObject resultJson);

    /**
     * 构建查询退款状态请求参数
     * 由子类实现，提供具体的支付参数
     *
     * @param request       支付请求
     * @param paymentConfig 支付配置
     * @return 请求参数
     */
    protected abstract JSONObject buildQueryRefundStatusRequestParams(QueryRefundStatusRequest request, JSONObject paymentConfig);


    /**
     * 构建查询退款状态请求参数
     * 由子类实现，提供具体的支付参数
     *
     * @param response   支付请求
     * @param resultJson 响应数据
     */
    protected abstract QueryRefundStatusResponse processQueryRefundStatusResponse(QueryRefundStatusResponse response, QueryRefundStatusRequest request, JSONObject resultJson);


    /**
     * 签名请求参数
     * 默认实现，子类可覆盖
     *
     * @param reqParams     请求参数
     * @param paymentConfig 支付配置
     */
    protected abstract void signRequest(JSONObject reqParams, JSONObject paymentConfig);

    /**
     * 发起支付请求
     * 默认实现，子类可覆盖
     *
     * @param reqParams     请求参数
     * @param paymentConfig 支付配置
     * @return 响应结果
     */
    protected abstract JSONObject executePaymentRequest(JSONObject reqParams, JSONObject paymentConfig);


    /**
     * 验签响应结果
     * 默认实现，子类可覆盖
     *
     * @param response      响应结果
     * @param paymentConfig 支付配置
     * @return 验签是否成功
     */
    protected abstract boolean verifyResponseSign(JSONObject response, JSONObject paymentConfig);

}
