package com.af.v4.system.common.payment.handler.impl.hbbk;

import com.af.v4.system.common.core.utils.StringUtils;
import com.af.v4.system.common.payment.annotation.PaymentHandlerConfig;
import com.af.v4.system.common.payment.dto.*;
import com.af.v4.system.common.payment.enums.BankName;
import com.af.v4.system.common.payment.enums.IntegrationType;
import com.af.v4.system.common.payment.enums.PaymentStatus;
import com.af.v4.system.common.payment.exceptions.PaymentException;
import com.af.v4.system.common.payment.handler.impl.AbstractPaymentHandler;
import com.af.v4.system.common.payment.handler.impl.ceb.CEBAggregatePaymentHandler;
import com.af.v4.system.common.payment.utils.PaymentUtils;
import com.af.v4.system.common.payment.utils.SignUtils;
import com.af.v4.system.common.plugins.date.DateTools;
import com.af.v4.system.common.plugins.http.RestTools;
import com.af.v4.system.common.plugins.json.JsonTools;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.util.Date;

/**
 * @Author: 柳博
 * @Description: 河北银行扫码支付（被扫）处理器
 * @Date: 2025-02-08 16:25
 */
@Component
@PaymentHandlerConfig(bankName = BankName.HBBK, integrationType = IntegrationType.AGGREGATE)
public class HbbkAbstractPaymentHandlerImpl extends AbstractPaymentHandler {

    private static final Logger LOGGER = LoggerFactory.getLogger(HbbkAbstractPaymentHandlerImpl.class);

    private final String SUCCESS_CODE = "0";

    /**
     * 构建支付订单请求参数
     *
     * @param request       支付请求
     * @param paymentConfig 支付配置
     *
     * @return 请求参数
     */
    @Override
    protected JSONObject buildPayOrderRequestParams(PaymentOrderRequest request, JSONObject paymentConfig) {
        LOGGER. info("构建获取收款二维码请求参数 > ");
        try {
            return getCommonRequest(paymentConfig)
                    .put("service", "unified.trade.native")
                    .put("out_trade_no", request.getOrderNo())
                    .put("body", request.getDescription())
                    .put("total_fee", request.getAmount())
                    // 支付通知地址(用于支付成功后银行主动通知)
                    .put("notify_url",
                            StringUtils.isBlank(request.getNotifyUrl()) ?
                            paymentConfig.optString("notifyUrl", "127.0.0.1") :
                            request.getNotifyUrl()
                    )
                    // 终端出网ip(不允许上传172.192.10等开头的内网ip)
                    .put("mch_create_ip", paymentConfig.opt("mchServiceIP"))
                    .put("terminal_info", new JSONObject()
                            .put("terminal_type", "08")
                            .put("terminal_id", "**")
                            .put("app_version", "1.000000").toString()
                    );
        } catch (Exception e) {
            LOGGER.error("构建支付订单请求参数异常", e);
            throw new PaymentException(PaymentStatus.PAY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.PAY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getMessage());
        }
    }

    /**
     * 订单响应结果处理
     *
     * @param response   支付响应
     * @param request    支付请求
     * @param resultJson 响应数据
     *
     * @return 处理结果
     */
    @Override
    protected PaymentOrderResponse processPayOrderResponse(PaymentOrderResponse response, PaymentOrderRequest request, JSONObject resultJson, JSONObject paymentConfig) {
        LOGGER. info("处理获取二维码响应结果 > ");
        try {
            // 0表示通信成功，非0表示失败此字段是通信标识，非交易标识，交易是否成功需要查看 result_code 来判断
            boolean needQuery = true;   // 默认需要再次查询支付状态
            String payState;
            if (SUCCESS_CODE.equals(resultJson.optString("status"))) {
                response.setState(PaymentStatus.PAY_SUCCESS);
                response.setMchId(resultJson.optString("mch_id"));
                // 交易是否成功需要查看 result_code 来判断 0表示成功，非0表示失败
                if (SUCCESS_CODE.equals(resultJson.optString("result_code"))) {
                    payState = "交易成功";
                    needQuery = false;
                    response.setPayUrl(resultJson.optString("code_url"));
                    response.setCustomParam(new JSONObject().put("payImgUrl", resultJson.optString("code_img_url")).toString());
                } else {
                    payState = resultJson.optString("err_msg", "交易失败");
                }
                response.setNeedQuery(needQuery);
                response.setPayResult(payState);
            } else {
                LOGGER. info("本次支付订单交易通信失败！");
                response.setState(PaymentStatus.COMMUNICATION_FAIL);
            }
        } catch (Exception e) {
            LOGGER.error("构建支付订单返回参数异常", e);
            throw new PaymentException(PaymentStatus.PAY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.PAY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getMessage());
        }
        return response;
    }

    /**
     * 构建查询支付状态请求参数
     *
     * @param request       支付请求
     * @param paymentConfig 支付配置
     *
     * @return 支付状态查询参数
     */
    @Override
    protected JSONObject buildQueryPaymentStatusRequestParams(QueryPaymentStatusRequest request, JSONObject paymentConfig) {
        LOGGER. info("构建查询支付状态请求参数 > ");
        try {
            return getCommonRequest(paymentConfig)
                    .put("service", "unified.trade.query")
                    .put("out_trade_no", request.getOrderNo());
        } catch (Exception e) {
            LOGGER.error("构建查询订单支付状态请求参数异常", e);
            throw new PaymentException(PaymentStatus.QUERY_ORDER_BUILD_REQUEST_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.QUERY_ORDER_BUILD_REQUEST_PARAM_ERROR.getMessage());
        }
    }

    /**
     * 处理支付状态查询响应结果
     *
     * @param response   支付响应
     * @param request    请求参数
     * @param resultJson 响应数据
     *
     * @return 处理结果
     */
    @Override
    protected QueryPaymentStatusResponse processQueryPaymentStatusResponse(QueryPaymentStatusResponse response, QueryPaymentStatusRequest request, JSONObject resultJson) {
        LOGGER. info("处理查询支付状态响应结果 > ");
        try {
            // 0表示通信成功，非0表示失败此字段是通信标识，非交易标识，交易是否成功需要查看 trade_state 来判断
            if (SUCCESS_CODE.equals(resultJson.optString("status"))) {
                // 业务结果 交易是否成功需要查看 trade_state 来判断 0表示成功，非0表示失败
                if (SUCCESS_CODE.equals(resultJson.optString("result_code"))) {
                    response.setMchId(resultJson.optString("mch_id"));
                    response.setOrderNo(resultJson.optString("out_trade_no"));
                    response.setTransactionId(resultJson.optString("transaction_id"));
                    response.setAmount(resultJson.optInt("total_fee"));

                    String tradeState = resultJson.optString("trade_state");

                    PaymentStatus status = switch (tradeState) {
                        case "SUCCESS" -> PaymentStatus.SUCCESS_FOR_PAYMENT;
                        case "REFUND" -> PaymentStatus.REFUND_FOR_PAYMENT;
                        case "NOTPAY" -> PaymentStatus.NOT_FOR_PAYMENT;
                        case "CLOSED" -> PaymentStatus.ORDER_BEEN_CLOSED;
                        default -> PaymentStatus.FAIL_FOR_PAYMENT;
                    };

                    response.setPaymentStatus(status.getMessage());
                    response.setState(status);
                    response.setPaySuccessDate(DateTools.formatDateTime(resultJson.optString("time_end", DateTools.getNow("yyyyMMddHHmmss"))));
                } else {
                    response.setState(PaymentStatus.PAY_FAIL);
                }
            } else {
                LOGGER. info("本次查询支付状态交易通信失败！");
                response.setCode(PaymentStatus.COMMUNICATION_FAIL.getCode());
                response.setMessage(PaymentStatus.COMMUNICATION_FAIL.getMessage());
            }
        } catch (Exception e) {
            LOGGER.error("构建查询支付状态返回参数异常", e);
            throw new PaymentException(PaymentStatus.QUERY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.QUERY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getMessage());
        }
        return response;
    }

    /**
     * 构建撤销订单请求参数(不支持)
     *
     * @param request       支付请求
     * @param paymentConfig 支付配置
     *
     * @return 订单撤销请求参数
     */
    @Override
    protected JSONObject buildCancelOrderRequestParams(CancelPaymentRequest request, JSONObject paymentConfig) {
        return null;
    }

    /**
     * 处理撤销订单响应结果(不支持)
     *
     * @param response   撤销响应
     * @param request    请求数据
     * @param resultJson 响应数据
     *
     * @return 处理结果
     */
    @Override
    protected CancelPaymentResponse processCancelOrderResponse(CancelPaymentResponse response, CancelPaymentRequest request, JSONObject resultJson) {
        return null;
    }

    /**
     * 构建退款订单请求参数
     * 同步返回，退到银行卡是非实时的，每个银行处理速度不同，一般需要1-3个工作日
     * 同一笔订单的部分退款需要设置相同的订单号和不同的out_refund_no，一笔失败后重新提交要采用原来的out_refund_no
     * 退款总金额不能超过用户实际支付金额（现金券金额不予退款）
     *
     * @param request       支付请求
     * @param paymentConfig 支付配置
     *
     * @return 请求参数
     */
    @Override
    protected JSONObject buildRefundOrderRequestParams(RefundPaymentRequest request, JSONObject paymentConfig) {
        LOGGER. info("构建退款订单请求参数 > ");
        try {
            return getCommonRequest(paymentConfig)
                    .put("service", "unified.trade.refund")
                    .put("out_trade_no", request.getOrderNo())
                    .put("out_refund_no", request.getRefundOrderNo())
                    .put("total_fee", request.getAmount())
                    .put("refund_fee", request.getRefundAmount())
                    .put("op_user_id", paymentConfig.getString("mchId"));
        } catch (Exception e) {
            LOGGER.error("构建订单撤销请求参数异常", e);
            throw new PaymentException(PaymentStatus.REFUND_BUILD_REQUEST_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.REFUND_BUILD_REQUEST_PARAM_ERROR.getMessage());
        }
    }

    /**
     * 处理退款订单响应结果
     *
     * @param response   支付请求
     * @param request    请求参数
     * @param resultJson 响应参数
     *
     * @return 处理结果
     */
    @Override
    protected RefundPaymentResponse processRefundOrderResponse(RefundPaymentResponse response, RefundPaymentRequest request, JSONObject resultJson) {
        LOGGER. info("处理订单退款响应结果 > ");
        try {
            // 0表示通信成功，非0表示失败此字段是通信标识，非交易标识，交易是否成功需要查看 result_code 来判断
            if(SUCCESS_CODE.equals(resultJson.optString("status"))){
                // 此处返回0 表示退款申请接收成功，实际的退款结果根据退款查询接口查询
                if(SUCCESS_CODE.equals(resultJson.optString("result_code"))) {
                    response.setMchId(resultJson.optString("mch_id"));
                    response.setOrderNo(resultJson.optString("out_trade_no"));
                    response.setTransactionId(resultJson.optString("transaction_id"));
                    response.setRefundOrderNo(resultJson.optString("out_refund_no"));
                    response.setRefundId(resultJson.optString("refund_id"));
                    response.setRefundAmount(resultJson.optInt("refund_fee"));

                    response.setRefundResult(PaymentStatus.REFUND_REQUEST_SUCCESS.getMessage());
                    response.setState(PaymentStatus.REFUND_REQUEST_SUCCESS);
                }else {
                    response.setCode(PaymentStatus.REFUND_REQUEST_FAIL.getCode());
                    response.setMessage(resultJson.optString("message", PaymentStatus.REFUND_REQUEST_FAIL.getMessage()));
                }
            }else{
                LOGGER. info("本次订单退款交易通信失败！");
                response.setState(PaymentStatus.COMMUNICATION_FAIL);
            }
        }catch (Exception e) {
            LOGGER.error(" 构建订单退款返回参数异常", e);
            throw new PaymentException(PaymentStatus.REFUND_BUILD_RESPONSE_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.REFUND_BUILD_RESPONSE_PARAM_ERROR.getMessage());
        }
        return response;
    }

    /**
     * 构建退款状态查询请求参数
     *
     * @param request       支付请求
     * @param paymentConfig 支付配置
     *
     * @return 请求参数
     */
    @Override
    protected JSONObject buildQueryRefundStatusRequestParams(QueryRefundStatusRequest request, JSONObject paymentConfig) {
        LOGGER. info("构建查询退款状态请求参数 > ");
        try {
            return getCommonRequest(paymentConfig)
                    .put("service", "unified.trade.refundquery")
                    .put("out_trade_no", request.getOrderNo())
                    .put("refund_id", request.getRefundId());
        } catch (Exception e) {
            LOGGER.error("构建查询退款请求参数异常", e);
            throw new PaymentException(PaymentStatus.QUERY_REFUND_BUILD_REQUEST_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.REFUND_BUILD_REQUEST_PARAM_ERROR.getMessage());
        }
    }

    /**
     * 处理退款状态查询响应结果
     *
     * @param response   支付请求
     * @param request    请求参数
     * @param resultJson 响应数据
     *
     * @return 处理结果
     */
    @Override
    protected QueryRefundStatusResponse processQueryRefundStatusResponse(QueryRefundStatusResponse response, QueryRefundStatusRequest request, JSONObject resultJson) {
        LOGGER. info("处理查询退款状态响应结果 > ");
        try {
            // 0表示通信成功，非0表示失败此字段是通信标识，非交易标识，交易是否成功需要查看 result_code 来判断
            if(SUCCESS_CODE.equals(resultJson.optString("status"))){
                // 交易是否成功需要查看 result_code 来判断 0表示成功，非0表示失败
                if(SUCCESS_CODE.equals(resultJson.optString("result_code"))) {
                    response.setMchId(resultJson.optString("mch_id"));
                    response.setTransactionId(resultJson.optString("transaction_id"));
                    response.setOrderNo(resultJson.optString("out_trade_no"));
                    response.setRefundOrderNo(resultJson.optString("out_refund_no_0"));
                    response.setRefundId(resultJson.optString("refund_id_0"));
                    response.setRefundAmount(resultJson.optInt("refund_fee_0"));
                    response.setRefundDateTime(resultJson.optString("refund_time_0"));

                    String refundStatus = resultJson.optString("refund_status_0");

                    PaymentStatus status = switch (refundStatus) {
                        case "SUCCESS" -> PaymentStatus.REFUND_SUCCESS;
                        case "PROCESSING" -> PaymentStatus.REFUND_PROCESSING;
                        case "CHANGE" -> PaymentStatus.REFUND_CHANGE;
                        default -> PaymentStatus.REFUND_FAIL;
                    };

                    response.setRefundResult(status.getMessage());
                    response.setState(status);
                }else {
                    response.setState(PaymentStatus.REFUND_FAIL);
                }
            }else{
                LOGGER. info("本次查询退款状态交易通信失败！");
                response.setState(PaymentStatus.COMMUNICATION_FAIL);
            }
        }catch (Exception e) {
            LOGGER.error("构建查询退款状态返回参数异常", e);
            throw new PaymentException(PaymentStatus.REFUND_BUILD_RESPONSE_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.REFUND_BUILD_RESPONSE_PARAM_ERROR.getMessage());
        }
        return response;
    }

    /**
     * 签名请求
     *
     * @param reqParams     请求参数
     * @param paymentConfig 支付配置
     */
    @Override
    protected void signRequest(JSONObject reqParams, JSONObject paymentConfig) {
        try {
            SignUtils.signRequestParams(reqParams, "RSA_1_256", "sign", paymentConfig.getString("privateKey"));
        } catch (Exception e) {
            LOGGER.error("签名失败", e);
            throw new PaymentException(
                    PaymentStatus.SIGN_FAIL.getCode(),
                    reqParams.toString(),
                    PaymentStatus.SIGN_FAIL.getMessage()
            );
        }
    }

    /**
     * 执行支付业务请求
     *
     * @param reqParams     请求参数
     * @param paymentConfig 支付配置
     *
     * @return 业务执行请求结果
     */
    @Override
    protected JSONObject executePaymentRequest(JSONObject reqParams, JSONObject paymentConfig) {
        try {
            JSONObject headers = new JSONObject();
            String formattedRequest = JsonTools.jsonConvertToXml(reqParams);
            LOGGER. info("请求支付业务, 请求参数：{}", formattedRequest);
            headers.put("Content-Type", "application/xml");
            String response = RestTools.post(
                    paymentConfig.getString("transactionurl"),
                    formattedRequest,
                    headers.toString()
            );
            JSONObject jsonObject = JsonTools.xmlConvertToJson(response);
            LOGGER. info("响应结果：" + jsonObject.toString());
            return jsonObject;
        } catch (Exception e) {
            LOGGER.error("请求支付业务接口异常", e);
            throw new PaymentException(
                    PaymentStatus.PAY_ORDER_ERROR.getCode(),
                    reqParams.toString(),
                    PaymentStatus.PAY_ORDER_ERROR.getMessage()
            );
        }
    }

    /**
     * 响应结果签名验证
     *
     * @param response      响应结果
     * @param paymentConfig 支付配置
     *
     * @return 验签是否通过
     */
    @Override
    protected boolean verifyResponseSign(JSONObject response, JSONObject paymentConfig) {
        try {
            return SignUtils.verifySign(response, response.getString("sign_type"), response.getString("sign"), paymentConfig.getString("publicKey"));
        } catch (Exception e) {
            LOGGER.error("验签失败！", e);
            throw new PaymentException(PaymentStatus.VERIFY_SIGN_FAIL.getCode(), "{}", PaymentStatus.VERIFY_SIGN_FAIL.getMessage());
        }
    }

    /**
     * 获取请求公共参数
     *
     * @param paymentConfig 支付配置
     *
     * @return 公共参数
     */
    private JSONObject getCommonRequest(JSONObject paymentConfig) {
        return new JSONObject()
                .put("version", "2.0")
                .put("charset", "UTF-8")
                .put("sign_type", "RSA_1_256")
                // 商户号
                .put("mch_id", paymentConfig.optString("mchId"))
                // 终端出网ip(不允许上传172.192.10等开头的内网ip)
                .put("mch_create_ip", paymentConfig.optString("mchServiceIP"))
                // 随机字符串(长度不超过32位)
                .put("nonce_str", PaymentUtils.generateRandomString(32));
    }
}
