package com.af.v4.system.common.payment.handler.impl.ccb;

import com.af.v4.system.common.payment.annotation.PaymentHandlerConfig;
import com.af.v4.system.common.payment.dto.*;
import com.af.v4.system.common.payment.enums.BankName;
import com.af.v4.system.common.payment.enums.IntegrationType;
import com.af.v4.system.common.payment.enums.PaymentStatus;
import com.af.v4.system.common.payment.exceptions.PaymentException;
import com.af.v4.system.common.payment.handler.impl.AbstractPaymentHandler;
import com.af.v4.system.common.payment.utils.PaymentUtils;
import com.af.v4.system.common.plugins.http.RestTools;
import ccb.pay.api.util.CCBPayUtil;
import com.af.v4.system.common.plugins.json.JsonTools;
import org.json.JSONObject;
import org.json.JSONParserConfiguration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.io.*;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.net.SocketTimeoutException;
import java.util.Objects;

/**
 * @Author: 柳博
 * @Description: 中国建设银行新疆分行扫码支付（被扫）处理器
 * @Date: 2025-02-08 16:25
 */
@Component
@PaymentHandlerConfig(bankName = BankName.XJ_CCB, integrationType = IntegrationType.SCAN_DEVICE)
public class XjCcbAbstractPaymentHandlerImpl extends AbstractPaymentHandler {

    private static final Logger LOGGER = LoggerFactory.getLogger(XjCcbAbstractPaymentHandlerImpl.class);

    // 创建一个有序配置
    JSONParserConfiguration JSON_CONFIG = new JSONParserConfiguration().setOrdered(true);

    /**
     * 构建支付订单请求参数
     *
     * @param request       支付请求
     * @param paymentConfig 支付配置
     * @return 请求参数
     */
    @Override
    protected JSONObject buildPayOrderRequestParams(PaymentOrderRequest request, JSONObject paymentConfig) {
        LOGGER.info(">>> 构建扫码收款请求参数");
        try {
            return new JSONObject(JSON_CONFIG)
                    .put("TXCODE", "PAY100")
                    // 1：线上商户    2：线下商户
                    .put("MERFLAG", "1")
                    .put("ORDERID", request.getOrderNo())
                    .put("QRCODE", request.getAuthCode())
                    .put("AMOUNT", request.getAmount() / 100.00);
        } catch (Exception e) {
            LOGGER.error("构建支付订单请求参数异常", e);
            throw new PaymentException(PaymentStatus.PAY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.PAY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getMessage());
        }
    }

    /**
     * 订单响应结果处理
     *
     * @param response   支付响应
     * @param request    支付请求
     * @param resultJson 响应数据
     * @return 处理结果
     */
    @Override
    protected PaymentOrderResponse processPayOrderResponse(PaymentOrderResponse response, PaymentOrderRequest request, JSONObject resultJson, JSONObject paymentConfig) {
        LOGGER.info(">>> 处理二维码收款响应结果");

        String res = resultJson.optString("value");
        JSONObject result;

        try {
            result = new JSONObject(res);
            // Y：成功
            // N：失败
            // U：不确定（交易超时，建议通过订单查询确认扣款结果）
            // Q：待轮询（需按照[3.2.3.支付及轮询流程说明]继续调用轮询接口查询订单结果）
            // 其他无法明确订单结果的情况，均继续调用轮询接口查
            // 营收下单不判断状态，这里只要没有明确通知失败都按照成功处理
            if ("N".equals(result.getString("RESULT"))) {
                response.setState(PaymentStatus.PAY_FAIL);
                response.setMessage(result.optString("ERRMSG"));
            } else {
                // 因为银行返回参数不同，没有排序方式，无法验签
                response.setState(PaymentStatus.PAY_SUCCESS);
                response.setOrderNo(result.optString("TRACEID"));
                response.setAmount((int) Math.round(result.optDouble("AMOUNT") * 100.00));
                response.setQrCodeType(result.optString("QRCODETYPE"));
                response.setTransactionId(result.optString("ORDERID"));
                response.setAllowRefunds(paymentConfig.optString("allowRefunds", "0"));
                response.setAllowCancel(paymentConfig.optString("allowCancel", "0"));
                response.setNeedQuery(true);
                response.setPayResult(PaymentStatus.SUCCESS_FOR_PAYMENT.getMessage());
            }
        } catch (Exception e) {
            LOGGER.error("解析银行响应数据失败", e);
            response.setState(PaymentStatus.PAY_FAIL);
            response.setMessage("解析银行响应数据失败");
        }

        return response;
    }

    /**
     * 构建查询支付状态请求参数
     *
     * @param request       支付请求
     * @param paymentConfig 支付配置
     * @return 支付状态查询参数
     */
    @Override
    protected JSONObject buildQueryPaymentStatusRequestParams(QueryPaymentStatusRequest request, JSONObject paymentConfig) {
        LOGGER.info(">>> 构建查询支付状态请求参数");
        try {
            String queryTime = Objects.nonNull(request.getCustomParams()) ? request.getCustomParams().optString("qrytime", "1") : "1";
            return new JSONObject(JSON_CONFIG)
                    .put("TXCODE", "PAY101")
                    // 1：线上商户    2：线下商户
                    .put("MERFLAG", "1")
                    .put("ORDERID", request.getOrderNo())
                    .put("QRCODETYPE", request.getCustomParams().optString("qrCodeType", "1"))
                    .put("QRCODE", request.getCustomParams().optString("authCode", ""))
                    .put("QRYTIME", queryTime);
        } catch (Exception e) {
            LOGGER.error("构建查询订单支付状态请求参数异常", e);
            throw new PaymentException(PaymentStatus.QUERY_ORDER_BUILD_REQUEST_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.QUERY_ORDER_BUILD_REQUEST_PARAM_ERROR.getMessage());
        }
    }

    /**
     * 处理支付状态查询响应结果
     *
     * @param response   支付响应
     * @param request    请求参数
     * @param resultJson 响应数据
     * @return 处理结果
     */
    @Override
    protected QueryPaymentStatusResponse processQueryPaymentStatusResponse(QueryPaymentStatusResponse response, QueryPaymentStatusRequest request, JSONObject resultJson) {
        LOGGER.info(">>> 处理查询支付状态响应结果");
        String res = resultJson.optString("value");
        JSONObject result;

        try {
            result = new JSONObject(res);
            // Y：成功
            // N：失败
            // U：不确定（交易超时，建议通过订单查询确认扣款结果）
            // Q：待轮询（需按照[3.2.3.支付及轮询流程说明]继续调用轮询接口查询订单结果）
            // 其他无法明确订单结果的情况，均继续调用轮询接口查
            // 营收下单不判断状态，这里只要没有明确通知失败都按照成功处理
            String resCode = result.optString("RESULT", "N");
            if ("Y".equals(resCode)) {
                response.setState(PaymentStatus.SUCCESS_FOR_PAYMENT);
                response.setOrderNo(result.optString("ORDERID"));
                response.setAmount((int) Math.round(result.optDouble("AMOUNT") * 100.00));
                response.setTransactionId(result.optString("ORDERID"));
            } else if ("Q".equals(resCode) || "U".equals(resCode)) {
                response.setState(PaymentStatus.PAYMENT_IN_PROGRESS);
            } else {
                response.setState(PaymentStatus.FAIL_FOR_PAYMENT);
                response.setMessage(result.optString("ERRMSG"));
            }
        } catch (Exception e) {
            LOGGER.error("解析银行响应数据失败", e);
            response.setState(PaymentStatus.PAY_FAIL);
            response.setMessage("解析银行响应数据失败");
        }

        return response;
    }

    /**
     * 支付订单
     *
     * @param request       支付订单请求
     * @param paymentConfig 支付配置
     * @return 支付响应
     */
    public PaymentOrderResponse paymentOrder(PaymentOrderRequest request, JSONObject paymentConfig) {
        LOGGER.info(">>> 开始执行支付请求！");
        PaymentOrderResponse response = new PaymentOrderResponse();
        try {
            // 构建请求参数
            JSONObject reqParams = buildPayOrderRequestParams(request, paymentConfig);

            // 发起支付请求
            JSONObject payOrderRes = executePaymentRequest(reqParams, paymentConfig);

            // 响应结果处理
            processPayOrderResponse(response, request, payOrderRes, paymentConfig);

            return response;
        } catch (Exception e) {
            LOGGER.error("支付失败: 请求报文 = {}, 异常信息 = {}", request, e.getMessage());
            response.setCode(PaymentStatus.PAY_FAIL.getCode());
            response.setMessage(PaymentStatus.PAY_FAIL.getMessage());
        }
        return response;
    }

    /**
     * 查询订单支付状态
     *
     * @param request       查询订单支付状态请求参数
     * @param paymentConfig 支付配置
     * @return 查询订单支付状态响应
     */
    @Override
    public QueryPaymentStatusResponse queryPaymentStatus(QueryPaymentStatusRequest request, JSONObject paymentConfig) {
        LOGGER.info(">>> 开始执行查询订单支付状态请求！");
        QueryPaymentStatusResponse response = new QueryPaymentStatusResponse();
        try {
            // 构建请求参数
            JSONObject reqParams = buildQueryPaymentStatusRequestParams(request, paymentConfig);

            // 发起支付请求
            JSONObject queryPaymentRes = executePaymentRequest(reqParams, paymentConfig);

            // 处理响应结果
            processQueryPaymentStatusResponse(response, request, queryPaymentRes);
        } catch (Exception e) {
            LOGGER.error("查询订单支付状态失败: 请求报文 = {}, 异常信息 = {}", request, e.getMessage());
            response.setState(PaymentStatus.QUERY_ORDER_FAIL);
            response.setRequestJson(new JSONObject(request.toString()));
        }
        return response;
    }

    /**
     * 订单退款
     *
     * @param request       订单退款请求参数
     * @param paymentConfig 支付配置
     * @return 订单退款响应
     */
    @Override
    public RefundPaymentResponse refundPayment(RefundPaymentRequest request, JSONObject paymentConfig) {
        LOGGER.info(">>> 开始执行订单退款请求");
        RefundPaymentResponse response = new RefundPaymentResponse();
        try {
            // 构建请求参数
            JSONObject reqParams = buildRefundOrderRequestParams(request, paymentConfig);
            LOGGER.info(">>> 退款订单请求参数构建完成！");

            JSONObject queryPaymentRes;
            // 发起支付请求
            LOGGER.info(">>> 发起退款订单请求！");
            if (paymentConfig.optBoolean("isSocket", true)) {
                LOGGER.info(">>> 启动socket请求");
                queryPaymentRes = executePaymentRequestForSocket(reqParams, paymentConfig);
            } else {
                LOGGER.info(">>> 启动http请求");
                queryPaymentRes = executePaymentRequestForHttp(reqParams, paymentConfig);
            }
            LOGGER.info(">>> 退款订单请求完成！");

            // 处理响应结果
            processRefundOrderResponse(response, request, queryPaymentRes);
        } catch (Exception e) {
            LOGGER.error("订单退款失败: 请求报文={}, 异常信息={}", request, e.getMessage());
            response.setCode(PaymentStatus.QUERY_ORDER_FAIL.getCode());
            response.setMessage(PaymentStatus.QUERY_ORDER_FAIL.getMessage());
            response.setRequestJson(new JSONObject(request.toString()));
        }
        return response;
    }


    /**
     * 执行支付业务请求
     *
     * @param reqParams     请求参数
     * @param paymentConfig 支付配置
     * @return 业务执行请求结果
     */
    @Override
    protected JSONObject executePaymentRequest(JSONObject reqParams, JSONObject paymentConfig) {
        try {
            LOGGER.info("请求支付业务, 请求参数：{}", reqParams);

            String host = paymentConfig.getString("transactionurl");
            String pubKey = paymentConfig.getString("publicKey");
            String param = convertJsonToQueryString(reqParams);
            LOGGER.info(">>> 参数信息：{}", param);
            String merInfo = convertJsonToQueryString(getCommonRequest(paymentConfig));
            LOGGER.info(">>> 商户信息：{}", merInfo);
            LOGGER.info(">>> 公钥：{}", pubKey);

            CCBPayUtil ccbPayUtil = new CCBPayUtil();
            String sign = ccbPayUtil.makeCCBParam(merInfo + "&" + param, pubKey);
            String url = host + "?" + merInfo + "&ccbParam=" + sign;

            LOGGER.info(">>> 发送至银行：{}", url);
            String notifyURLParam = RestTools.post(url, new JSONObject());
            LOGGER.info(">>> 银行响应结果：{}", notifyURLParam);

            return new JSONObject().put("value", notifyURLParam);
        } catch (Exception e) {
            LOGGER.error("请求支付业务接口异常", e);
            throw new PaymentException(
                    PaymentStatus.PAY_ORDER_ERROR.getCode(),
                    reqParams.toString(),
                    PaymentStatus.PAY_ORDER_ERROR.getMessage()
            );
        }
    }

    private JSONObject executePaymentRequestForHttp(JSONObject reqParams, JSONObject paymentConfig) {
        String host = paymentConfig.optString("host");
        int port = paymentConfig.optInt("port", -1);
        // 连接和读取的超时时间，默认为 6000 * 10 毫秒 (1分钟)
        int timeout = paymentConfig.optInt("timeout", 6000 * 10);

        // 检查必要的配置信息
        if (host == null || host.trim().isEmpty() || port == -1) {
            throw new IllegalArgumentException("Http 请求配置不完整，请检查配置信息中的 host 和 port。");
        }

        String reqStr = "requestXml=" + JsonTools.jsonConvertToXml(reqParams, "TX", true, paymentConfig.optString("charset", "UTF-8"));
        String url = "http://" + host + ":" + port;
        JSONObject headers = new JSONObject()
                .put("Content-Type", "application/xml");
        LOGGER.info(">>> Http 请求地址: {}", url);
        LOGGER.info(">>> Http 请求报文: {}", reqStr);
        String resp = RestTools.post(
                url,
                reqStr.replaceAll(">\\s+<", "><")  // 去掉标签间的换行和空格
                        .replaceAll("[\\n\\r\\t]", "") // 去掉换行、回车、制表符
                        .trim(),
                headers.toString()
        );
        LOGGER.info(">>> Http 请求响应报文: {}", resp);

        return JsonTools.xmlConvertToJson(resp, paymentConfig.optString("charset", "UTF-8"));
    }

    /**
     * 通过 Socket 发送支付请求并接收响应。
     *
     * @param reqParams     包含请求参数的 JSONObject。
     * @param paymentConfig 包含 Socket 连接配置的 JSONObject (如 host, port, timeout)。
     * @return 包含响应数据的 JSONObject，或包含错误信息的 JSONObject。
     */
    private JSONObject executePaymentRequestForSocket(JSONObject reqParams, JSONObject paymentConfig) {
        String host = paymentConfig.optString("host");
        int port = paymentConfig.optInt("port", -1);
        // 连接和读取的超时时间，默认为 6000 * 10 毫秒 (1分钟)
        int timeout = paymentConfig.optInt("timeout", 6000 * 10);

        // 检查必要的配置信息
        if (host == null || host.trim().isEmpty() || port == -1) {
            throw new IllegalArgumentException("Socket 连接配置不完整，请检查配置信息中的 socketHost 和 socketPort。");
        }

        // 使用 try-with-resources 确保 Socket 和流能够被正确关闭
        try (Socket socket = new Socket()) {

            LOGGER.info(">>> 连接到 Socket 服务: {}:{}", host, port);
            // 连接到服务器，设置连接超时
            socket.connect(new InetSocketAddress(host, port), timeout);

            // 设置读取数据的超时时间
            socket.setSoTimeout(timeout);

            // 获取输入/输出流并包装，便于读写文本
            // 为 PrintWriter 开启自动刷新
            try (OutputStream outputStream = socket.getOutputStream();
                 InputStream inputStream = socket.getInputStream();
                 PrintWriter writer = new PrintWriter(new OutputStreamWriter(outputStream, paymentConfig.optString("charset", "UTF-8")), true); // autoFlush = true
                 BufferedReader reader = new BufferedReader(new InputStreamReader(inputStream, paymentConfig.optString("charset", "UTF-8")))) {

                //  准备并发送请求数据
                String requestString = JsonTools.jsonConvertToXml(reqParams, "TX", true, paymentConfig.optString("charset", "UTF-8"));
                LOGGER.info(">>> 请求报文: {}", requestString);
                writer.println(requestString);

                // 读取服务器响应
                StringBuilder responseBuilder = new StringBuilder();

                boolean receivedResponse = false;

                String line;
                while ((line = reader.readLine()) != null) {
                    receivedResponse = true;
                    responseBuilder.append(line).append(System.lineSeparator());
                    // 表示响应已结束，跳出循环
                    if (line.contains("</TX>")) {
                        break;
                    }
                }

                // 如果根本没读到任何响应
                if (!receivedResponse) {
                    throw new IOException("未收到任何响应，服务器可能未返回数据");
                }

                String responseString = responseBuilder.toString();

                // 将响应字符串解析为 JSONObject
                if (responseString.trim().isEmpty()) {
                    // 检查接收到的字符串是否为空白
                    throw new IOException("从服务器接收了空响应字符串");
                }

                LOGGER.info(">>> Socket 响应报文: {}", responseString);
                return JsonTools.xmlConvertToJson(responseString, paymentConfig.optString("charset", "UTF-8"));
            }

        } catch (SocketTimeoutException e) {
            // 捕获连接或读取超时异常
            LOGGER.error("Socket 操作超时: {}", e.getMessage());
        } catch (IOException e) {
            // 捕获其他 IO 异常 (连接拒绝、连接重置、读写错误等)
            LOGGER.error("Socket 通信错误: {}", e.getMessage());
        } catch (Exception e) {
            // 捕获任何其他未预料的异常
            LOGGER.error("发生未知错误: {}", e.getMessage());
        }

        return null;
    }

    /**
     * 获取请求公共参数
     *
     * @param paymentConfig 支付配置
     * @return 公共参数
     */
    private JSONObject getCommonRequest(JSONObject paymentConfig) {
        return new JSONObject(JSON_CONFIG)
                .put("MERCHANTID", paymentConfig.optString("mchId"))
                .put("POSID", paymentConfig.optString("posId"))
                .put("BRANCHID", paymentConfig.optString("branchId"));
    }

    /**
     * 将JSON对象转换为查询字符串
     *
     * @param jsonObject 要转换的JSON对象
     * @return 转换后的查询字符串
     */
    private String convertJsonToQueryString(JSONObject jsonObject) {
        StringBuilder queryString = new StringBuilder();
        boolean first = true;

        for (String key : jsonObject.keySet()) {
            if (!first) {
                queryString.append("&");
            } else {
                first = false;
            }
            queryString.append(key.toUpperCase())
                    .append("=")
                    .append(jsonObject.opt(key));
        }

        return queryString.toString();
    }

    /**
     * 构建撤销订单请求参数(不支持)
     *
     * @param request       支付请求
     * @param paymentConfig 支付配置
     * @return 订单撤销请求参数
     */
    @Override
    protected JSONObject buildCancelOrderRequestParams(CancelPaymentRequest request, JSONObject paymentConfig) {
        return null;
    }

    /**
     * 处理撤销订单响应结果(不支持)
     *
     * @param response   撤销响应
     * @param request    请求数据
     * @param resultJson 响应数据
     * @return 处理结果
     */
    @Override
    protected CancelPaymentResponse processCancelOrderResponse(CancelPaymentResponse response, CancelPaymentRequest request, JSONObject resultJson) {
        return null;
    }

    /**
     * 构建退款订单请求参数(需要在商户平台手动操作退款)
     *
     * @param request       支付请求
     * @param paymentConfig 支付配置
     * @return 请求参数
     */
    @Override
    protected JSONObject buildRefundOrderRequestParams(RefundPaymentRequest request, JSONObject paymentConfig) {
        LOGGER.info(">>> 构建退款订单请求参数");
        try {
            JSONObject body = new JSONObject()
                    .put("MONEY", request.getRefundAmount() / 100.00)
                    .put("ORDER", request.getTransactionId())
                    .put("REFUND_CODE", "");
            return new JSONObject()
                    .put("REQUEST_SN", PaymentUtils.generateRandomString(16, PaymentUtils.Mode.DIGITS_ONLY))
                    .put("CUST_ID", paymentConfig.optString("custId"))
                    .put("USER_ID", paymentConfig.optString("userId"))
                    .put("PASSWORD", paymentConfig.optString("password"))
                    .put("TX_CODE", "5W1004")
                    .put("LANGUAGE", "CN")
                    .put("SIGN_INFO", "")
                    .put("SIGNCERT", "")
                    .put("TX_INFO", body);
        } catch (Exception e) {
            LOGGER.error("构建退款订单请求参数异常", e);
            throw new PaymentException(PaymentStatus.REFUND_BUILD_REQUEST_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.REFUND_BUILD_REQUEST_PARAM_ERROR.getMessage());
        }
    }

    /**
     * 处理退款订单响应结果(需要在商户平台手动操作退款)
     *
     * @param response   支付请求
     * @param request    请求参数
     * @param resultJson 响应参数
     * @return 处理结果
     */
    @Override
    protected RefundPaymentResponse processRefundOrderResponse(RefundPaymentResponse response, RefundPaymentRequest request, JSONObject resultJson) {
        LOGGER.info(">>> 处理退款订单响应结果 --> ");
        try {
            if ("000000".equals(resultJson.optString("RETURN_CODE"))) {
                JSONObject txInfo = resultJson.optJSONObject("TX_INFO");
                response.setMchId(resultJson.optString("CUST_ID"));
                response.setOrderNo(txInfo.optString("ORDER_NUM"));
                response.setTransactionId(txInfo.optString("ORDER_NUM"));
                response.setRefundOrderNo(txInfo.optString("ORDER_NUM"));
                response.setRefundAmount(resultJson.optInt("refund_fee"));

                response.setRefundResult(PaymentStatus.REFUND_REQUEST_SUCCESS.getMessage());
                response.setState(PaymentStatus.REFUND_REQUEST_SUCCESS);
            } else {
                response.setState(PaymentStatus.REFUND_REQUEST_FAIL);
                response.setMessage(resultJson.optString("RETURN_MSG", PaymentStatus.REFUND_REQUEST_FAIL.getMessage()));
            }
        } catch (Exception e) {
            LOGGER.error(">>> 处理订单退款返回结果异常", e);
            throw new PaymentException(PaymentStatus.REFUND_BUILD_RESPONSE_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.REFUND_BUILD_RESPONSE_PARAM_ERROR.getMessage());
        }

        return response;
    }

    /**
     * 构建退款状态查询请求参数(需要在商户平台手动操作退款)
     *
     * @param request       支付请求
     * @param paymentConfig 支付配置
     * @return 请求参数
     */
    @Override
    protected JSONObject buildQueryRefundStatusRequestParams(QueryRefundStatusRequest request, JSONObject paymentConfig) {
        return null;
    }

    /**
     * 处理退款状态查询响应结果
     *
     * @param response   支付请求
     * @param request    请求参数
     * @param resultJson 响应数据
     * @return 处理结果
     */
    @Override
    protected QueryRefundStatusResponse processQueryRefundStatusResponse(QueryRefundStatusResponse response, QueryRefundStatusRequest request, JSONObject resultJson) {
        return null;
    }

    /**
     * 签名请求
     *
     * @param reqParams     请求参数
     * @param paymentConfig 支付配置
     */
    @Override
    protected void signRequest(JSONObject reqParams, JSONObject paymentConfig) {
    }

    /**
     * 响应结果签名验证
     *
     * @param response      响应结果
     * @param paymentConfig 支付配置
     * @return 验签是否通过
     */
    @Override
    protected boolean verifyResponseSign(JSONObject response, JSONObject paymentConfig) {
        return true;
    }
}
