package com.af.v4.system.common.payment.utils;

import com.af.v4.system.common.payment.exceptions.PaymentException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.charset.StandardCharsets;
import org.apache.commons.codec.binary.Base64;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * 用于生成签名、验证签名及参数处理等
 *
 * @author llzh
 * @DateTime 2024/12/26 15:10
 */
public class SignUtils {

    private static final Logger LOGGER = LoggerFactory.getLogger(SignUtils.class);

    private static final String SIGN_TYPE_RSA = "RSA_1_256";


    /**
     * 生成签名并添加到请求参数中
     *
     * @param reqParams  请求参数集合
     * @param signType   签名类型 可选  不传的默认为 RSA_1_256
     * @param signKey    签名字段 可选  不传的默认为 sign
     * @param privateKey 密钥
     * @throws Exception 如果签名生成失败
     */
    public static void signRequestParams(JSONObject reqParams, String signType, String signKey, String privateKey) throws Exception {
        try {
            Map<String, String> filteredParams = filterParams(reqParams.toMap());
            String preStr = buildSignString(filteredParams);
            LOGGER. info("待签名字符串: {}", preStr);
            // 使用传入的签名类型或默认签名类型
            signType = Optional.ofNullable(signType).orElse(SIGN_TYPE_RSA);
            String sign = generateSign(signType, preStr, privateKey);
            reqParams.put(signKey, sign);
            LOGGER. info("签名生成成功: signType={}, sign={}", signType, sign);
            LOGGER. info("加签后的请求参数: reqParams={}", reqParams);
        } catch (Exception e) {
            LOGGER.error(" 签名生成失败: {}", e.getMessage(), e);
        }
    }

    /**
     * 生成签名并添加到请求参数中，签名类型默认为RSA_1_256
     *
     * @param reqParams  请求参数集合
     * @param privateKey 密钥
     * @throws Exception 如果签名生成失败
     */
    public static void signRequestParams(JSONObject reqParams, String privateKey) throws Exception {
        signRequestParams(reqParams, "RSA_1_256", "sign", privateKey);
    }

    /**
     * 生成签名
     *
     * @param signType   签名类型（目前仅支持 RSA）
     * @param preStr     待签名字符串
     * @param privateKey 私钥字符串（Base64 编码）
     * @return 签名结果（Base64 编码）
     * @throws Exception 如果签名失败或签名类型不支持
     */
    private static String generateSign(String signType, String preStr, String privateKey) {
        if (SIGN_TYPE_RSA.equalsIgnoreCase(signType)) {
            try {
                // 使用 RSA 签名并返回 Base64 编码结果
                byte[] signature = RSAUtils.sign(RSAUtils.SignatureSuite.SHA256, preStr.getBytes(StandardCharsets.UTF_8), privateKey);
                return new String(Base64.encodeBase64(signature), StandardCharsets.UTF_8);
            } catch (Exception e) {
                LOGGER.error("签名生成失败: {}", e.getMessage(), e);
            }
        } else {
            // 对于不支持的签名类型抛出明确异常
            throw new PaymentException("不支持的签名方式: " + signType);
        }
        return null;
    }
    public static boolean verifySign(JSONObject response, String signType, String sign, String publicKey) {
        if("RSA_1_256".equals(signType)) {
            Map<String, String> Reparams = filterParams(response.toMap());
            String repreStr = buildSignString(Reparams);
            LOGGER. info("待验签字符串: {}", repreStr);
            return RSAUtils.verifySign(RSAUtils.SignatureSuite.SHA256, repreStr, sign, publicKey);
        }
        return false;
    }


    /**
     * 过滤无效字段（移除 sign 字段和空值字段）
     *
     * @param params 参数 Map
     * @return 过滤后的 Map
     */
    public static Map<String, String> filterParams(Map<String, Object> params) {
        return params.entrySet().stream()
                .filter(entry -> !"sign".equals(entry.getKey()) && entry.getValue() != null && !entry.getValue().toString().isEmpty())
                .collect(Collectors.toMap(Map.Entry::getKey, entry -> entry.getValue().toString()));
    }


    /**
     * 按字段名排序并拼接成签名原始串
     *
     * @param params 过滤后的参数 Map
     * @return 签名原始串
     */
    public static String buildSignString(Map<String, String> params) {
        return params.entrySet().stream().sorted(Map.Entry.comparingByKey())
                .map(entry -> entry.getKey() + "=" + entry.getValue())
                .collect(Collectors.joining("&"));
    }




}
