package com.af.v4.system.common.payment.handler.impl.ceb;

import com.af.v4.system.common.payment.annotation.PaymentHandlerConfig;
import com.af.v4.system.common.payment.dto.*;
import com.af.v4.system.common.payment.enums.BankName;
import com.af.v4.system.common.payment.enums.IntegrationType;
import com.af.v4.system.common.payment.enums.PaymentStatus;
import com.af.v4.system.common.payment.exceptions.PaymentException;
import com.af.v4.system.common.payment.handler.impl.AbstractPaymentHandler;
import com.af.v4.system.common.payment.utils.SignUtils;
import com.af.v4.system.common.plugins.date.DateTools;
import com.af.v4.system.common.plugins.http.RestTools;
import com.af.v4.system.common.plugins.json.JsonTools;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.util.Date;

/**
 * @ClassName CEBAggregatePaymentHandler
 * @Description 光大银行扫码设备处理器
 * @Author llz
 * @Date 2024/12/26 22:12
 */
@Component
@PaymentHandlerConfig(bankName = BankName.CEB, integrationType = IntegrationType.SCAN_DEVICE)
public class CEBAggregatePaymentHandler extends AbstractPaymentHandler {
    private static final Logger LOGGER = LoggerFactory.getLogger(CEBAggregatePaymentHandler.class);


    /**
     * 构建支付订单请求参数
     * 由子类实现，提供具体的支付参数
     *
     * @param request       支付请求
     * @param paymentConfig 支付配置
     * @return 请求参数
     */
    @Override
    protected JSONObject buildPayOrderRequestParams(PaymentOrderRequest request, JSONObject paymentConfig) {
        try {
            JSONObject reqParams = getRequestHeard(paymentConfig);
            reqParams.put("service", "unified.trade.micropay");
            reqParams.put("out_trade_no", request.getOrderNo());
            reqParams.put("body", request.getDescription());
            reqParams.put("total_fee", request.getAmount());
            reqParams.put("mch_create_ip", paymentConfig.getString("mchServiceIP"));
            reqParams.put("auth_code", request.getAuthCode());
            reqParams.put("version", "2.0");

            JSONObject terminalInfo = new JSONObject();
            terminalInfo.put("terminal_type", "11");
            // terminalInfo.put("terminal_id", "**");
            terminalInfo.put("app_version", "1.000000");
            reqParams.put("terminal_info", terminalInfo.toString());
            return reqParams;

        } catch (Exception e) {
            LOGGER.error("构建支付订单请求参数异常", e);
            throw new PaymentException(PaymentStatus.PAY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.PAY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getMessage());
        }
    }

    /**
     * 处理支付响应结果
     * 由子类实现
     *
     * @param response   支付响应
     * @param request    支付请求
     * @param resultJson 响应数据
     */
    @Override
    protected PaymentOrderResponse processPayOrderResponse(PaymentOrderResponse response, PaymentOrderRequest request, JSONObject resultJson) {
        LOGGER.info("开始构建支付订单返回参数!");
        try {
            //0表示通信成功，非0表示失败此字段是通信标识，非交易标识，交易是否成功需要查看 result_code 来判断
            boolean neddQuery = false;
            String payState = "未支付";
            if ("0".equals(resultJson.optString("status"))) {
                response.setCode(PaymentStatus.PAY_SUCCESS.getCode());
                response.setMessage(PaymentStatus.PAY_SUCCESS.getMessage());
                response.setOrderNo(resultJson.optString("out_trade_no"));
                response.setMchId(resultJson.optString("mch_id"));
                response.setTransactionId(resultJson.optString("transaction_id"));
                if (resultJson.has("total_fee")) {
                    response.setAmount(resultJson.optInt("total_fee"));
                }
                //交易是否成功需要查看 result_code 来判断 0表示成功，非0表示失败
                if ("0".equals(resultJson.optString("result_code"))) {
                    if ("0".equals(resultJson.optString("pay_result"))) {
                        payState = "支付成功";
                    } else if (!"0".equals(resultJson.optString("pay_result")) && "Y".equals(resultJson.optString("need_query"))) {
                        neddQuery = true;
                    }

                    if (resultJson.has("time_end")) {
                        response.setPaySuccessDate(DateTools.formatDateTime(resultJson.optString("time_end")));
                    }
                } else if ("1".equals(resultJson.optString("result_code")) && "04021_1211037".equals(resultJson.optString("code"))) {
                    neddQuery = true;
                    response.setPaySuccessDate(DateTools.formatDateTime(resultJson.optString("time_end")));
                }
                response.setNeedQuery(neddQuery);
                response.setPayResult(payState);
            } else {
                LOGGER.info("本次支付订单交易通信失败！");
                response.setCode(PaymentStatus.COMMUNICATION_FAIL.getCode());
                response.setMessage(PaymentStatus.COMMUNICATION_FAIL.getMessage());
            }
        } catch (Exception e) {
            LOGGER.error("构建支付订单返回参数异常", e);
            throw new PaymentException(PaymentStatus.PAY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.PAY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getMessage());
        }
        return response;
    }

    /**
     * 构建查询订单支付状态请求参数
     * 由子类实现，提供具体的支付参数
     *
     * @param request       支付请求
     * @param paymentConfig 支付配置
     * @return 请求参数
     */
    @Override
    protected JSONObject buildQueryPaymentStatusRequestParams(QueryPaymentStatusRequest request, JSONObject paymentConfig) {
        try {
            JSONObject reqParams = getRequestHeard(paymentConfig);
            reqParams.put("service", "unified.trade.query");
            reqParams.put("out_trade_no", request.getOrderNo());
            reqParams.put("transaction_id", request.getTransactionId());
            return reqParams;
        } catch (Exception e) {
            LOGGER.error("构建查询订单支付状态请求参数异常", e);
            throw new PaymentException(PaymentStatus.QUERY_ORDER_BUILD_REQUEST_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.QUERY_ORDER_BUILD_REQUEST_PARAM_ERROR.getMessage());
        }
    }

    /**
     * 处理查询支付状态响应结果
     * 由子类实现
     *
     * @param response   支付响应
     * @param request
     * @param resultJson 响应数据
     */
    @Override
    protected QueryPaymentStatusResponse processQueryPaymentStatusResponse(QueryPaymentStatusResponse response, QueryPaymentStatusRequest request, JSONObject resultJson) {
        LOGGER.info("开始构建查询支付状态返回参数!");
        try {
            //0表示通信成功，非0表示失败此字段是通信标识，非交易标识，交易是否成功需要查看 trade_state 来判断
            if ("0".equals(resultJson.optString("status"))) {
                //业务结果 交易是否成功需要查看 trade_state 来判断 0表示成功，非0表示失败
                if ("0".equals(resultJson.optString("result_code"))) {
                    response.setCode(PaymentStatus.PAY_SUCCESS.getCode());
                    response.setMessage(PaymentStatus.PAY_SUCCESS.getMessage());
                    response.setMchId(resultJson.optString("mch_id"));
                    response.setOrderNo(resultJson.optString("out_trade_no"));
                    response.setTransactionId(resultJson.optString("transaction_id"));
                    response.setAmount(resultJson.optInt("total_fee"));
                    if ("SUCCESS".equals(resultJson.optString("trade_state"))) {
                        response.setPaymentStatus(PaymentStatus.SUCCESS_FOR_PAYMENT.getMessage());
                        response.setCode(PaymentStatus.SUCCESS_FOR_PAYMENT.getCode());
                        response.setMessage(PaymentStatus.SUCCESS_FOR_PAYMENT.getMessage());
                    } else if ("REFUND".equals(resultJson.optString("trade_state"))) {
                        response.setPaymentStatus(PaymentStatus.REFUND_FOR_PAYMENT.getMessage());
                        response.setCode(PaymentStatus.REFUND_FOR_PAYMENT.getCode());
                        response.setMessage(PaymentStatus.REFUND_FOR_PAYMENT.getMessage());
                    } else if ("NOTPAY".equals(resultJson.optString("trade_state"))) {
                        response.setPaymentStatus(PaymentStatus.NOT_FOR_PAYMENT.getMessage());
                        response.setCode(PaymentStatus.NOT_FOR_PAYMENT.getCode());
                        response.setMessage(PaymentStatus.NOT_FOR_PAYMENT.getMessage());
                    } else if ("REVOKED".equals(resultJson.optString("trade_state")) || "REVERSE".equals(resultJson.optString("trade_state"))) {
                        response.setPaymentStatus(PaymentStatus.CANCEL_FOR_PAYMENT.getMessage());
                        response.setCode(PaymentStatus.CANCEL_FOR_PAYMENT.getCode());
                        response.setMessage(PaymentStatus.CANCEL_FOR_PAYMENT.getMessage());
                    } else if ("USERPAYING".equals(resultJson.optString("trade_state"))) {
                        response.setPaymentStatus(PaymentStatus.PAYMENT_IN_PROGRESS.getMessage());
                        response.setCode(PaymentStatus.PAYMENT_IN_PROGRESS.getCode());
                        response.setMessage(PaymentStatus.PAYMENT_IN_PROGRESS.getMessage());
                    } else if ("PAYERROR".equals(resultJson.optString("trade_state"))) {
                        response.setPaymentStatus(PaymentStatus.FAIL_FOR_PAYMENT.getMessage());
                        response.setCode(PaymentStatus.FAIL_FOR_PAYMENT.getCode());
                        response.setMessage(PaymentStatus.FAIL_FOR_PAYMENT.getMessage());
                    }
                    if(!"".equals(resultJson.optString("time_end"))){
                        response.setPaySuccessDate(DateTools.formatDateTime(resultJson.optString("time_end")));
                    } else {
                        response.setPaySuccessDate("");
                    }
                } else {
                    response.setCode(PaymentStatus.PAY_FAIL.getCode());
                    response.setMessage(PaymentStatus.PAY_FAIL.getMessage());
                }
            } else {
                LOGGER.info("本次查询支付状态交易通信失败！");
                response.setCode(PaymentStatus.COMMUNICATION_FAIL.getCode());
                response.setMessage(PaymentStatus.COMMUNICATION_FAIL.getMessage());
            }
        } catch (Exception e) {
            LOGGER.error("构建查询支付状态返回参数异常", e);
            throw new PaymentException(PaymentStatus.QUERY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.QUERY_ORDER_BUILD_RESPONSE_PARAM_ERROR.getMessage());
        }
        return response;
    }

    /**
     * 构建订单撤销请求参数
     * 由子类实现，提供具体的支付参数
     *
     * @param request       支付请求
     * @param paymentConfig 支付配置
     * @return 请求参数
     */
    @Override
    protected JSONObject buildCancelOrderRequestParams(CancelPaymentRequest request, JSONObject paymentConfig) {
        try {
            JSONObject reqParams = getRequestHeard(paymentConfig);
            reqParams.put("service", "unified.micropay.reverse");
            reqParams.put("out_trade_no", request.getOrderNo());
            return reqParams;
        } catch (Exception e) {
            LOGGER.error("构建订单撤销请求参数异常", e);
            throw new PaymentException(PaymentStatus.CANCEL_ORDER_BUILD_REQUEST_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.CANCEL_ORDER_BUILD_REQUEST_PARAM_ERROR.getMessage());
        }
    }

    /**
     * 处理订单撤销响应结果
     *
     * @param response   支付请求
     * @param request
     * @param resultJson 响应参数
     * @return 请求参数
     */
    @Override
    protected CancelPaymentResponse processCancelOrderResponse(CancelPaymentResponse response, CancelPaymentRequest request, JSONObject resultJson) {
        LOGGER.info("开始构建撤销订单返回参数!");
        try {
            //0表示通信成功，非0表示失败此字段是通信标识，非交易标识，交易是否成功需要查看 result_code 来判断
            if ("0".equals(resultJson.optString("status"))) {
                //交易是否成功需要查看 result_code 来判断 0表示成功，非0表示失败
                if ("0".equals(resultJson.optString("result_code"))) {
                    response.setCode(PaymentStatus.CANCEL_ORDER_SUCCESS.getCode());
                    response.setMessage(PaymentStatus.CANCEL_ORDER_SUCCESS.getMessage());
                    response.setMchId(resultJson.optString("mch_id"));
                    response.setOrderNo(resultJson.optString("out_trade_no"));
                    response.setCancelResult(resultJson.optString("transaction_id"));
                    String cancelState = "撤销成功";
                    if (!"0".equals(resultJson.optString("result_code"))) {
                        cancelState = "撤销失败";
                        response.setCode(PaymentStatus.CANCEL_ORDER_FAIL.getCode());
                        response.setMessage(PaymentStatus.CANCEL_ORDER_FAIL.getMessage());
                    }
                    response.setCancelResult(cancelState);
                } else {
                    response.setCode(PaymentStatus.PAY_FAIL.getCode());
                    response.setMessage(PaymentStatus.PAY_FAIL.getMessage());
                }
            } else {
                LOGGER.info("本次撤销订单交易通信失败！");
                response.setCode(PaymentStatus.COMMUNICATION_FAIL.getCode());
                response.setMessage(PaymentStatus.COMMUNICATION_FAIL.getMessage());
            }
        } catch (Exception e) {
            LOGGER.error("构建撤销订单返回参数异常", e);
            throw new PaymentException(PaymentStatus.CANCEL_ORDER_BUILD_RESPONSE_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.CANCEL_ORDER_BUILD_RESPONSE_PARAM_ERROR.getMessage());
        }
        return response;
    }

    /**
     * 构建退款请求参数
     * 由子类实现，提供具体的支付参数
     *
     * @param request       支付请求
     * @param paymentConfig 支付配置
     * @return 请求参数
     */
    @Override
    protected JSONObject buildRefundOrderRequestParams(RefundPaymentRequest request, JSONObject paymentConfig) {
        try {
            JSONObject reqParams = getRequestHeard(paymentConfig);
            reqParams.put("service", "unified.trade.refund");
            reqParams.put("out_trade_no", request.getOrderNo());
            reqParams.put("out_refund_no", request.getRefundOrderNo());
            reqParams.put("total_fee", request.getAmount());
            reqParams.put("refund_fee", request.getRefundAmount());
            reqParams.put("op_user_id", reqParams.getString("mch_id"));
            return reqParams;
        } catch (Exception e) {
            LOGGER.error("构建订单撤销请求参数异常", e);
            throw new PaymentException(PaymentStatus.REFUND_BUILD_REQUEST_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.REFUND_BUILD_REQUEST_PARAM_ERROR.getMessage());
        }
    }

    /**
     * 构建退款返回参数
     *
     * @param response   支付请求
     * @param request    请求参数
     * @param resultJson 退款响应
     */
    @Override
    protected RefundPaymentResponse processRefundOrderResponse(RefundPaymentResponse response, RefundPaymentRequest request, JSONObject resultJson) {
        LOGGER.info("开始构建订单退款返回参数!");
        try {
            //0表示通信成功，非0表示失败此字段是通信标识，非交易标识，交易是否成功需要查看 result_code 来判断
            if ("0".equals(resultJson.optString("status"))) {
                //交易是否成功需要查看 result_code 来判断 0表示成功，非0表示失败
                if ("0".equals(resultJson.optString("result_code"))) {
                    response.setCode(PaymentStatus.REFUND_SUCCESS.getCode());
                    response.setMessage(PaymentStatus.REFUND_SUCCESS.getMessage());
                    response.setMchId(resultJson.optString("mch_id"));
                    response.setOrderNo(resultJson.optString("out_trade_no"));
                    response.setTransactionId(resultJson.optString("transaction_id"));
                    response.setRefundOrderNo(resultJson.optString("out_refund_no_$0"));
                    response.setRefundId(resultJson.optString("out_refund_id_$0"));
                    response.setRefundAmount(resultJson.optInt("refund_fee_$0"));
                    if ("SUCCESS".equals(resultJson.optString("refund_status_$n0"))) {
                        response.setRefundResult(PaymentStatus.REFUND_SUCCESS.getMessage());
                        response.setCode(PaymentStatus.REFUND_SUCCESS.getCode());
                        response.setMessage(PaymentStatus.REFUND_SUCCESS.getMessage());
                    } else if ("FAIL".equals(resultJson.optString("refund_status_$n0"))) {
                        response.setRefundResult(PaymentStatus.REFUND_FAIL.getMessage());
                        response.setCode(PaymentStatus.REFUND_FAIL.getCode());
                        response.setMessage(PaymentStatus.REFUND_FAIL.getMessage());
                    } else if ("PROCESSING".equals(resultJson.optString("refund_status_$n0"))) {
                        response.setRefundResult(PaymentStatus.REFUND_PROCESSING.getMessage());
                        response.setCode(PaymentStatus.REFUND_PROCESSING.getCode());
                        response.setMessage(PaymentStatus.REFUND_PROCESSING.getMessage());
                    } else if ("CHANGE".equals(resultJson.optString("refund_status_$n0"))) {
                        response.setRefundResult(PaymentStatus.REFUND_CHANGE.getMessage());
                        response.setCode(PaymentStatus.REFUND_CHANGE.getCode());
                        response.setMessage(PaymentStatus.REFUND_CHANGE.getMessage());
                    }
                } else {
                    response.setCode(PaymentStatus.PAY_FAIL.getCode());
                    response.setMessage(PaymentStatus.PAY_FAIL.getMessage());
                }
            } else {
                LOGGER.info("本次订单退款交易通信失败！");
                response.setCode(PaymentStatus.COMMUNICATION_FAIL.getCode());
                response.setMessage(PaymentStatus.COMMUNICATION_FAIL.getMessage());
            }
        } catch (Exception e) {
            LOGGER.error("构建订单退款返回参数异常", e);
            throw new PaymentException(PaymentStatus.REFUND_BUILD_RESPONSE_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.REFUND_BUILD_RESPONSE_PARAM_ERROR.getMessage());
        }
        return response;
    }

    /**
     * 构建查询退款状态请求参数
     *
     * @param request       支付请求
     * @param paymentConfig 支付配置
     * @return 请求参数
     */
    @Override
    protected JSONObject buildQueryRefundStatusRequestParams(QueryRefundStatusRequest request, JSONObject paymentConfig) {
        try {
            JSONObject reqParams = getRequestHeard(paymentConfig);
            reqParams.put("service", "unified.trade.refundquery");
            reqParams.put("out_trade_no", request.getOrderNo());
            reqParams.put("refund_id", request.getRefundId());
            return reqParams;
        } catch (Exception e) {
            LOGGER.error("构建查询退款请求参数异常", e);
            throw new PaymentException(PaymentStatus.QUERY_REFUND_BUILD_REQUEST_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.REFUND_BUILD_REQUEST_PARAM_ERROR.getMessage());
        }
    }

    /**
     * 构建查询退款状态请求参数
     *
     * @param response   支付请求
     * @param request    请求参数
     * @param resultJson 查询退款状态响应
     * @return 请求参数
     */
    @Override
    protected QueryRefundStatusResponse processQueryRefundStatusResponse(QueryRefundStatusResponse response, QueryRefundStatusRequest request, JSONObject resultJson) {
        LOGGER.info("开始构建查询退款状态返回参数!");
        try {
            //0表示通信成功，非0表示失败此字段是通信标识，非交易标识，交易是否成功需要查看 result_code 来判断
            if ("0".equals(resultJson.optString("status"))) {
                //交易是否成功需要查看 result_code 来判断 0表示成功，非0表示失败
                if ("0".equals(resultJson.optString("result_code"))) {
                    response.setCode(PaymentStatus.REFUND_SUCCESS.getCode());
                    response.setMessage(PaymentStatus.REFUND_SUCCESS.getMessage());
                    response.setMchId(resultJson.optString("mch_id"));
                    response.setOrderNo(resultJson.optString("out_trade_no"));
                    response.setRefundId(resultJson.optString("out_refund_id_$0"));
                    response.setTransactionId(resultJson.optString("transaction_id"));
                    response.setRefundOrderNo(resultJson.optString("out_refund_no_$0"));
                    response.setRefundAmount(resultJson.optInt("refund_fee_$0"));
                    response.setRefundDateTime(resultJson.optString("refund_time_$0"));
                    if ("SUCCESS".equals(resultJson.optString("refund_status_$n0"))) {
                        response.setRefundResult(PaymentStatus.REFUND_SUCCESS.getMessage());
                        response.setCode(PaymentStatus.REFUND_SUCCESS.getCode());
                        response.setMessage(PaymentStatus.REFUND_SUCCESS.getMessage());
                    } else if ("FAIL".equals(resultJson.optString("refund_status_$n0"))) {
                        response.setRefundResult(PaymentStatus.REFUND_FAIL.getMessage());
                        response.setCode(PaymentStatus.REFUND_FAIL.getCode());
                        response.setMessage(PaymentStatus.REFUND_FAIL.getMessage());
                    } else if ("PROCESSING".equals(resultJson.optString("refund_status_$n0"))) {
                        response.setRefundResult(PaymentStatus.REFUND_PROCESSING.getMessage());
                        response.setCode(PaymentStatus.REFUND_PROCESSING.getCode());
                        response.setMessage(PaymentStatus.REFUND_PROCESSING.getMessage());
                    } else if ("CHANGE".equals(resultJson.optString("refund_status_$n0"))) {
                        response.setRefundResult(PaymentStatus.REFUND_CHANGE.getMessage());
                        response.setCode(PaymentStatus.REFUND_CHANGE.getCode());
                        response.setMessage(PaymentStatus.REFUND_CHANGE.getMessage());
                    }
                } else {
                    response.setCode(PaymentStatus.PAY_FAIL.getCode());
                    response.setMessage(PaymentStatus.PAY_FAIL.getMessage());
                }
            } else {
                LOGGER.info("本次查询退款状态交易通信失败！");
                response.setCode(PaymentStatus.COMMUNICATION_FAIL.getCode());
                response.setMessage(PaymentStatus.COMMUNICATION_FAIL.getMessage());
            }
        } catch (Exception e) {
            LOGGER.error("构建查询退款状态返回参数异常", e);
            throw new PaymentException(PaymentStatus.REFUND_BUILD_RESPONSE_PARAM_ERROR.getCode(), request.toString(), PaymentStatus.REFUND_BUILD_RESPONSE_PARAM_ERROR.getMessage());
        }
        return response;
    }

    /**
     * 签名请求参数
     * 默认实现，子类可覆盖
     *
     * @param reqParams     请求参数
     * @param paymentConfig 支付配置
     */
    @Override
    protected void signRequest(JSONObject reqParams, JSONObject paymentConfig) {
        try {
            SignUtils.signRequestParams(reqParams, "RSA_1_256", "sign", paymentConfig.getString("privateKey"));
        } catch (Exception e) {
            LOGGER.error("签名失败", e);
            throw new PaymentException(PaymentStatus.SIGN_FAIL.getCode(), reqParams.toString(), PaymentStatus.SIGN_FAIL.getMessage());
        }
    }

    /**
     * 发起支付请求
     * 默认实现，子类可覆盖
     *
     * @param reqParams     请求参数
     * @param paymentConfig 支付配置
     * @return 响应结果
     */
    @Override
    protected JSONObject executePaymentRequest(JSONObject reqParams, JSONObject paymentConfig) {
        try {
            JSONObject headers = new JSONObject();
            String formattedRequest = JsonTools.jsonConvertToXml(reqParams);
            LOGGER.info("请求支付接口, 请求报文={}", formattedRequest);
            headers.put("Content-Type", "application/xml");
            String response = RestTools.post(paymentConfig.getString("transactionurl"), formattedRequest, headers.toString());
            return JsonTools.xmlConvertToJson(response);
        } catch (Exception e) {
            LOGGER.error("请求支付接口异常", e);
            throw new PaymentException(PaymentStatus.PAY_ORDER_ERROR.getCode(), reqParams.toString(), PaymentStatus.PAY_ORDER_ERROR.getMessage());
        }
    }


    /**
     * 验签响应结果
     * 默认实现，子类可覆盖
     *
     * @param response      响应结果
     * @param paymentConfig 支付配置
     * @return 验签是否成功
     */
    @Override
    protected boolean verifyResponseSign(JSONObject response, JSONObject paymentConfig) {
        try {
            return SignUtils.verifySign(response, response.getString("sign_type"), response.getString("sign"), paymentConfig.getString("publicKey"));
        } catch (Exception e) {
            LOGGER.error("验签失败！", e);
            throw new PaymentException(PaymentStatus.VERIFY_SIGN_FAIL.getCode(), "{}", PaymentStatus.VERIFY_SIGN_FAIL.getMessage());
        }
    }


    /**
     * 获取请求头
     *
     * @param paymentConfig 支付配置
     * @return 请求头
     */
    private JSONObject getRequestHeard(JSONObject paymentConfig) {
        JSONObject reqParams = new JSONObject();
        reqParams.put("sign_type", "RSA_1_256");
        reqParams.put("mch_id", paymentConfig.getString("mchId"));
        reqParams.put("nonce_str", String.valueOf(new Date().getTime()));
        return reqParams;
    }
}


