package com.af.v4.system.common.mq;

import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.core.utils.uuid.IdUtils;
import com.af.v4.system.common.logic.service.LogicService;
import org.apache.rocketmq.client.producer.SendCallback;
import org.apache.rocketmq.client.producer.SendResult;
import org.apache.rocketmq.common.message.Message;
import org.apache.rocketmq.spring.core.RocketMQTemplate;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.nio.charset.StandardCharsets;

/**
 * RocketMQ生产者实现
 *
 * @author Mr.river
 */
@Component
public class RocketMQProducer {
    public static final String DEFAULT_LOGIC_TOPIC = "AF_LOGIC_TOPIC";
    private static final Logger LOGGER = LoggerFactory.getLogger(RocketMQProducer.class);
    private final RocketMQTemplate rocketMQTemplate;
    private final LogicService logicService;

    public RocketMQProducer(RocketMQTemplate rocketMQTemplate, LogicService logicService) {
        this.rocketMQTemplate = rocketMQTemplate;
        this.logicService = logicService;
    }

    /**
     * 同步发送 sync
     * <p>
     * 发送消息采用同步模式，这种方式只有在消息完全发送完成之后才返回结果，此方式存在需要同步等待发送结果的时间代价。
     * 这种方式具有内部重试机制，即在主动声明本次消息发送失败之前，内部实现将重试一定次数，默认为2次（DefaultMQProducer＃getRetryTimesWhenSendFailed）。
     * 发送的结果存在同一个消息可能被多次发送给给broker，这里需要应用的开发者自己在消费端处理幂等性问题。
     * </p>
     *
     * @param topic topic
     * @param tags  tags
     * @param value 内容
     * @return 结果
     */
    public JSONObject syncSend(String topic, String tags, String value) {
        Message msg = new Message(topic, tags, value.getBytes(StandardCharsets.UTF_8));
        msg.setKeys(IdUtils.fastSimpleUUID());
        msg.setDeliverTimeMs(0);
        // 发送单向消息
        SendResult sendResult = rocketMQTemplate.syncSend(topic + ":" + tags, msg);
        JSONObject result = new JSONObject();
        result.put("msgId", sendResult.getMsgId());
        result.put("sendStatus", sendResult.getSendStatus().name());
        return result;
    }

    /**
     * 异步发送 async
     * <p>
     * 发送消息采用异步发送模式，消息发送后立刻返回，当消息完全完成发送后，会调用回调函数sendCallback来告知发送者本次发送是成功或者失败。
     * 异步模式通常用于响应时间敏感业务场景，即承受不了同步发送消息时等待返回的耗时代价。同同步发送一样，异步模式也在内部实现了重试机制，
     * 默认次数为2次（DefaultMQProducer#getRetryTimesWhenSendAsyncFailed}）。发送的结果同样存在同一个消息可能被多次发送给给broker，需要应用的开发者自己在消费端处理幂等性问题。
     * </p>
     *
     * @param topic             topic
     * @param tags              tags
     * @param value             内容
     * @param callbackLogicName 用于接收回调结果的logic名称
     */
    public void asyncSend(String topic, String tags, String value, String callbackLogicName) {
        asyncSend(topic, tags, value, sendResult -> logicService.run(callbackLogicName, parseMsgResult(sendResult).toString()));
    }

    public void asyncSend(String topic, String tags, String value, Function successFunction) {
        Message msg = new Message(topic, tags, value.getBytes(StandardCharsets.UTF_8));
        msg.setKeys(IdUtils.fastSimpleUUID());
        msg.setDeliverTimeMs(0);
        // 发送单向消息
        rocketMQTemplate.asyncSend(topic + ":" + tags, msg, new SendCallback() {
            @Override
            public void onSuccess(SendResult sendResult) {
                successFunction.callback(sendResult);
            }

            @Override
            public void onException(Throwable e) {
                String msg = "队列消息发送失败";
                LOGGER.error(msg, e);
                throw new ServiceException(msg);
            }
        });
    }

    /**
     * 直接发送 one-way
     * <p>
     * 采用one-way发送模式发送消息的时候，发送端发送完消息后会立即返回，不会等待来自broker的ack来告知本次消息发送是否完全完成发送。
     * 这种方式吞吐量很大，但是存在消息丢失的风险，所以其适用于不重要的消息发送，比如日志收集
     * </p>
     *
     * @param topic topic
     * @param tags  tags
     * @param value 内容
     */
    public void sendOneway(String topic, String tags, String value) {
        Message msg = new Message(topic, tags, value.getBytes(StandardCharsets.UTF_8));
        msg.setKeys(IdUtils.fastSimpleUUID());
        msg.setDeliverTimeMs(0);
        // 发送单向消息
        rocketMQTemplate.sendOneWay(topic + ":" + tags, msg);
    }

    /**
     * 将{@link SendResult}转换为{@link org.json.JSONObject}
     *
     * @param sendResult sendResult
     * @return org.json.JSONObject
     */
    private JSONObject parseMsgResult(SendResult sendResult) {
        JSONObject result = new JSONObject();
        result.put("msgId", sendResult.getMsgId());
        result.put("sendStatus", sendResult.getSendStatus().name());
        return result;
    }

    public interface Function {
        void callback(SendResult sendResult);
    }
}
