package com.af.v4.system.common.mongodb.service;

import com.mongodb.client.result.DeleteResult;
import com.mongodb.client.result.UpdateResult;
import org.bson.types.ObjectId;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.aggregation.Aggregation;
import org.springframework.data.mongodb.core.aggregation.AggregationResults;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Component;

import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * {@link MongoTemplate}的org.json实现
 *
 * @author Mr.river
 */
@Component
public class MongodbService {
    private static final Logger LOGGER = LoggerFactory.getLogger(MongodbService.class);
    /**
     * mongodb操作集
     */
    private final MongoTemplate mongoTemplate;

    public MongodbService(MongoTemplate mongoTemplate) {
        this.mongoTemplate = mongoTemplate;
    }

    /**
     * 保存/更新数据
     *
     * @param value          JSON数据
     * @param collectionName 集合名称
     * @return 数据id
     */
    public String save(JSONObject value, String collectionName) {
        Map<String, Object> result = mongoTemplate.save(value.toMap(), collectionName);
        ObjectId id = (ObjectId) result.get("_id");
        return id.toString();
    }

    /**
     * 批量保存/更新数据
     *
     * @param value          JSON数据集合
     * @param collectionName 集合名称
     * @return 数据id集合
     */
    public JSONArray save(JSONArray value, String collectionName) {
        JSONArray resultIds = new JSONArray(value.length());
        value.forEach(item -> {
            Map<String, Object> result = mongoTemplate.save(((JSONObject) item).toMap(), collectionName);
            ObjectId id = (ObjectId) result.get("_id");
            resultIds.put(id);
        });
        return resultIds;
    }

    /**
     * 根据ID更新数据
     *
     * @param id             ID
     * @param updateContent  更新内容
     * @param collectionName 集合名称
     * @return 更新的数量
     */
    public Long update(String id, JSONObject updateContent, String collectionName) {
        Update update = new Update();
        updateContent.toMap().forEach(update::set);
        UpdateResult updateResult = mongoTemplate.updateMulti(SimpleQuery.byId(id), update, collectionName);
        return updateResult.getModifiedCount();
    }

    /**
     * 根据ID删除某个字段
     *
     * @param id             ID
     * @param fieldName      需要删除的字段
     * @param collectionName 集合名称
     * @return 更新的数量
     */
    public Long updateUnset(String id, String fieldName, String collectionName) {
        Update update = new Update().unset(fieldName);
        UpdateResult updateResult = mongoTemplate.updateMulti(SimpleQuery.byId(id), update, collectionName);
        return updateResult.getModifiedCount();
    }

    /**
     * 根据条件更新数据
     *
     * @param criteria       查询用API
     * @param updateContent  更新内容
     * @param collectionName 集合名称
     * @return 更新的数量
     */
    public Long update(Criteria criteria, JSONObject updateContent, String collectionName) {
        Update update = new Update();
        updateContent.toMap().forEach(update::set);
        UpdateResult updateResult = mongoTemplate.updateMulti(new Query(criteria), update, collectionName);
        return updateResult.getModifiedCount();
    }

    /**
     * 根据ID删除数据
     *
     * @param id             ID
     * @param collectionName 集合名称
     * @return 删除的数量
     */
    public Long removeOne(String id, String collectionName) {
        DeleteResult result = mongoTemplate.remove(SimpleQuery.byId(id), Map.class, collectionName);
        return result.getDeletedCount();
    }

    /**
     * 根据ID集合删除数据
     *
     * @param ids            ID
     * @param collectionName 集合名称
     * @return 删除的数量
     */
    public Long removeByIds(JSONArray ids, String collectionName) {
        DeleteResult result = mongoTemplate.remove(SimpleQuery.byIds(ids.toList()), Map.class, collectionName);
        return result.getDeletedCount();
    }

    /**
     * 根据条件删除数据
     *
     * @param criteria       查询用API
     * @param collectionName 集合名称
     * @return 删除的数量
     */
    public Long remove(Criteria criteria, String collectionName) {
        DeleteResult result = mongoTemplate.remove(new Query(criteria), Map.class, collectionName);
        return result.getDeletedCount();
    }

    /**
     * 根据ID查询数据
     *
     * @param id             ID
     * @param collectionName 集合名称
     * @return 查询结果
     */
    public JSONObject findOne(String id, String collectionName) {
        Map result = mongoTemplate.findById(new ObjectId(id), Map.class, collectionName);
        if (result != null) {
            result.put("_id", result.get("_id").toString());
        }
        return new JSONObject(result);
    }

    /**
     * 根据ID集合查询数据
     *
     * @param ids            ID集合
     * @param collectionName 集合名称
     * @return 查询结果
     */
    public JSONArray findByIds(JSONArray ids, String collectionName) {
        List<Map> result = mongoTemplate.find(SimpleQuery.byIds(ids.toList()), Map.class, collectionName).stream().peek(
                item -> {
                    item.put("_id", item.get("_id").toString());
                }
        ).toList();
        return new JSONArray(result);
    }

    /**
     * 根据条件查询数据
     *
     * @param criteria       查询用API
     * @param collectionName 集合名称
     * @return 查询结果
     */
    public JSONArray find(Criteria criteria, String collectionName) {
        List<Map> result = mongoTemplate.find(new Query(criteria), Map.class, collectionName).stream().peek(
                item -> {
                    item.put("_id", item.get("_id").toString());
                }
        ).toList();
        return new JSONArray(result);
    }

    /**
     * 聚合查询数据
     *
     * @param aggregation    聚合查询
     * @param collectionName 集合名称
     * @return 查询结果
     */
    public JSONArray aggregate(Aggregation aggregation, String collectionName) {
        AggregationResults<Map> results = mongoTemplate.aggregate(aggregation, collectionName, Map.class);
        JSONArray jsonArray = new JSONArray();
        for (Map map : results) {
            jsonArray.put(new JSONObject(map));
        }
        return jsonArray;
    }

    /**
     * 通用查询辅助
     */
    public static class SimpleQuery {
        public static Query byId(String id) {
            return new Query(Criteria.where("_id").is(new ObjectId(id)));
        }

        public static Query byIds(Collection<?> ids) {
            return new Query(Criteria.where("_id").in(ids.stream().map(item -> new ObjectId(item.toString())).toList()));
        }
    }
}
