package com.af.v4.system.common.mobile.controller;

import com.af.v4.system.common.log.annotation.Log;
import com.af.v4.system.common.log.enums.BusinessType;
import com.af.v4.system.common.resource.enums.ResourceType;
import com.af.v4.system.common.resource.mapper.AbstractResourceMapper;
import com.af.v4.system.common.resource.mapper.LogicMapper;
import com.af.v4.system.common.resource.mapper.SqlMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.*;

import java.math.BigInteger;
import java.security.MessageDigest;
import java.util.HashMap;
import java.util.Map;

@RestController
@RequestMapping("/dir2")
public class MobileLogicResourceController {
    private static final Logger LOGGER = LoggerFactory.getLogger(MobileLogicResourceController.class);
    // 生成的各种xml文件内容，key：logic、sql、path等，value：文件内容
    private static final Map<String, String> xmls = new HashMap<>();
    private final LogicMapper logicMapper;
    private final SqlMapper sqlMapper;

    public MobileLogicResourceController(LogicMapper logicMapper, SqlMapper sqlMapper) {
        this.logicMapper = logicMapper;
        this.sqlMapper = sqlMapper;
    }

    @Log(title = "getFiles", businessType = BusinessType.OTHER)
    @GetMapping(produces = MediaType.APPLICATION_JSON_VALUE)
    public String getFiles() {
        String result = "";
        result += getLogicMap(true);
        result += STR."|\{getSqlMap(true)}";
        return result;
    }

    @Log(title = "getFilesNoXML", businessType = BusinessType.OTHER)
    @GetMapping(value = "/noxml", produces = MediaType.APPLICATION_JSON_VALUE)
    public String getFilesNoXML() {
        String result = "";
        result = this.union(result, getLogicMap(false), "|");
        result = this.union(result, getSqlMap(false), "|");
        return result;
    }

    /**
     * 获取具体文件内容
     */
    @Log(title = "getFile", businessType = BusinessType.OTHER)
    @PostMapping(produces = MediaType.APPLICATION_JSON_VALUE)
    public String getFile(@RequestBody String fileName) {
        // 如果获取xml文件内容，从产生的xml中获取
        if (fileName.endsWith(".xml")) {
            String type = fileName.substring(0, fileName.length() - 4);
            return xmls.get(type);
        } else {
            String name = fileName.substring(fileName.lastIndexOf("/") + 1, fileName.lastIndexOf("."));
            AbstractResourceMapper<?> mapper;
            if (fileName.endsWith(ResourceType.LOGIC.getValue())) {
                mapper = logicMapper;
            } else {
                mapper = sqlMapper;
            }
            String res = mapper.getResource(name).getSource();
            if(res.startsWith("$")){
                res = res.substring(1);
            }
            return res;
        }
    }

    // 获取业务逻辑文件名列表
    // hasXml: 是否需要xml文件
    // return: 业务逻辑文件名,业务逻辑MD5值|业务逻辑文件名,业务逻辑MD5值
    private String getLogicMap(boolean hasXml) {
        Map<String, LogicMapper.LogicResource> logicMap = logicMapper.getAllMap();
        return this.getMap(logicMap, ResourceType.LOGIC, hasXml);
    }

    private String getSqlMap(boolean hasXml) {
        Map<String, SqlMapper.SqlResource> sqlMap = sqlMapper.getAllMap();
        return this.getMap(sqlMap, ResourceType.SQL, hasXml);
    }

    // 从map中获取相关文件内容，map有可能是logic、sql或者path
    // type: 指明是logic.xml等
    // hasXml: 是否需要xml文件，新版不要xml注册文件
    private String getMap(Map<String, ? extends AbstractResourceMapper.MobileResourceSupport> resMap, ResourceType type, boolean hasXml) {
        StringBuilder result = new StringBuilder();
        // xml文件内容
        StringBuilder xml = new StringBuilder("<cfg>");
        for (Map.Entry<String, ?> entry : resMap.entrySet()) {
            if (((AbstractResourceMapper.MobileResourceSupport) entry.getValue()).isMobile()) {
                AbstractResourceMapper.CommonResource resource = (AbstractResourceMapper.CommonResource) entry.getValue();
                if (!result.toString().isEmpty()) {
                    result.append("|");
                }
                String file = entry.getKey();
                file = STR."\{type.getValue()}s/\{file}.\{type.getValue()}";
                result.append(file).append(",").append(this.getMD5(resource));
                file = file.substring(file.indexOf("/") + 1);
                // xml文件中的path要去掉/logics/等内容
                // 给xml文件加一行配置
                xml.append("<").append(type).append(" alias='").append(entry.getKey()).append("' path='").append(file).append("' />");
            }
        }

        xml.append("</cfg>");
        // 设置对应xml文件内容
        if (!xmls.containsKey(type.getValue())) {
            xmls.put(type.getValue(), xml.toString());
        }

        // 如果不需要xml文件，直接返回
        if (!hasXml) {
            return result.toString();
        }

        // 返回对应的xml文件
        if (!result.toString().isEmpty()) {
            result.append("|");
        }
        result.append("/").append(type).append(".xml").append(",").append(this.getStringMD5(xml.toString()));

        return result.toString();
    }

    // 计算文件内容的md5
    private String getMD5(AbstractResourceMapper.CommonResource resource) {
        String str = resource.getSource();
        return this.getStringMD5(str);
    }

    // 计算字符串的MD5
    private String getStringMD5(String str) {
        try {
            MessageDigest md = MessageDigest.getInstance("MD5");
            md.update(str.getBytes());
            return new BigInteger(1, md.digest()).toString(16);
        } catch (Exception e) {
            throw new RuntimeException("md5加密失败", e);
        }
    }

    /**
     * 把两个字符串按给定字符串进行连接，考虑了字符串为空的情况
     */
    private String union(String str1, String str2, String ch) {
        if (str1 == null || str1.isEmpty()) {
            return str2;
        }
        if (str2 == null || str2.isEmpty()) {
            return str1;
        }
        return str1 + ch + str2;
    }
}
