package com.af.v4.system.common.mobile.controller;

import com.af.v4.system.common.log.annotation.Log;
import com.af.v4.system.common.log.enums.BusinessType;
import com.af.v4.system.common.mobile.config.MobileConfig;
import com.af.v4.system.common.plugins.io.IOTools;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;

//目录服务
@RestController
@RequestMapping("/dir")
public class MobileStaticResourceController {

    private static final Logger LOGGER = LoggerFactory.getLogger(MobileStaticResourceController.class);

    private final MobileConfig mobileConfig;

    public MobileStaticResourceController(MobileConfig mobileConfig) {
        this.mobileConfig = mobileConfig;
    }

    /**
     * 获取所有卡文件的最后修改时间
     */
    @Log(title = "getFiles", businessType = BusinessType.OTHER)
    @PostMapping(value = "/getfile", produces = MediaType.APPLICATION_JSON_VALUE)
    public String getFiles(@RequestBody String path) {
        StringBuilder result = new StringBuilder();
        if (path == null || path.trim().isEmpty())
            throw new RuntimeException("文件路径为null");
        //递归获取所有文件的最后修改日期
        File file = new File(path);
        List<File> files = list(file);
        for (File f : files) {
            if (!result.toString().isEmpty()) {
                result.append("|");
            }
            result.append(f.getPath()).append(",").append(f.lastModified());
        }
        return result.toString();
    }

    @Log(title = "getGlobalFileByName", businessType = BusinessType.OTHER)
    @PostMapping(value = "/{name}", produces = MediaType.APPLICATION_JSON_VALUE)
    public String getGlobalFileByName(@PathVariable String name, HttpServletResponse response) {
        try {
            response.setStatus(HttpServletResponse.SC_OK);
            response.setContentType("application/octet-stream");
            // 把文件的内容送入响应流中
            try (InputStream is = new FileInputStream(mobileConfig.getGlobalPath());
                 OutputStream os = new BufferedOutputStream(response.getOutputStream())
            ) {
                transformStream(is, os);
            }
        } catch (IOException e) {
            LOGGER.error("getGlobalFileByName发生错误", e);
        }
        return "";
    }

    @Log(title = "getVersion", businessType = BusinessType.OTHER)
    @GetMapping(value = "/version", produces = MediaType.APPLICATION_JSON_VALUE)
    public String getVersion(HttpServletResponse response) {
        String value = IOTools.readText("app.json");
        try {
            response.setStatus(HttpServletResponse.SC_OK);
            response.setContentType("application/octet-stream");
            // 把文件的内容送入响应流中
            value = URLDecoder.decode(value, StandardCharsets.UTF_8);
            try (InputStream is = new FileInputStream(value);
                 OutputStream os = new BufferedOutputStream(response.getOutputStream())
            ) {
                transformStream(is, os);
            }
        } catch (IOException e) {
            LOGGER.error("getVersion发生错误", e);
        }
        return "";
    }

    @Log(title = "getFile", businessType = BusinessType.OTHER)
    @PostMapping(produces = MediaType.APPLICATION_JSON_VALUE)
    public String getFile(HttpServletResponse response, @RequestBody String fileName) {
        try {
            fileName = URLDecoder.decode(fileName, StandardCharsets.UTF_8);
            response.setStatus(HttpServletResponse.SC_OK);
            response.setContentType("application/octet-stream");
            // 把文件的内容送入响应流中
            try (InputStream is = new FileInputStream(fileName);
                 OutputStream os = new BufferedOutputStream(response.getOutputStream())
            ) {
                transformStream(is, os);
            }
        } catch (IOException e) {
            LOGGER.error("getFile发生错误", e);
        }
        return "";
    }

    public void transformStream(InputStream is, OutputStream os) {
        try {
            byte[] buffer = new byte[1024];
            // 读取的实际长度
            int length = is.read(buffer);
            while (length != -1) {
                os.write(buffer, 0, length);
                length = is.read(buffer);
            }
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    //递归获取目录下所有文件，包括子目录
    private List<File> list(File dir) {
        List<File> result = new ArrayList<>();
        File[] files = dir.listFiles();
        assert files != null;
        for (File file : files) {
            if (file.isDirectory()) {
                result.addAll(list(file));
            } else {
                result.add(file);
            }
        }
        return result;
    }


}
