package com.af.v4.system.common.logic.service;

import com.af.v4.system.api.RemoteLogicService;
import com.af.v4.system.api.factory.DynamicFeignClientFactory;
import com.af.v4.system.common.core.constant.CodeNormsConstants;
import com.af.v4.system.common.core.constant.SecurityConstants;
import com.af.v4.system.common.core.context.GlobalVisualThreadContext;
import com.af.v4.system.common.core.domain.R;
import com.af.v4.system.common.core.enums.EnvType;
import com.af.v4.system.common.core.exception.LogicException;
import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.core.proxy.logic.ILogicServiceProxy;
import com.af.v4.system.common.core.service.ApplicationService;
import com.af.v4.system.common.datasource.DynamicDataSource;
import com.af.v4.system.common.expression.Expression;
import com.af.v4.system.common.redis.RedisService;
import com.af.v4.system.common.resource.constant.CacheKey;
import com.af.v4.system.common.resource.mapper.LogicMapper;
import com.af.v4.system.common.resource.mapper.PluginMapper;
import org.apache.logging.log4j.LogManager;
import org.json.JSONArray;
import org.json.JSONObject;
import org.json.JSONTokener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.Primary;
import org.springframework.stereotype.Service;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.DefaultTransactionDefinition;
import org.springframework.util.StopWatch;

import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Semaphore;
import java.util.function.Supplier;

/**
 * Logic运行时服务
 *
 * @author Mr.river
 * @since 1.0.0
 */
@Primary
@Service
@Transactional(rollbackFor = Exception.class)
public class LogicService implements ILogicServiceProxy {
    private static final Logger LOGGER = LoggerFactory.getLogger(LogicService.class);
    private static final org.apache.logging.log4j.Logger LOG4J_LOGGER = LogManager.getLogger(LogicService.class);
    private static final Semaphore DEFAULT_ASYNC_LOGIC_SEMAPHORE = new Semaphore(32);
    private final DynamicFeignClientFactory<RemoteLogicService> dynamicFeignClientFactory;
    private final LogicMapper logicMapper;
    private final ApplicationService applicationService;
    private final RedisService redisService;
    private final PlatformTransactionManager transactionManager;

    public LogicService(DynamicFeignClientFactory<RemoteLogicService> dynamicFeignClientFactory, LogicMapper logicMapper, ApplicationService applicationService, RedisService redisService, PlatformTransactionManager transactionManager) {
        this.dynamicFeignClientFactory = dynamicFeignClientFactory;
        this.logicMapper = logicMapper;
        this.applicationService = applicationService;
        this.redisService = redisService;
        this.transactionManager = transactionManager;
    }

    /**
     * 获取logic使用的数据源名称
     * 优先级从高到低排列如下
     * 1. run方法中指定的数据源名称
     * 2. logic注册文件中指定的数据源名称
     * 3. 当前线程中已有的数据源名称
     *
     * @param dataSourceName 指定的数据源名称
     * @param resource       logic注册资源
     * @return 数据源名称
     */
    private static String getDataSourceName(String dataSourceName, LogicMapper.LogicResource resource) {
        return dataSourceName != null ? dataSourceName
                : (resource.getDataSource() != null ? resource.getDataSource() : DynamicDataSource.getDataSource());
    }

    /**
     * 是否存在指定名称的Logic资源
     *
     * @param logicName Logic名称
     * @return 是否存在
     */
    public boolean has(String logicName) {
        return logicMapper.hasResource(logicName);
    }

    /**
     * 执行Logic
     *
     * @param logicName      Logic名称
     * @param param          参数
     * @param dataSourceName 数据源名称
     * @return 执行结果
     */
    public Object run(String logicName, JSONObject param, String dataSourceName) {
        // 获取Logic资源
        LogicMapper.LogicResource resource = logicMapper.getResource(logicName);
        String name = resource.getAlias();
        boolean cache = resource.isCache();
        String dataSource = getDataSourceName(dataSourceName, resource);
        LOGGER.info(CodeNormsConstants.DEBUG_PREFIX + "执行Logic[{}]，useCache: {}，params: {}，dataSource：{}", name, cache, param, dataSource);
        if (applicationService.getEnvType() == EnvType.LOCAL && resource.isDeprecated()) {
            String replacedBy = resource.getReplacedBy();
            if (replacedBy != null) {
                LOGGER.warn("Logic[{}]已标记废弃，可能会在后续版本中移除，请迁移到新API[{}]", name, replacedBy);
            } else {
                LOGGER.warn("Logic[{}]已标记废弃，可能会在后续版本中移除", name);
            }
        }
        String source = resource.getSource();
        // 附加用户注册的对象到业务逻辑中
        Map<String, Object> params = new HashMap<>(PluginMapper.getPluginMap());
        params.put("data", param);
        params.put("log", LOG4J_LOGGER);
        params.put("ENV", applicationService.getValues());
        params.put("RESOURCE", resource.getJsonParams());
        params.put("logic", this);

        return logicMonitor(name, resource.getPath(), dataSource, param.toString(), () -> runExpression(source, params));
    }

    public Object run(String name, JSONObject param) {
        return this.run(name, param, null);
    }

    public Object run(String name, String str) {
        return this.run(name, buildLogicParams(str));
    }

    /**
     * 异步执行 Logic（独立事务）
     *
     * @param name      Logic 名称
     * @param param     参数
     * @param semaphore 并发信号量，通过设置信号量控制并发能力，传入 null 使用默认信号量
     * @return CompletableFuture 异步返回结果
     */
    public CompletableFuture<Object> runAsync(String name, JSONObject param, Semaphore semaphore) {
        // 参数校验
        if (name == null || name.isEmpty()) {
            throw new IllegalArgumentException("Logic 名称不能为空");
        }
        if (param == null) {
            throw new IllegalArgumentException("参数不能为空");
        }
        final Semaphore finalSemaphore = Objects.requireNonNullElse(semaphore, DEFAULT_ASYNC_LOGIC_SEMAPHORE);

        CompletableFuture<Object> future;
        try {
            finalSemaphore.acquire();
            future = GlobalVisualThreadContext.runAsync(() -> run(name, param)).whenComplete((result, throwable) -> {
                // 确保信号量释放
                finalSemaphore.release();
                if (throwable != null) {
                    LOGGER.error("异步 Logic 执行失败. Logic 名称: {}, 参数: {}", name, param, throwable);
                }
            });
        } catch (InterruptedException e) {
            LOGGER.error("线程中断异常. Logic 名称: {}", name, e);
            Thread.currentThread().interrupt(); // 恢复中断标志
            throw new ServiceException("线程被中断"); // 保留原始异常信息
        }
        return future;
    }

    public CompletableFuture<Object> runAsync(String name, JSONObject param) {
        return runAsync(name, param, null);
    }

    public CompletableFuture<Object> runAsync(String name, String str) {
        return this.runAsync(name, buildLogicParams(str));
    }

    /**
     * 远程调用Logic
     *
     * @param serviceName 服务名
     * @param logicName   Logic名称
     * @param params      Logic参数
     * @return 执行结果
     */
    //    @GlobalTransactional
    public JSONObject remoteRun(String serviceName, String logicName, String params) {
        LOGGER.info(CodeNormsConstants.DEBUG_PREFIX + "远程调用{}的Logic[{}]，参数：{}", serviceName, logicName, params);
        return (JSONObject) logicMonitor(logicName, "http://" + serviceName + "/logic/" + logicName, null, params, () -> {
            RemoteLogicService remoteLogicService = dynamicFeignClientFactory.getFeignClient(RemoteLogicService.class, serviceName);
            R<Object> logicResult = remoteLogicService.run(logicName, params, SecurityConstants.INNER);
            return logicResult.parseResponseJson();
        });
    }

    //    @GlobalTransactional
    public JSONObject remoteRun(String serviceName, String logicName, JSONObject params) {
        return remoteRun(serviceName, logicName, params.toString());
    }

    /**
     * 执行原生表达式
     *
     * @param source 表达式内容
     * @param params 参数
     * @return 执行结果
     */
    public Object runExpression(String source, Map<String, Object> params) {
        return Expression.run(source, params);
    }

    /**
     * 构建Logic参数
     *
     * @param str 字符串参数
     * @return JSON格式参数
     */
    private JSONObject buildLogicParams(String str) {
        JSONObject param = null;
        // 把传递过来的参数，放到data里，以便跟entity，sql等对象区别开来
        // 判断传来的数据是否是XML格式
        str = str.trim();
        if (str.startsWith("<") && str.endsWith(">")) {
            // 传来数据为XML格式
            param = new JSONObject();
            param.put("xml", str);
        } else {
            // 传来数据为JSON格式
            Object json = new JSONTokener(str).nextValue();
            if (json instanceof JSONObject st) {
                param = st;
                if (!param.isNull("data") && param.keySet().size() == 1) {
                    Object dataParam = param.get("data");
                    if (dataParam instanceof JSONObject) {
                        param = (JSONObject) dataParam;
                        param.put("standardData", new JSONObject(str).toString());
                    }
                }
            } else if (json instanceof JSONArray ar) {
                param = new JSONObject();
                param.put("arr", ar);
            }
        }
        return param;
    }

    private Object logicMonitor(String name, String path, String dataSource, String params, Supplier<Object> business) {
        StopWatch stopWatch = new StopWatch();
        stopWatch.start();
        try {
            return DynamicDataSource.withDataSource(dataSource, business);
        } catch (Exception e) {
            throw buildServiceException(name, path, params, e);
        } finally {
            stopWatch.stop();
            logExecutionTime(name, stopWatch);
        }
    }

    private void logExecutionTime(String name, StopWatch stopWatch) {
        long elapsedTime = stopWatch.getTotalTimeMillis();
        String message = CodeNormsConstants.DEBUG_PREFIX + "Logic[" + name + "]处理耗时:" + elapsedTime + "ms";
        if (elapsedTime >= 8000) {
            LOGGER.error(message);
        } else if (elapsedTime >= 4000) {
            LOGGER.warn(message);
        } else {
            LOGGER.info(message);
        }
    }

    /**
     * 使用独立事务执行业务
     *
     * @param business 业务实现
     * @return 执行结果
     */
    private Object runWithTransaction(Supplier<Object> business) {
        DefaultTransactionDefinition def = new DefaultTransactionDefinition();
        def.setPropagationBehavior(Propagation.REQUIRES_NEW.value());
        TransactionStatus status = transactionManager.getTransaction(def);
        try {
            Object result = business.get();
            transactionManager.commit(status);
            return result;
        } catch (Exception e) {
            transactionManager.rollback(status);
            throw e;
        }
    }

    /**
     * 构建业务异常
     *
     * @param ex 运行时异常
     * @return 业务异常
     */
    private ServiceException buildServiceException(String logicName, String logicPath, String logicParams, Exception ex) {
        Throwable cause = ex.getCause();
        if (cause instanceof InvocationTargetException targetException) {
            cause = targetException.getTargetException();
        }
        if (cause instanceof LogicException logicException) {
            return new LogicException(logicException.getMessage(), logicException.getCode(), ex, logicName, logicParams, logicPath, logicException);
        } else if (cause instanceof ServiceException serviceException) {
            return new LogicException(serviceException.getMessage(), serviceException.getCode(), ex, logicName, logicParams, logicPath);
        } else {
            return new LogicException(ex, logicName, logicParams, logicPath);
        }
    }

    /**
     * 删除缓存
     *
     * @param logicName Logic名称
     */
    public void deleteCache(String logicName) {
        String cacheKey = CacheKey.getLogicDataCacheKey(logicName, "*");
        Iterable<String> cacheKeyList = redisService.getKeys(cacheKey);
        redisService.deleteList(cacheKeyList);
        LOGGER.info("手动清除Logic[{}]缓存完成", logicName);
    }
}
