package com.af.v4.system.common.logic.service;

import com.af.v4.system.api.RemoteLogicService;
import com.af.v4.system.api.factory.DynamicFeignClientFactory;
import com.af.v4.system.common.core.constant.CodeNormsConstants;
import com.af.v4.system.common.core.constant.HttpStatus;
import com.af.v4.system.common.core.constant.SecurityConstants;
import com.af.v4.system.common.core.domain.R;
import com.af.v4.system.common.core.exception.LogicException;
import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.datasource.DynamicDataSource;
import com.af.v4.system.common.expression.Expression;
import com.af.v4.system.common.redis.RedisService;
import com.af.v4.system.common.resource.constant.CacheKey;
import com.af.v4.system.common.resource.core.ResourceIndexer;
import com.af.v4.system.common.resource.enums.ResourceType;
import com.af.v4.system.common.resource.mapper.LogicMapper;
import com.af.v4.system.common.resource.mapper.PluginMapper;
import org.apache.logging.log4j.LogManager;
import org.json.JSONArray;
import org.json.JSONObject;
import org.json.JSONTokener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.lang.reflect.InvocationTargetException;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;

@Service
@Transactional(rollbackFor = Exception.class)
public class LogicService {
    private static final Logger LOGGER = LoggerFactory.getLogger(LogicService.class);
    private static final org.apache.logging.log4j.Logger LOG4J_LOGGER = LogManager.getLogger(LogicService.class);
    private final LogicNewTranService logicNewTranService;
    private final DynamicFeignClientFactory<RemoteLogicService> dynamicFeignClientFactory;
    private final RedisService redisService;
    private final LogicMapper logicMapper;
    private final ThreadPoolTaskExecutor taskExecutor;

    public LogicService(LogicNewTranService newLogicService, DynamicFeignClientFactory<RemoteLogicService> dynamicFeignClientFactory, RedisService redisService, LogicMapper logicMapper, @Qualifier("logicAsyncTaskExecutor") ThreadPoolTaskExecutor taskExecutor) {
        this.logicNewTranService = newLogicService;
        this.dynamicFeignClientFactory = dynamicFeignClientFactory;
        this.redisService = redisService;
        this.logicMapper = logicMapper;
        this.taskExecutor = taskExecutor;
    }

    /**
     * 执行Logic
     *
     * @param logicName      Logic名称
     * @param param          参数
     * @param dataSourceName 数据源名称
     * @return 执行结果
     */
    public Object run(String logicName, JSONObject param, String dataSourceName) {
        // 获取源程序内容
        LogicMapper.LogicResource resource = logicMapper.getValue(logicName);
        String name = resource.getAlias();
        boolean cache = resource.getCache();
        String dataSource;
        if (dataSourceName == null) {
            Map<String, String> res = logicMapper.getRes(name);
            String mappingDataSource = null;
            if (res != null) {
                mappingDataSource = res.get("dataSource");
            }
            dataSource = Objects.requireNonNullElse(mappingDataSource, DynamicDataSource.DEFAULT_DATASOURCE_NAME);
        } else {
            dataSource = dataSourceName;
        }
        LOGGER.info(CodeNormsConstants.DEBUG_PREFIX + "开始执行Logic[" + name + "]，useCache: " + cache + "，params: " + param + "，dataSource：" + dataSource);
        if (cache) {
            Map<String, Object> params = new TreeMap<>(param.toMap());
            // 获取自定义缓存Key
            String resCacheKey = resource.getCacheKey();
            String paramsStr;
            if (Optional.ofNullable(resCacheKey).isPresent()) {
                // 如果调用Logic的参数中含有该key，则取该key对应的值，否则直接以该key的名称作为缓存key
                paramsStr = params.containsKey(resCacheKey) ? params.get(resCacheKey).toString() : resCacheKey;
            } else {
                paramsStr = params.toString();
            }
            String cacheKey = CacheKey.getLogicDataCacheKey(logicName, paramsStr);
            Object result = redisService.get(cacheKey);
            if (result != null) {
                LOGGER.info(CodeNormsConstants.DEBUG_PREFIX + "Logic[" + name + "]命中缓存");
                return result;
            }
            AtomicReference<Object> returnResult = new AtomicReference<>();
            redisService.lock(cacheKey, 60, 300, () -> {
                Object newResult = redisService.get(cacheKey);
                if (newResult == null) {
                    newResult = action(resource, param, dataSource);
                    long time;
                    if (newResult instanceof JSONObject resultObj) {
                        // 根据返回值中的缓存时间重新赋值
                        time = resultObj.optLong("cacheTime", resource.getTime());
                    } else {
                        time = resource.getTime();
                    }
                    if (time > 0) {
                        redisService.set(cacheKey, newResult, time);
                        LOGGER.info(CodeNormsConstants.DEBUG_PREFIX + "Logic[" + name + "]缓存成功，存活时间：" + time + "秒");
                    }
                } else {
                    LOGGER.info(CodeNormsConstants.DEBUG_PREFIX + "Logic[" + name + "]命中缓存");
                }
                returnResult.set(newResult);
            });
            return returnResult.get();
        }
        return action(resource, param, dataSource);
    }

    public Object run(String name, JSONObject param) {
        return this.run(name, param, null);
    }

    public Object run(String name, String str) {
        return this.run(name, buildLogicParams(str));
    }

    /**
     * 异步执行Logic
     *
     * @param name  Logic名称
     * @param param 参数
     */
    public CompletableFuture<Object> runAsync(String name, JSONObject param) {
        return CompletableFuture.supplyAsync(() -> this.run(name, param), taskExecutor).exceptionally(throwable -> {
            LOGGER.error("异步请求发生异常.", throwable);
            return null;
        });
    }

    public CompletableFuture<Object> runAsync(String name, String str) {
        return this.runAsync(name, buildLogicParams(str));
    }

    /**
     * 远程调用Logic
     *
     * @param serviceName 服务名
     * @param logicName   Logic名称
     * @param params      Logic参数
     * @return 执行结果
     */
    //    @GlobalTransactional
    public JSONObject remoteRun(String serviceName, String logicName, String params) {
        LOGGER.info(CodeNormsConstants.DEBUG_PREFIX + "开始远程调用" + serviceName + "的Logic[" + logicName + "]，参数：" + params);
        return (JSONObject) logicMonitor(logicName, "http://" + serviceName + "/logic/" + logicName, params, () -> {
            RemoteLogicService remoteLogicService = dynamicFeignClientFactory.getFeignClient(RemoteLogicService.class, serviceName);
            R<Object> logicResult = remoteLogicService.run(logicName, params, SecurityConstants.INNER);
            return logicResult.parseResponseJson();
        });
    }

    //    @GlobalTransactional
    public JSONObject remoteRun(String serviceName, String logicName, JSONObject params) {
        return remoteRun(serviceName, logicName, params.toString());
    }

    /**
     * 执行资源
     *
     * @param resource       资源
     * @param param          参数
     * @param dataSourceName 数据源
     * @return 执行结果
     */
    private Object action(LogicMapper.LogicResource resource, JSONObject param, String dataSourceName) {
        String name = resource.getAlias();
        String path = resource.getPath();
        if (Optional.ofNullable(path).isEmpty()) {
            throw new ServiceException("业务逻辑未找到: " + name, HttpStatus.BAD_REQUEST);
        }
        String source = ResourceIndexer.getString(ResourceType.LOGIC, name, path);

        // 附加用户注册的对象到业务逻辑中
        Map<String, Object> params = new HashMap<>(PluginMapper.pluginMap);
        params.put("data", param);
        params.put("log", LOG4J_LOGGER);
        params.put("redis", redisService);
        params.put("logic", this);

        Supplier<Object> doBusiness = () -> {
            DynamicDataSource.setDataSource(dataSourceName);
            return logicMonitor(name, path, param.toString(), () -> {
                //使用当前事务执行
                return runExpression(source, params);
            }, DynamicDataSource::clearDataSource);
        };
        if (dataSourceName.equals(DynamicDataSource.DEFAULT_DATASOURCE_NAME)) {
            return doBusiness.get();
        } else {
            try {
                return CompletableFuture.supplyAsync(doBusiness, taskExecutor).exceptionally(throwable -> {
                    Throwable cause = throwable.getCause();
                    throw (ServiceException) cause;
                }).get();
            } catch (InterruptedException | ExecutionException e) {
                throw (ServiceException) e.getCause();
            }
        }
    }

    /**
     * 执行原生表达式
     *
     * @param source 表达式内容
     * @param params 参数
     * @return 执行结果
     */
    public Object runExpression(String source, Map<String, Object> params) {
        return Optional.ofNullable(Expression.run(source, params)).orElse("");
    }

    /**
     * 构建业务异常
     *
     * @param ex 运行时异常
     * @return 业务异常
     */
    private ServiceException buildServiceException(String logicName, String logicPath, String logicParams, Exception ex) {
        Throwable cause = ex.getCause();
        if (cause instanceof InvocationTargetException targetException) {
            cause = targetException.getTargetException();
        }
        if (cause instanceof LogicException logicException) {
            return new LogicException(logicException.getMessage(), logicException.getCode(), ex, logicName, logicParams, logicPath, logicException);
        } else if (cause instanceof ServiceException serviceException) {
            return new LogicException(serviceException.getMessage(), serviceException.getCode(), ex, logicName, logicParams, logicPath);
        } else {
            return new LogicException(ex, logicName, logicParams, logicPath);
        }
    }

    /**
     * 构建Logic参数
     *
     * @param str 字符串参数
     * @return JSON格式参数
     */
    private JSONObject buildLogicParams(String str) {
        JSONObject param = null;
        // 把传递过来的参数，放到data里，以便跟entity，sql等对象区别开来
        // 判断传来的数据是否是XML格式
        str = str.trim();
        if (str.startsWith("<") && str.endsWith(">")) {
            // 传来数据为XML格式
            param = new JSONObject();
            param.put("xml", str);
        } else {
            // 传来数据为JSON格式
            Object json = new JSONTokener(str).nextValue();
            if (json instanceof JSONObject st) {
                param = st;
                if (!param.isNull("data")) {
                    Object dataParam = param.get("data");
                    if (dataParam instanceof JSONObject) {
                        param = (JSONObject) dataParam;
                        param.put("standardData", new JSONObject(str).toString());
                    }
                }
            } else if (json instanceof JSONArray ar) {
                param = new JSONObject();
                param.put("arr", ar);
            }
        }
        return param;
    }

    private Object logicMonitor(String name, String path, String params, Process process, Finally finallyFun) {
        long begin = System.currentTimeMillis();
        try {
            return process.apply();
        } catch (Exception e) {
            throw buildServiceException(name, path, params, e);
        } finally {
            if (finallyFun != null) {
                finallyFun.apply();
            }
            long end = System.currentTimeMillis();
            long time = end - begin;
            String engMessage = CodeNormsConstants.DEBUG_PREFIX + "Logic[" + name + "]处理耗时:" + time;
            if (time >= 8000) {
                LOGGER.error(engMessage);
            } else if (time >= 4000) {
                LOGGER.warn(engMessage);
            } else {
                LOGGER.info(engMessage);
            }
        }
    }

    private Object logicMonitor(String name, String path, String params, Process process) {
        return logicMonitor(name, path, params, process, null);
    }

    @FunctionalInterface
    interface Process {
        Object apply();
    }

    @FunctionalInterface
    interface Finally {
        void apply();
    }
}
