package com.af.v4.system.common.logic.service;

import com.af.v4.system.api.RemoteEntityService;
import com.af.v4.system.api.RemoteLogicService;
import com.af.v4.system.common.core.constant.SecurityConstants;
import com.af.v4.system.common.core.domain.R;
import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.core.utils.StringUtils;
import com.af.v4.system.common.datasource.DynamicDataSource;
import com.af.v4.system.common.expression.Expression;
import com.af.v4.system.common.gpt.service.PromptService;
import com.af.v4.system.common.jpa.service.EntityService;
import com.af.v4.system.common.jpa.service.SqlService;
import com.af.v4.system.common.logic.utils.ExceptionHelper;
import com.af.v4.system.common.redis.RedisService;
import com.af.v4.system.common.resource.enums.ResourceType;
import com.af.v4.system.common.resource.mapper.AbstractResourceMapper;
import com.af.v4.system.common.resource.mapper.LogicMapper;
import com.af.v4.system.common.resource.mapper.PluginMapper;
import com.af.v4.system.common.resource.mapper.ResourceMapper;
import org.apache.logging.log4j.LogManager;
import org.json.JSONArray;
import org.json.JSONObject;
import org.json.JSONTokener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.lang.reflect.InvocationTargetException;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

@Service
@Transactional(rollbackFor = Exception.class)
public class LogicService {
    private static final Logger LOGGER = LoggerFactory.getLogger(LogicService.class);
    private static final org.apache.logging.log4j.Logger LOG4J_LOGGER = LogManager.getLogger(LogicService.class);
    private final SqlService sqlService;
    private final LogicNewTranService logicNewTranService;
    private final ResourceMapper resourceMapper;
    private final RedisService redisService;
    private final RemoteLogicService remoteLogicService;
    private final RemoteEntityService remoteEntityService;
    private final EntityService entityService;
    private final PromptService promptService;
    private final PluginMapper pluginMapper;
    private final LogicMapper logicMapper;

    public LogicService(SqlService sqlService, LogicNewTranService newLogicService, ResourceMapper resourceMapper, RedisService redisService, RemoteLogicService remoteLogicService, RemoteEntityService remoteEntityService, EntityService entityService, PromptService promptService, PluginMapper pluginMapper, LogicMapper logicMapper) {
        this.sqlService = sqlService;
        this.logicNewTranService = newLogicService;
        this.resourceMapper = resourceMapper;
        this.redisService = redisService;
        this.remoteLogicService = remoteLogicService;
        this.remoteEntityService = remoteEntityService;
        this.entityService = entityService;
        this.promptService = promptService;
        this.pluginMapper = pluginMapper;
        this.logicMapper = logicMapper;
    }

    /**
     * 执行Logic
     *
     * @param name Logic名称
     * @param str  字符串参数
     * @return 执行结果
     */
    public Object run(String name, String str) {
        JSONObject param = null;
        // 把传递过来的参数，放到data里，以便跟entity，sql等对象区别开来
        // 判断传来的数据是否是XML格式
        str = str.trim();
        if (str.startsWith("<") && str.endsWith(">")) {
            // 传来数据为XML格式
            param = new JSONObject();
            param.put("xml", str);
        } else {
            // 传来数据为JSON格式
            Object json = new JSONTokener(str).nextValue();
            if (json instanceof JSONObject st) {
                param = st;
                if (!param.isNull("data")) {
                    Object dataParam = param.get("data");
                    if (dataParam instanceof JSONObject) {
                        param = (JSONObject) dataParam;
                        param.put("standardData", new JSONObject(str).toString());
                    }
                }
            } else if (json instanceof JSONArray ar) {
                param = new JSONObject();
                param.put("arr", ar);
            }
        }

        return run(name, param);
    }

    /**
     * 执行Logic
     *
     * @param name  Logic名称
     * @param param 参数
     * @return 执行结果
     */
    public Object run(String name, JSONObject param) {
        Map<String, String> res = logicMapper.getRes(name);
        String dataSourceName = res.get("dataSource");
        return this.run(name, param, dataSourceName);
    }

    /**
     * 执行Logic
     *
     * @param name           Logic名称
     * @param param          参数
     * @param dataSourceName 数据源名称
     * @return 执行结果
     */
    public Object run(String name, JSONObject param, String dataSourceName) {
        // 获取源程序内容
        AbstractResourceMapper.CommonResource<String> commonResource = logicMapper.getValue(name);
        String path = commonResource.getPath();
        name = commonResource.getAlias();
        LOGGER.info("维护信息:开始执行Logic[" + name + "]，参数：" + param);
        if (path == null) {
            throw new ServiceException("业务逻辑未找到: " + name);
        }

        String source;
        source = resourceMapper.getString(ResourceType.LOGIC, name, path);
        // 附加用户注册的对象到业务逻辑中
        Map<String, Object> params = pluginMapper.getAllMap().entrySet().stream()
                .collect(Collectors.toMap(Map.Entry::getKey, object -> object.getValue().get("value")));
        // 执行源程序
        params.put("data", param);
        // 附加entityServer, sqlServer等对象到参数中
        params.put("log", LOG4J_LOGGER);
        params.put("entity", entityService);
        params.put("remoteEntity", remoteEntityService);
        params.put("sql", sqlService);
        params.put("prompt", promptService);
        params.put("redis", redisService);
        // 把LogicServer自身注册进去
        params.put("logic", this);
        long begin = System.currentTimeMillis();
        try {
            //判断数据源
            if (dataSourceName != null) {
                LOGGER.info("维护信息:Logic[" + name + "]使用数据源：" + dataSourceName);
                if (!Objects.equals(dataSourceName, DynamicDataSource.getDataSource())) {
                    DynamicDataSource.setDataSource(dataSourceName);
                }
                //使用新事务执行
                return logicNewTranService.runNewTranExpression(source, params);
            } else {
                DynamicDataSource.setDataSource(null);
            }
            //使用当前事务执行
            return runExpression(source, params);
        } finally {
            DynamicDataSource.clearDataSource();
            long end = System.currentTimeMillis();
            long time = end - begin;
            String engMessage = "维护信息:执行Logic[" + name + "]耗时:" + time;
            if (time >= 8000) {
                LOGGER.error(engMessage);
            } else if (time >= 4000) {
                LOGGER.warn(engMessage);
            } else {
                LOGGER.info(engMessage);
            }
        }
    }

    /**
     * 执行表达式
     *
     * @param source 表达式内容
     * @param params 参数
     * @return 执行结果
     */
    private Object runExpression(String source, Map<String, Object> params) {
        Object runResult;
        try {
            runResult = Expression.run(source, params);
            if (runResult == null) {
                runResult = "";
            }
        } catch (Exception ex) {
            throw buildServiceException(ex);
        }
        return runResult;
    }

    /**
     * 远程调用Logic
     *
     * @param serviceName 服务名
     * @param logicName   Logic名称
     * @param params      Logic参数
     * @return 执行结果
     */
    //    @GlobalTransactional
    public JSONObject remoteRun(String serviceName, String logicName, String params) {
        LOGGER.info("维护信息:开始远程调用" + serviceName + "的Logic[" + logicName + "]，参数：" + params);
        long begin = System.currentTimeMillis();
        try {
            R<Object> logicResult = remoteLogicService.run(serviceName, logicName, params, SecurityConstants.INNER);
            return logicResult.parseResponseJson();
        } finally {
            long end = System.currentTimeMillis();
            long time = end - begin;
            String engMessage = "维护信息:远程调用Logic[" + logicName + "]耗时:" + time;
            if (time >= 8000) {
                LOGGER.error(engMessage);
            } else if (time >= 4000) {
                LOGGER.warn(engMessage);
            } else {
                LOGGER.info(engMessage);
            }
        }
    }

    /**
     * 远程调用Logic
     *
     * @param serviceName 服务名
     * @param logicName   Logic名称
     * @param params      Logic JSON参数
     * @return 执行结果
     */
    //    @GlobalTransactional
    public JSONObject remoteRun(String serviceName, String logicName, JSONObject params) {
        return remoteRun(serviceName, logicName, params.toString());
    }

    /**
     * 构建业务异常
     *
     * @param ex 运行时异常
     * @return 业务异常
     */
    private ServiceException buildServiceException(Exception ex) {
        ServiceException serviceException = null;
        if (ex instanceof ServiceException e) {
            serviceException = e;
        } else {
            Throwable cause = ex.getCause();
            if (cause instanceof InvocationTargetException targetException) {
                if (targetException.getTargetException() instanceof ServiceException e) {
                    serviceException = e;
                }
            }
        }
        if (serviceException == null) {
            serviceException = new ServiceException();
            serviceException.setMessage("服务器内部错误");
        }
        if (StringUtils.isEmpty(serviceException.getDetailMessage())) {
            // 获取异常信息
            String stack = ExceptionLogService.getErrorInfo(ExceptionHelper.stackToString(ex), ExceptionLogService.BusinessType.LOGIC);
            serviceException.setDetailMessage(stack);
        }
        return serviceException;
    }
}
