package com.af.v4.system.common.logic.service;

import com.af.v4.system.api.RemoteEntityService;
import com.af.v4.system.api.RemoteLogicService;
import com.af.v4.system.common.core.constant.SecurityConstants;
import com.af.v4.system.common.core.domain.R;
import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.core.utils.StringUtils;
import com.af.v4.system.common.datasource.DynamicDataSource;
import com.af.v4.system.common.expression.Expression;
import com.af.v4.system.common.gpt.service.PromptService;
import com.af.v4.system.common.jpa.service.EntityService;
import com.af.v4.system.common.jpa.service.SqlService;
import com.af.v4.system.common.logic.utils.ExceptionHelper;
import com.af.v4.system.common.redis.RedisService;
import com.af.v4.system.common.resource.cache.CacheKey;
import com.af.v4.system.common.resource.core.ResourceIndexer;
import com.af.v4.system.common.resource.enums.ResourceType;
import com.af.v4.system.common.resource.mapper.LogicMapper;
import com.af.v4.system.common.resource.mapper.PluginMapper;
import org.apache.logging.log4j.LogManager;
import org.json.JSONArray;
import org.json.JSONObject;
import org.json.JSONTokener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.lang.reflect.InvocationTargetException;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicReference;

@Service
@Transactional(rollbackFor = Exception.class)
public class LogicService {
    private static final Logger LOGGER = LoggerFactory.getLogger(LogicService.class);
    private static final org.apache.logging.log4j.Logger LOG4J_LOGGER = LogManager.getLogger(LogicService.class);
    private final SqlService sqlService;
    private final LogicNewTranService logicNewTranService;
    private final RedisService redisService;
    private final RemoteLogicService remoteLogicService;
    private final RemoteEntityService remoteEntityService;
    private final EntityService entityService;
    private final PromptService promptService;
    private final LogicMapper logicMapper;
    @Qualifier("logicAsyncTaskExecutor")
    private final ThreadPoolTaskExecutor taskExecutor;

    public LogicService(SqlService sqlService, LogicNewTranService newLogicService, RedisService redisService, RemoteLogicService remoteLogicService, RemoteEntityService remoteEntityService, EntityService entityService, PromptService promptService, LogicMapper logicMapper, ThreadPoolTaskExecutor taskExecutor) {
        this.sqlService = sqlService;
        this.logicNewTranService = newLogicService;
        this.redisService = redisService;
        this.remoteLogicService = remoteLogicService;
        this.remoteEntityService = remoteEntityService;
        this.entityService = entityService;
        this.promptService = promptService;
        this.logicMapper = logicMapper;
        this.taskExecutor = taskExecutor;
    }

    /**
     * 执行Logic
     *
     * @param logicName      Logic名称
     * @param param          参数
     * @param dataSourceName 数据源名称
     * @return 执行结果
     */
    public Object run(String logicName, JSONObject param, String dataSourceName) {
        // 获取源程序内容
        LogicMapper.LogicResource<String> resource = logicMapper.getValue(logicName);
        String name = resource.getAlias();
        boolean cache = resource.getCache();
        long time = resource.getTime();
        LOGGER.info("维护信息:开始执行Logic[" + name + "]，cacheInfo：[" + cache + "," + time + "]，params：" + param);
        if (cache) {
            String paramsStr = new TreeMap<>(param.toMap()).toString();
            String cacheKey = CacheKey.getLogicDataCacheKey(logicName, paramsStr);
            Object result = redisService.get(cacheKey);
            if (result != null) {
                LOGGER.debug("维护信息:Logic[" + name + "]命中缓存");
                return result;
            }
            AtomicReference<Object> returnResult = new AtomicReference<>();
            String cacheLockKey = CacheKey.getLogicDataCacheLockKey(logicName, paramsStr);
            redisService.lock(cacheLockKey, 60, 300, () -> {
                Object newResult = redisService.get(cacheKey);
                if (newResult == null) {
                    newResult = action(resource, param, dataSourceName);
                    if (newResult != null) {
                        if (time > 0) {
                            redisService.set(cacheKey, newResult, time);
                            LOGGER.debug("维护信息:Logic[" + name + "]缓存成功，存活时间：" + time + "秒");
                        }
                    }
                } else {
                    LOGGER.debug("维护信息:Logic[" + name + "]命中缓存");
                }
                returnResult.set(newResult);
            });
            return returnResult.get();
        }
        return action(resource, param, dataSourceName);
    }

    public Object run(String name, JSONObject param) {
        return this.run(name, param, null);
    }

    public Object run(String name, String str) {
        return this.run(name, buildLogicParams(str));
    }

    /**
     * 异步执行Logic
     *
     * @param name  Logic名称
     * @param param 参数
     */
    public CompletableFuture<Object> runAsync(String name, JSONObject param) {
        return CompletableFuture.supplyAsync(() -> this.run(name, param), taskExecutor).exceptionally(throwable -> {
            if (throwable.getCause() instanceof ServiceException ex) {
                LOGGER.error("发生服务端异常.\n {}", ex.getDetailMessage());
            } else {
                LOGGER.error("发生其他异常.", throwable);
            }
            return null;
        });
    }

    public CompletableFuture<Object> runAsync(String name, String str) {
        return this.runAsync(name, buildLogicParams(str));
    }

    /**
     * 远程调用Logic
     *
     * @param serviceName 服务名
     * @param logicName   Logic名称
     * @param params      Logic参数
     * @return 执行结果
     */
    //    @GlobalTransactional
    public JSONObject remoteRun(String serviceName, String logicName, String params) {
        LOGGER.info("维护信息:开始远程调用" + serviceName + "的Logic[" + logicName + "]，参数：" + params);
        return (JSONObject) logicMonitor(logicName, () -> {
            R<Object> logicResult = remoteLogicService.run(serviceName, logicName, params, SecurityConstants.INNER);
            return logicResult.parseResponseJson();
        });
    }

    //    @GlobalTransactional
    public JSONObject remoteRun(String serviceName, String logicName, JSONObject params) {
        return remoteRun(serviceName, logicName, params.toString());
    }

    /**
     * 执行资源
     *
     * @param resource       资源
     * @param param          参数
     * @param dataSourceName 数据源
     * @return 执行结果
     */
    private Object action(LogicMapper.LogicResource<String> resource, JSONObject param, String dataSourceName) {
        String name = resource.getAlias();
        String path = resource.getPath();
        if (Optional.ofNullable(path).isEmpty()) {
            throw new ServiceException("业务逻辑未找到: " + name);
        }
        String source = ResourceIndexer.getString(ResourceType.LOGIC, name, path);

        // 附加用户注册的对象到业务逻辑中
        Map<String, Object> params = new HashMap<>(PluginMapper.pluginMap);
        params.put("data", param);
        params.put("log", LOG4J_LOGGER);
        params.put("entity", entityService);
        params.put("remoteEntity", remoteEntityService);
        params.put("sql", sqlService);
        params.put("prompt", promptService);
        params.put("redis", redisService);
        params.put("logic", this);

        return logicMonitor(name, () -> {
            String dataSource = dataSourceName;
            //判断数据源
            if (Optional.ofNullable(dataSource).isEmpty()) {
                Map<String, String> res = logicMapper.getRes(name);
                dataSource = res.get("dataSource");
            }
            if (Optional.ofNullable(dataSource).isPresent()) {
                LOGGER.info("维护信息:Logic[" + name + "]使用数据源：" + dataSource);
                if (!Objects.equals(dataSource, DynamicDataSource.getDataSource())) {
                    DynamicDataSource.setDataSource(dataSource);
                }
                //使用新事务执行
                return logicNewTranService.runNewTranExpression(source, params);
            } else {
                DynamicDataSource.setDataSource(null);
            }
            //使用当前事务执行
            return runExpression(source, params);
        }, DynamicDataSource::clearDataSource);
    }

    /**
     * 执行原生表达式
     *
     * @param source 表达式内容
     * @param params 参数
     * @return 执行结果
     */
    public Object runExpression(String source, Map<String, Object> params) {
        Object runResult;
        try {
            runResult = Optional.ofNullable(Expression.run(source, params)).orElse("");
        } catch (Exception ex) {
            throw buildServiceException(ex);
        }
        return runResult;
    }

    /**
     * 构建业务异常
     *
     * @param ex 运行时异常
     * @return 业务异常
     */
    private ServiceException buildServiceException(Exception ex) {
        ServiceException serviceException = null;
        if (ex instanceof ServiceException e) {
            serviceException = e;
        } else {
            Throwable cause = ex.getCause();
            if (cause instanceof InvocationTargetException targetException) {
                if (targetException.getTargetException() instanceof ServiceException e) {
                    serviceException = e;
                }
            }
        }
        if (Optional.ofNullable(serviceException).isEmpty()) {
            serviceException = new ServiceException();
            serviceException.setMessage("服务器内部错误");
        }
        if (StringUtils.isEmpty(serviceException.getDetailMessage())) {
            // 获取异常信息
            String stack = ExceptionLogService.getErrorInfo(ExceptionHelper.stackToString(ex), ExceptionLogService.BusinessType.LOGIC);
            serviceException.setDetailMessage(stack);
        }
        return serviceException;
    }

    /**
     * 构建Logic参数
     *
     * @param str 字符串参数
     * @return JSON格式参数
     */
    private JSONObject buildLogicParams(String str) {
        JSONObject param = null;
        // 把传递过来的参数，放到data里，以便跟entity，sql等对象区别开来
        // 判断传来的数据是否是XML格式
        str = str.trim();
        if (str.startsWith("<") && str.endsWith(">")) {
            // 传来数据为XML格式
            param = new JSONObject();
            param.put("xml", str);
        } else {
            // 传来数据为JSON格式
            Object json = new JSONTokener(str).nextValue();
            if (json instanceof JSONObject st) {
                param = st;
                if (!param.isNull("data")) {
                    Object dataParam = param.get("data");
                    if (dataParam instanceof JSONObject) {
                        param = (JSONObject) dataParam;
                        param.put("standardData", new JSONObject(str).toString());
                    }
                }
            } else if (json instanceof JSONArray ar) {
                param = new JSONObject();
                param.put("arr", ar);
            }
        }
        return param;
    }

    private Object logicMonitor(String name, Process process, Finally finallyFun) {
        long begin = System.currentTimeMillis();
        try {
            return process.apply();
        } finally {
            if (finallyFun != null) {
                finallyFun.apply();
            }
            long end = System.currentTimeMillis();
            long time = end - begin;
            String engMessage = "维护信息:Logic[" + name + "]处理耗时:" + time;
            if (time >= 8000) {
                LOGGER.error(engMessage);
            } else if (time >= 4000) {
                LOGGER.warn(engMessage);
            } else {
                LOGGER.info(engMessage);
            }
        }
    }

    private Object logicMonitor(String name, Process process) {
        return logicMonitor(name, process, null);
    }

    @FunctionalInterface
    interface Process {
        Object apply();
    }

    @FunctionalInterface
    interface Finally {
        void apply();
    }
}
