package com.af.v4.system.common.log.aspect;

import com.af.v4.system.common.core.utils.ServletUtils;
import com.af.v4.system.common.core.utils.StringUtils;
import com.af.v4.system.common.log.annotation.Log;
import com.af.v4.system.common.log.filter.PropertyPreExcludeFilter;
import com.af.v4.system.common.log.service.LogResolveService;
import com.alibaba.fastjson2.JSON;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import org.apache.commons.lang3.ArrayUtils;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpMethod;
import org.springframework.stereotype.Component;
import org.springframework.validation.BindingResult;
import org.springframework.web.multipart.MultipartFile;

import java.util.Collection;
import java.util.Map;

/**
 * 操作日志记录处理
 */
@Aspect
@Component
public class LogAspect {
    /**
     * 排除敏感属性字段
     */
    public static final String[] EXCLUDE_PROPERTIES = {"password", "oldPassword", "newPassword", "confirmPassword"};
    private static final Logger log = LoggerFactory.getLogger(LogAspect.class);

    private final LogResolveService logResolveService;

    public LogAspect(LogResolveService logResolveService) {
        this.logResolveService = logResolveService;
    }


    @Around(value = "@annotation(controllerLog)")
    public Object around(ProceedingJoinPoint joinPoint, Log controllerLog) throws Throwable {
        long beginTime = System.currentTimeMillis();
        try {
            Object result = joinPoint.proceed();
            long endTime = System.currentTimeMillis();
            handleLog(joinPoint, controllerLog, null, result, endTime - beginTime);
            return result;
        } catch (Throwable t) {
            long endTime = System.currentTimeMillis();
            handleLog(joinPoint, controllerLog, t, null, endTime - beginTime);
            throw t;
        }
    }

    protected void handleLog(final JoinPoint joinPoint, Log controllerLog, final Throwable e, Object jsonResult, long costTime) {
        logResolveService.resolveLog(
                joinPoint.getTarget().getClass().getName(),
                joinPoint.getSignature().getName(),
                controllerLog.businessType().name(),
                controllerLog.title(),
                controllerLog.operatorType().name(),
                null,
                costTime,
                e,
                (jsonObject) -> {
                    HttpServletRequest request = ServletUtils.getRequest();
                    if (request != null) {
                        // 处理设置注解上的参数
                        getControllerMethodDescription(joinPoint, controllerLog, jsonObject, jsonResult);
                    }
                }
        );
    }

    /**
     * 获取注解中对方法的描述信息 用于Controller层注解
     *
     * @param log     日志
     * @param operLog 操作日志
     */
    public void getControllerMethodDescription(JoinPoint joinPoint, Log log, JSONObject operLog, Object jsonResult) {
        // 是否需要保存request，参数和值
        if (log.isSaveRequestData()) {
            // 获取参数的信息，传入到数据库中。
            if (!operLog.has("f_oper_param")) {
                setRequestValue(joinPoint, operLog, log.excludeParamNames());
            }
        }
        // 是否需要保存response，参数和值
        if (log.isSaveResponseData() && StringUtils.isNotNull(jsonResult)) {
            if (!operLog.has("f_json_result")) {
                operLog.put("f_json_result", StringUtils.substring(jsonResult.toString(), 0, 2000));
            }
        }
    }

    /**
     * 获取请求的参数，放到log中
     *
     * @param operLog 操作日志
     */
    private void setRequestValue(JoinPoint joinPoint, JSONObject operLog, String[] excludeParamNames) {
        Map<?, ?> paramsMap = ServletUtils.getParamMap(ServletUtils.getRequest());
        String requestMethod = operLog.optString("f_request_method", null);
        if (requestMethod != null && StringUtils.isEmpty(paramsMap) && StringUtils.equalsAny(requestMethod, HttpMethod.PUT.name(), HttpMethod.POST.name(), HttpMethod.DELETE.name())) {
            String params = argsArrayToString(joinPoint.getArgs(), excludeParamNames);
            operLog.put("f_oper_param", StringUtils.substring(params, 0, 2000));
        } else {
            operLog.put("f_oper_param", StringUtils.substring(JSON.toJSONString(paramsMap, excludePropertyPreFilter(excludeParamNames)), 0, 2000));
        }
    }

    /**
     * 参数拼装
     */
    private String argsArrayToString(Object[] paramsArray, String[] excludeParamNames) {
        StringBuilder params = new StringBuilder();
        if (paramsArray != null) {
            for (Object o : paramsArray) {
                if (StringUtils.isNotNull(o) && !isFilterObject(o)) {
                    try {
                        String jsonObj = JSON.toJSONString(o, excludePropertyPreFilter(excludeParamNames));
                        params.append(jsonObj).append(" ");
                    } catch (Exception ignored) {
                    }
                }
            }
        }
        return params.toString().trim();
    }

    /**
     * 忽略敏感属性
     */
    public PropertyPreExcludeFilter excludePropertyPreFilter(String[] excludeParamNames) {
        return new PropertyPreExcludeFilter().addExcludes(ArrayUtils.addAll(EXCLUDE_PROPERTIES, excludeParamNames));
    }

    /**
     * 判断是否需要过滤的对象。
     *
     * @param o 对象信息。
     * @return 如果是需要过滤的对象，则返回true；否则返回false。
     */
    @SuppressWarnings("rawtypes")
    public boolean isFilterObject(final Object o) {
        Class<?> clazz = o.getClass();
        if (clazz.isArray()) {
            return clazz.getComponentType().isAssignableFrom(MultipartFile.class);
        } else if (Collection.class.isAssignableFrom(clazz)) {
            Collection collection = (Collection) o;
            for (Object value : collection) {
                return value instanceof MultipartFile;
            }
        } else if (Map.class.isAssignableFrom(clazz)) {
            Map map = (Map) o;
            for (Object value : map.entrySet()) {
                Map.Entry entry = (Map.Entry) value;
                return entry.getValue() instanceof MultipartFile;
            }
        }
        return o instanceof MultipartFile || o instanceof HttpServletRequest || o instanceof HttpServletResponse
                || o instanceof BindingResult;
    }
}
