package com.af.v4.system.common.log.nacos;

import com.af.v4.system.common.core.enums.EnvType;
import jakarta.annotation.Nonnull;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.core.Appender;
import org.apache.logging.log4j.core.Filter;
import org.apache.logging.log4j.core.LoggerContext;
import org.apache.logging.log4j.core.appender.ConsoleAppender;
import org.apache.logging.log4j.core.appender.RollingRandomAccessFileAppender;
import org.apache.logging.log4j.core.appender.rolling.CompositeTriggeringPolicy;
import org.apache.logging.log4j.core.appender.rolling.DefaultRolloverStrategy;
import org.apache.logging.log4j.core.appender.rolling.SizeBasedTriggeringPolicy;
import org.apache.logging.log4j.core.appender.rolling.TimeBasedTriggeringPolicy;
import org.apache.logging.log4j.core.appender.rolling.TriggeringPolicy;
import org.apache.logging.log4j.core.async.AsyncLoggerConfig;
import org.apache.logging.log4j.core.config.AppenderRef;
import org.apache.logging.log4j.core.config.Configuration;
import org.apache.logging.log4j.core.config.LoggerConfig;
import org.apache.logging.log4j.core.filter.ThresholdFilter;
import org.apache.logging.log4j.core.layout.PatternLayout;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.beans.factory.config.InstantiationAwareBeanPostProcessor;
import org.springframework.stereotype.Component;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;


@Component
public class Log4jInNacosConfigFactory implements InstantiationAwareBeanPostProcessor {
    final String NODE_NAME;
    private final Log4jInNacosConfig log4JInNacosConfig;
    private final List<Appender> appenderList = new ArrayList<>(3);
    private final AppenderRef[] appenderRefs = new AppenderRef[3];
    Configuration config;
    @Value("${spring.profiles.active}")
    private String envType;

    public Log4jInNacosConfigFactory(Log4jInNacosConfig log4JInNacosConfig) {
        this.log4JInNacosConfig = log4JInNacosConfig;
        NODE_NAME = System.getenv("NODE_NAME");
    }

    private void addLoggerConfig(Configuration config, String name, Level level) {
        AsyncLoggerConfig loggerConfig = (AsyncLoggerConfig) AsyncLoggerConfig
                .newAsyncBuilder().withLoggerName(name).withLevel(level).withConfig(config).withRefs(appenderRefs).build();
        config.addLogger(name, loggerConfig);
    }

    public void createConfiguration() {
        LoggerContext context = (LoggerContext) LogManager.getContext(false);
        config = context.getConfiguration();

        PatternLayout layout = PatternLayout.newBuilder()
                .withCharset(StandardCharsets.UTF_8)
                .withPattern(log4JInNacosConfig.getPattern())
                .build();

        EnvType envTypeEnum = EnvType.toType(envType);

        // Console Appender
        Appender consoleAppender = ConsoleAppender.newBuilder()
                .setName("Console")
                .setFilter(ThresholdFilter.createFilter(Level.getLevel(log4JInNacosConfig.getConsoleLevel(envTypeEnum)), null, null))
                .setLayout(layout)
                .build();

        String logPath;
        String historyLogPath;
        String errorLogPath;
        String historyErrorLogPath;
        if (NODE_NAME != null) {
            logPath = log4JInNacosConfig.getLogFilePath() + "/" + NODE_NAME + "/log.log";
            historyLogPath = log4JInNacosConfig.getLogFilePath() + "/" + NODE_NAME + "/dailyLog/log_%d{yyyy-MM-dd_HH}.log";
            errorLogPath = log4JInNacosConfig.getLogFilePath() + "/" + NODE_NAME + "/error.log";
            historyErrorLogPath = log4JInNacosConfig.getLogFilePath() + "/" + NODE_NAME + "/dailyError/error_%d{yyyy-MM-dd}.log";
        } else {
            logPath = log4JInNacosConfig.getLogFilePath() + "/log.log";
            historyLogPath = log4JInNacosConfig.getLogFilePath() + "/dailyLog/log_%d{yyyy-MM-dd_HH}.log";
            errorLogPath = log4JInNacosConfig.getLogFilePath() + "/error.log";
            historyErrorLogPath = log4JInNacosConfig.getLogFilePath() + "/dailyError/error_%d{yyyy-MM-dd}.log";
        }
        // Log FileAppender
        TriggeringPolicy timePolicy = TimeBasedTriggeringPolicy.newBuilder().withInterval(1).withModulate(true).build();
        TriggeringPolicy sizePolicy = SizeBasedTriggeringPolicy.createPolicy("100MB");
        TriggeringPolicy compositePolicy = CompositeTriggeringPolicy.createPolicy(timePolicy, sizePolicy);

        Appender rollingFileInfoAppender = RollingRandomAccessFileAppender.newBuilder()
                .setName("RollingFileInfo")
                .withFileName(logPath)
                .setImmediateFlush(false)
                .withFilePattern(historyLogPath)
                .setFilter(ThresholdFilter.createFilter(Level.DEBUG, Filter.Result.ACCEPT, Filter.Result.DENY))
                .setLayout(layout)
                .withPolicy(compositePolicy)
                .withStrategy(DefaultRolloverStrategy.newBuilder().withMax(log4JInNacosConfig.getMaxFileNum()).build())
                .build();

        // Error FileAppender
        Appender rollingFileErrorAppender = RollingRandomAccessFileAppender.newBuilder()
                .setName("RollingFileError")
                .withFileName(errorLogPath)
                .setImmediateFlush(false)
                .withFilePattern(historyErrorLogPath)
                .setFilter(ThresholdFilter.createFilter(Level.WARN, Filter.Result.ACCEPT, Filter.Result.DENY))
                .setLayout(layout)
                .withPolicy(TimeBasedTriggeringPolicy.newBuilder().withInterval(1).withModulate(true).build())
                .build();

        appenderList.add(consoleAppender);
        appenderList.add(rollingFileInfoAppender);
        appenderList.add(rollingFileErrorAppender);

        LoggerConfig rootConfig = config.getRootLogger();
        rootConfig.removeAppender(consoleAppender.getName());
        rootConfig.setLevel(Level.WARN);

        for (int i = 0; i < appenderList.size(); i++) {
            Appender appender = appenderList.get(i);
            appender.start();
            appenderRefs[i] = AppenderRef.createAppenderRef(appender.getName(), null, null);
            rootConfig.addAppender(appender, null, null);
        }
        addLoggerConfig(config, "com.af", envTypeEnum == EnvType.PROD ? Level.INFO : Level.DEBUG);
        addLoggerConfig(config, "org.springframework", Level.ERROR);
        addLoggerConfig(config, "com.microsoft.sqlserver", Level.ERROR);
        addLoggerConfig(config, "com.clickhouse", Level.ERROR);
        // 避免hibernate6因mapping文件过时导致的警告
        addLoggerConfig(config, "org.hibernate", Level.ERROR);

        context.updateLoggers();
    }

    @Override
    public Object postProcessBeforeInstantiation(@Nonnull Class<?> beanClass, String beanName) throws BeansException {
        // 项目启动初始化时加载log4j2配置
        if (beanName.equals("tomcatServletWebServerFactory") || beanName.equals("nettyReactiveWebServerFactory")) {
            createConfiguration();
        }
        return InstantiationAwareBeanPostProcessor.super.postProcessBeforeInstantiation(beanClass, beanName);
    }
}
