package com.af.v4.system.common.log.aspect;

import com.af.v4.system.api.model.LoginUser;
import com.af.v4.system.common.core.context.GlobalVisualThreadContext;
import com.af.v4.system.common.core.domain.R;
import com.af.v4.system.common.core.enums.EnvType;
import com.af.v4.system.common.core.exception.LogicException;
import com.af.v4.system.common.core.proxy.liuli.ILiuLiMonitorServiceProxy;
import com.af.v4.system.common.core.service.ApplicationService;
import com.af.v4.system.common.core.utils.ServletUtils;
import com.af.v4.system.common.core.utils.StringUtils;
import com.af.v4.system.common.core.utils.ip.IpUtils;
import com.af.v4.system.common.log.annotation.Log;
import com.af.v4.system.common.log.enums.BusinessStatus;
import com.af.v4.system.common.log.filter.PropertyPreExcludeFilter;
import com.af.v4.system.common.log.service.AsyncLogService;
import com.af.v4.system.common.security.utils.SecurityUtils;
import com.alibaba.fastjson2.JSON;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import org.apache.commons.lang3.ArrayUtils;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.*;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpMethod;
import org.springframework.stereotype.Component;
import org.springframework.validation.BindingResult;
import org.springframework.web.multipart.MultipartFile;

import java.util.Collection;
import java.util.Map;

/**
 * 操作日志记录处理
 */
@Aspect
@Component
public class LogAspect {
    /**
     * 排除敏感属性字段
     */
    public static final String[] EXCLUDE_PROPERTIES = {"password", "oldPassword", "newPassword", "confirmPassword"};
    private static final Logger log = LoggerFactory.getLogger(LogAspect.class);

    private final AsyncLogService asyncLogService;

    private final ILiuLiMonitorServiceProxy liuLiMonitorService;

    private final ApplicationService applicationService;

    public LogAspect(AsyncLogService asyncLogService, ILiuLiMonitorServiceProxy liuLiMonitorService, ApplicationService applicationService) {
        this.asyncLogService = asyncLogService;
        this.liuLiMonitorService = liuLiMonitorService;
        this.applicationService = applicationService;
    }


    @Around(value = "@annotation(controllerLog)")
    public Object around(ProceedingJoinPoint joinPoint, Log controllerLog) throws Throwable {
        long beginTime = System.currentTimeMillis();
        try {
            Object result = joinPoint.proceed();
            long endTime = System.currentTimeMillis();
            handleLog(joinPoint, controllerLog, null, result, endTime - beginTime);
            return result;
        } catch (Throwable t) {
            long endTime = System.currentTimeMillis();
            handleLog(joinPoint, controllerLog, t, null, endTime - beginTime);
            throw t;
        }
    }

    protected void handleLog(final JoinPoint joinPoint, Log controllerLog, final Throwable e, Object jsonResult, long costTime) {
        // 本地环境不存储操作日志
        if (applicationService.getEnvType() == EnvType.LOCAL) {
            return;
        }
        LoginUser loginUser = SecurityUtils.getLoginUser();
        if (loginUser == null && e == null) {
            // 如果没有登录，且没有发生异常时，不存储操作日志
            return;
        }
        JSONObject operLog = new JSONObject();
        operLog.put("f_oper_ip", IpUtils.getIpAddr());
        if (e == null) {
            operLog.put("f_status", BusinessStatus.SUCCESS.ordinal());
        } else {
            operLog.put("f_status", BusinessStatus.FAIL.ordinal());
            if (e instanceof LogicException logicException) {
                operLog.put("f_method", logicException.getLogicPath());
                operLog.put("f_title", "Logic异常:" + logicException.getTitle());
                operLog.put("f_oper_param", logicException.getLogicParams());
                operLog.put("f_error_msg", logicException.getStack());
            } else {
                operLog.put("f_error_msg", e.getMessage());
            }
        }
        if (loginUser != null) {
            String username = loginUser.getUsername();
            if (StringUtils.isNotBlank(username)) {
                operLog.put("f_oper_name", username);
            }
        }
        if (!operLog.has("f_method")) {
            // 设置方法名称
            String className = joinPoint.getTarget().getClass().getName();
            String methodName = joinPoint.getSignature().getName();
            operLog.put("f_method", className + "." + methodName + "()");
        }
        HttpServletRequest request = ServletUtils.getRequest();
        if (request != null) {
            // 请求地址
            operLog.put("f_oper_url", StringUtils.substring(request.getRequestURI(), 0, 255));
            // 请求方式
            operLog.put("f_request_method", request.getMethod());
            // 处理设置注解上的参数
            getControllerMethodDescription(joinPoint, controllerLog, operLog, jsonResult);
        }
        // 设置action动作
        if (!operLog.has("f_business_type")) {
            operLog.put("f_business_type", controllerLog.businessType().ordinal());
        }
        // 设置标题
        if (!operLog.has("f_title")) {
            operLog.put("f_title", controllerLog.title());
        }
        // 设置操作人类别
        if (!operLog.has("f_operator_type")) {
            operLog.put("f_operator_type", controllerLog.operatorType().ordinal());
        }
        // 设置消耗时间
        operLog.put("f_cost_time", costTime);
        // 处理关键字
        if (operLog.has("f_error_msg")) {
            operLog.put("f_error_msg", StringUtils.substring(operLog.getString("f_error_msg").replace("'", "''"), 0, 8000));
        }
        if (operLog.has("f_oper_param")) {
            operLog.put("f_oper_param", StringUtils.substring(operLog.getString("f_oper_param").replace("'", "''"), 0, 8000));
        }
        if (operLog.has("f_title")) {
            operLog.put("f_title", StringUtils.substring(operLog.getString("f_title").replace("'", "''"), 0, 300));
        }
        if (operLog.has("f_json_result")) {
            operLog.put("f_json_result", StringUtils.substring(operLog.getString("f_json_result").replace("'", "''"), 0, 2000));
        }
        // 保存数据库
        asyncLogService.saveSysLog(operLog.toString());
        // 避免uploadErrorLog报错导致循环调用
        if (e instanceof LogicException ex && ex.getLogicName().equals("uploadErrorLog")) {
            return;
        }
        if (e != null) {
            liuLiMonitorService.uploadLogicErrorLog(operLog);
        }
    }

    /**
     * 获取注解中对方法的描述信息 用于Controller层注解
     *
     * @param log     日志
     * @param operLog 操作日志
     */
    public void getControllerMethodDescription(JoinPoint joinPoint, Log log, JSONObject operLog, Object jsonResult) {
        // 是否需要保存request，参数和值
        if (log.isSaveRequestData()) {
            // 获取参数的信息，传入到数据库中。
            if (!operLog.has("f_oper_param")) {
                setRequestValue(joinPoint, operLog, log.excludeParamNames());
            }
        }
        // 是否需要保存response，参数和值
        if (log.isSaveResponseData() && StringUtils.isNotNull(jsonResult)) {
            if (!operLog.has("f_json_result")) {
                operLog.put("f_json_result", StringUtils.substring(jsonResult.toString(), 0, 2000));
            }
        }
    }

    /**
     * 获取请求的参数，放到log中
     *
     * @param operLog 操作日志
     */
    private void setRequestValue(JoinPoint joinPoint, JSONObject operLog, String[] excludeParamNames) {
        Map<?, ?> paramsMap = ServletUtils.getParamMap(ServletUtils.getRequest());
        String requestMethod = operLog.getString("f_request_method");
        if (StringUtils.isEmpty(paramsMap) && StringUtils.equalsAny(requestMethod, HttpMethod.PUT.name(), HttpMethod.POST.name(), HttpMethod.DELETE.name())) {
            String params = argsArrayToString(joinPoint.getArgs(), excludeParamNames);
            operLog.put("f_oper_param", StringUtils.substring(params, 0, 2000));
        } else {
            operLog.put("f_oper_param", StringUtils.substring(JSON.toJSONString(paramsMap, excludePropertyPreFilter(excludeParamNames)), 0, 2000));
        }
    }

    /**
     * 参数拼装
     */
    private String argsArrayToString(Object[] paramsArray, String[] excludeParamNames) {
        StringBuilder params = new StringBuilder();
        if (paramsArray != null) {
            for (Object o : paramsArray) {
                if (StringUtils.isNotNull(o) && !isFilterObject(o)) {
                    try {
                        String jsonObj = JSON.toJSONString(o, excludePropertyPreFilter(excludeParamNames));
                        params.append(jsonObj).append(" ");
                    } catch (Exception ignored) {
                    }
                }
            }
        }
        return params.toString().trim();
    }

    /**
     * 忽略敏感属性
     */
    public PropertyPreExcludeFilter excludePropertyPreFilter(String[] excludeParamNames) {
        return new PropertyPreExcludeFilter().addExcludes(ArrayUtils.addAll(EXCLUDE_PROPERTIES, excludeParamNames));
    }

    /**
     * 判断是否需要过滤的对象。
     *
     * @param o 对象信息。
     * @return 如果是需要过滤的对象，则返回true；否则返回false。
     */
    @SuppressWarnings("rawtypes")
    public boolean isFilterObject(final Object o) {
        Class<?> clazz = o.getClass();
        if (clazz.isArray()) {
            return clazz.getComponentType().isAssignableFrom(MultipartFile.class);
        } else if (Collection.class.isAssignableFrom(clazz)) {
            Collection collection = (Collection) o;
            for (Object value : collection) {
                return value instanceof MultipartFile;
            }
        } else if (Map.class.isAssignableFrom(clazz)) {
            Map map = (Map) o;
            for (Object value : map.entrySet()) {
                Map.Entry entry = (Map.Entry) value;
                return entry.getValue() instanceof MultipartFile;
            }
        }
        return o instanceof MultipartFile || o instanceof HttpServletRequest || o instanceof HttpServletResponse
                || o instanceof BindingResult;
    }
}
