package com.af.v4.system.common.log.aspect;

import com.af.v4.system.common.core.exception.LogicException;
import com.af.v4.system.common.core.utils.ServletUtils;
import com.af.v4.system.common.core.utils.StringUtils;
import com.af.v4.system.common.core.utils.ip.IpUtils;
import com.af.v4.system.common.liuli.service.LiuLiMonitorService;
import com.af.v4.system.common.log.annotation.Log;
import com.af.v4.system.common.log.enums.BusinessStatus;
import com.af.v4.system.common.log.filter.PropertyPreExcludeFilter;
import com.af.v4.system.common.log.service.AsyncLogService;
import com.af.v4.system.common.security.utils.SecurityUtils;
import com.alibaba.fastjson2.JSON;
import org.apache.commons.lang3.ArrayUtils;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.AfterReturning;
import org.aspectj.lang.annotation.AfterThrowing;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.NamedThreadLocal;
import org.springframework.http.HttpMethod;
import org.springframework.stereotype.Component;
import org.springframework.validation.BindingResult;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Collection;
import java.util.Map;

/**
 * 操作日志记录处理
 */
@Aspect
@Component
public class LogAspect {
    /**
     * 排除敏感属性字段
     */
    public static final String[] EXCLUDE_PROPERTIES = {"password", "oldPassword", "newPassword", "confirmPassword"};
    private static final Logger log = LoggerFactory.getLogger(LogAspect.class);
    /**
     * 计算操作消耗时间
     */
    private static final ThreadLocal<Long> TIME_THREADLOCAL = new NamedThreadLocal<>("Cost Time");

    private final AsyncLogService asyncLogService;

    private final LiuLiMonitorService liuLiMonitorService;

    public LogAspect(AsyncLogService asyncLogService, LiuLiMonitorService liuLiMonitorService) {
        this.asyncLogService = asyncLogService;
        this.liuLiMonitorService = liuLiMonitorService;
    }

    /**
     * 处理请求前执行
     */
    @Before(value = "@annotation(controllerLog)")
    public void boBefore(JoinPoint joinPoint, Log controllerLog) {
        TIME_THREADLOCAL.set(System.currentTimeMillis());
    }

    /**
     * 处理完请求后执行
     *
     * @param joinPoint 切点
     */
    @AfterReturning(pointcut = "@annotation(controllerLog)", returning = "jsonResult")
    public void doAfterReturning(JoinPoint joinPoint, Log controllerLog, Object jsonResult) {
        handleLog(joinPoint, controllerLog, null, jsonResult);
    }

    /**
     * 拦截异常操作
     *
     * @param joinPoint 切点
     * @param e         异常
     */
    @AfterThrowing(value = "@annotation(controllerLog)", throwing = "e")
    public void doAfterThrowing(JoinPoint joinPoint, Log controllerLog, Exception e) {
        handleLog(joinPoint, controllerLog, e, null);
    }

    protected void handleLog(final JoinPoint joinPoint, Log controllerLog, final Exception e, Object jsonResult) {
        try {
            // *========数据库日志=========*//
            JSONObject operLog = new JSONObject();
            operLog.put("f_status", BusinessStatus.SUCCESS.ordinal());
            // 请求的地址
            String ip = IpUtils.getIpAddr();
            operLog.put("f_oper_ip", ip);
            operLog.put("f_oper_url", StringUtils.substring(ServletUtils.getRequest().getRequestURI(), 0, 255));
            String username = SecurityUtils.getUsername();
            if (StringUtils.isNotBlank(username)) {
                operLog.put("f_oper_name", username);
            }

            if (e != null) {
                operLog.put("f_status", BusinessStatus.FAIL.ordinal());
                if (e instanceof LogicException logicException) {
                    operLog.put("f_method", logicException.getLogicPath());
                    operLog.put("f_title", "Logic异常:" + logicException.getTitle());
                    operLog.put("f_oper_param", logicException.getLogicParams());
                    operLog.put("f_error_msg", logicException.getStack());
                } else {
                    operLog.put("f_error_msg", e.getMessage());
                }
            }
            if (!operLog.has("f_method")) {
                // 设置方法名称
                String className = joinPoint.getTarget().getClass().getName();
                String methodName = joinPoint.getSignature().getName();
                operLog.put("f_method", className + "." + methodName + "()");
            }

            // 设置请求方式
            operLog.put("f_request_method", ServletUtils.getRequest().getMethod());
            // 处理设置注解上的参数
            getControllerMethodDescription(joinPoint, controllerLog, operLog, jsonResult);
            // 设置消耗时间
            operLog.put("f_cost_time", System.currentTimeMillis() - TIME_THREADLOCAL.get());
            // 处理关键字
            if (operLog.has("f_error_msg")) {
                operLog.put("f_error_msg", StringUtils.substring(operLog.getString("f_error_msg").replace("'", "''"), 0, 8000));
            }
            if (operLog.has("f_oper_param")) {
                operLog.put("f_oper_param", StringUtils.substring(operLog.getString("f_oper_param").replace("'", "''"), 0, 8000));
            }
            if (operLog.has("f_title")) {
                operLog.put("f_title", StringUtils.substring(operLog.getString("f_title").replace("'", "''"), 0, 800));
            }
            if (operLog.has("f_json_result")) {
                operLog.put("f_json_result", StringUtils.substring(operLog.getString("f_json_result").replace("'", "''"), 0, 2000));
            }
            // 保存数据库
            asyncLogService.saveSysLog(operLog.toString());
            if (e instanceof LogicException) {
                log.info("推送错误日志到琉璃中心");
                liuLiMonitorService.uploadLogicErrorLog(operLog);
            }
        } catch (Exception exp) {
            // 记录本地异常日志
            log.error("==前置通知异常==");
            log.error("异常信息:{}", exp.getMessage());
            exp.printStackTrace();
        } finally {
            TIME_THREADLOCAL.remove();
        }
    }

    /**
     * 获取注解中对方法的描述信息 用于Controller层注解
     *
     * @param log     日志
     * @param operLog 操作日志
     */
    public void getControllerMethodDescription(JoinPoint joinPoint, Log log, JSONObject operLog, Object jsonResult) {
        // 设置action动作
        if (!operLog.has("f_business_type")) {
            operLog.put("f_business_type", log.businessType().ordinal());
        }
        // 设置标题
        if (!operLog.has("f_title")) {
            operLog.put("f_title", log.title());
        }
        if (!operLog.has("f_operator_type")) {
            // 设置操作人类别
            operLog.put("f_operator_type", log.operatorType().ordinal());
        }
        // 是否需要保存request，参数和值
        if (log.isSaveRequestData()) {
            // 获取参数的信息，传入到数据库中。
            if (!operLog.has("f_oper_param")) {
                setRequestValue(joinPoint, operLog, log.excludeParamNames());
            }
        }
        // 是否需要保存response，参数和值
        if (log.isSaveResponseData() && StringUtils.isNotNull(jsonResult)) {
            if (!operLog.has("f_json_result")) {
                operLog.put("f_json_result", StringUtils.substring(new JSONObject(jsonResult).toString(), 0, 2000));
            }
        }
    }

    /**
     * 获取请求的参数，放到log中
     *
     * @param operLog 操作日志
     */
    private void setRequestValue(JoinPoint joinPoint, JSONObject operLog, String[] excludeParamNames) {
        String requestMethod = operLog.getString("f_request_method");
        Map<?, ?> paramsMap = ServletUtils.getParamMap(ServletUtils.getRequest());
        if (StringUtils.isEmpty(paramsMap) && (HttpMethod.PUT.name().equals(requestMethod) || HttpMethod.POST.name().equals(requestMethod))) {
            String params = argsArrayToString(joinPoint.getArgs(), excludeParamNames);
            operLog.put("f_oper_param", StringUtils.substring(params, 0, 2000));
        } else {
            operLog.put("f_oper_param", StringUtils.substring(JSON.toJSONString(paramsMap, excludePropertyPreFilter(excludeParamNames)), 0, 2000));
        }
    }

    /**
     * 参数拼装
     */
    private String argsArrayToString(Object[] paramsArray, String[] excludeParamNames) {
        StringBuilder params = new StringBuilder();
        if (paramsArray != null) {
            for (Object o : paramsArray) {
                if (StringUtils.isNotNull(o) && !isFilterObject(o)) {
                    try {
                        String jsonObj = JSON.toJSONString(o, excludePropertyPreFilter(excludeParamNames));
                        params.append(jsonObj).append(" ");
                    } catch (Exception ignored) {
                    }
                }
            }
        }
        return params.toString().trim();
    }

    /**
     * 忽略敏感属性
     */
    public PropertyPreExcludeFilter excludePropertyPreFilter(String[] excludeParamNames) {
        return new PropertyPreExcludeFilter().addExcludes(ArrayUtils.addAll(EXCLUDE_PROPERTIES, excludeParamNames));
    }

    /**
     * 判断是否需要过滤的对象。
     *
     * @param o 对象信息。
     * @return 如果是需要过滤的对象，则返回true；否则返回false。
     */
    @SuppressWarnings("rawtypes")
    public boolean isFilterObject(final Object o) {
        Class<?> clazz = o.getClass();
        if (clazz.isArray()) {
            return clazz.getComponentType().isAssignableFrom(MultipartFile.class);
        } else if (Collection.class.isAssignableFrom(clazz)) {
            Collection collection = (Collection) o;
            for (Object value : collection) {
                return value instanceof MultipartFile;
            }
        } else if (Map.class.isAssignableFrom(clazz)) {
            Map map = (Map) o;
            for (Object value : map.entrySet()) {
                Map.Entry entry = (Map.Entry) value;
                return entry.getValue() instanceof MultipartFile;
            }
        }
        return o instanceof MultipartFile || o instanceof HttpServletRequest || o instanceof HttpServletResponse
                || o instanceof BindingResult;
    }
}
