package com.af.v4.system.common.liuli.monitor.service;

import com.af.v4.system.api.RemoteLiuLiService;
import com.af.v4.system.api.factory.DynamicFeignClientFactory;
import com.af.v4.system.common.core.constant.CacheConstants;
import com.af.v4.system.common.core.constant.HttpStatus;
import com.af.v4.system.common.core.constant.SecurityConstants;
import com.af.v4.system.common.core.constant.ServiceNameConstants;
import com.af.v4.system.common.core.domain.R;
import com.af.v4.system.common.core.enums.EnvType;
import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.core.proxy.liuli.ILiuLiMonitorServiceProxy;
import com.af.v4.system.common.core.proxy.logic.ILogicServiceProxy;
import com.af.v4.system.common.core.service.ApplicationService;
import com.af.v4.system.common.core.utils.SpringUtils;
import com.af.v4.system.common.liuli.api.LiuLiApi;
import com.af.v4.system.common.liuli.config.LiuLiClientConfig;
import com.af.v4.system.common.liuli.utils.enums.LiuLiClientOperationalModeEnum;
import com.af.v4.system.common.plugins.date.DateTools;
import com.af.v4.system.common.plugins.http.RestAsyncTools;
import com.af.v4.system.common.plugins.http.config.CloseHttpLogHandler;
import com.af.v4.system.common.plugins.http.core.response.EmptyAsyncResponse;
import com.af.v4.system.common.plugins.http.core.response.StandardAsyncResponse;
import com.af.v4.system.common.redis.RedisService;
import jakarta.annotation.PostConstruct;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.Primary;
import org.springframework.stereotype.Service;

import java.util.Map;

/**
 * 琉璃-监控服务
 *
 * @author Mr.river
 * @since 1.1.6
 */
@Primary
@Service
public class LiuLiMonitorService implements ILiuLiMonitorServiceProxy {

    private static final Logger LOGGER = LoggerFactory.getLogger(LiuLiMonitorService.class);


    private final ApplicationService applicationService;
    private final LiuLiClientConfig liuLiClientConfig;
    private final RedisService redisService;
    private final DynamicFeignClientFactory dynamicFeignClientFactory;

    public LiuLiMonitorService(ApplicationService applicationService, LiuLiClientConfig liuLiClientConfig, RedisService redisService, DynamicFeignClientFactory dynamicFeignClientFactory) {
        this.applicationService = applicationService;
        this.liuLiClientConfig = liuLiClientConfig;
        this.redisService = redisService;
        this.dynamicFeignClientFactory = dynamicFeignClientFactory;
    }

    /**
     * 获取应用的独占Key
     *
     * @param namespaceName 命名空间名称
     * @return key
     */
    public static String geApplicationLockKey(String namespaceName) {
        return CacheConstants.APPLICATION_UPLOAD_INFO_KEY + namespaceName;
    }

    @PostConstruct
    public void init() {
        if (liuLiClientConfig.getLiuLiClientOperationalMode() == LiuLiClientOperationalModeEnum.OFFLINE) {
            return;
        }
        EnvType envType = applicationService.getEnvType();
        if (envType == EnvType.LOCAL) {
            checkV4Version();
        } else if (envType == EnvType.PROD) {
            uploadApplicationInfo();
        }
    }

    /**
     * 上传生产环境应用信息
     */
    private void uploadApplicationInfo() {
        String applicationName = applicationService.getApplicationName();
        if (applicationName.equals("af-runtime")) {
            return;
        }
        String applicationLockKey = geApplicationLockKey(applicationName);
        try {
            redisService.lock(applicationLockKey, 60, 300, () -> {
                //调用琉璃中心需传递的参数
                JSONObject params = new JSONObject()
                        .put("tenantUUID", applicationService.getTenantName())
                        .put("name", applicationName)
                        .put("coreVersion", ApplicationService.getSystemV4Version())
                        .put("applicationVersion", ApplicationService.getAppVersion())
                        .put("lastStartDate", DateTools.getNow2());

                StandardAsyncResponse.SuccessHandler handler = response -> {
                    R<JSONObject> result = R.build(new JSONObject(response));
                    if (!result.isSuccess()) {
                        LOGGER.warn("应用信息上传失败:{}", result.getMsg());
                    }
                };
                if (applicationName.equals(ServiceNameConstants.LIULI_SERVICE)) {
                    // 琉璃自身调用
                    JSONObject response = new JSONObject();
                    try {
                        Object localResult = SpringUtils.getBean(ILogicServiceProxy.class).run("saveApplicationInfo", params);
                        response.put("code", HttpStatus.SUCCESS);
                        response.put("data", localResult);
                    } catch (ServiceException e) {
                        response.put("code", e.getCode());
                        response.put("msg", e.getMessage());
                    }
                    handler.run(response.toString());
                } else if (liuLiClientConfig.getLiuLiClientOperationalMode() == LiuLiClientOperationalModeEnum.INTRANET) {
                    RemoteLiuLiService remoteLiuLiService = (RemoteLiuLiService) dynamicFeignClientFactory.getFeignClient(RemoteLiuLiService.class, ServiceNameConstants.LIULI_SERVICE);
                    R<Map<String, Object>> remoteResult = remoteLiuLiService.saveApplicationInfo(params.toString(), SecurityConstants.INNER);
                    handler.run(remoteResult.parseResponseJson().toString());
                } else {
                    try (CloseHttpLogHandler logHandler = CloseHttpLogHandler.build()) {
                        logHandler.apply();
                        RestAsyncTools.post(LiuLiApi.SAVE_APPLICATION_INFO_PATH, params.toString(), null,
                                new StandardAsyncResponse(handler, e -> LOGGER.error("应用信息上传失败:{}", e.getMessage())));
                    }
                }
            });
        } catch (InterruptedException ignored) {
        }
    }

    /**
     * 上报错误日志
     */
    @Override
    public void uploadLogicErrorLog(JSONObject operLog) {
        if (!liuLiClientConfig.getUploadErrorLog()) {
            return;
        }
        if (liuLiClientConfig.getLiuLiClientOperationalMode() == LiuLiClientOperationalModeEnum.OFFLINE) {
            return;
        }
        EnvType envType = applicationService.getEnvType();
        if (envType == EnvType.LOCAL) {
            return;
        }
        //调用琉璃中心需传递的参数
        JSONObject jsonObject = new JSONObject();
        jsonObject.put("tenantUUID", applicationService.getTenantName());
        jsonObject.put("applicationName", applicationService.getApplicationName());
        jsonObject.put("applicationVersion", ApplicationService.getAppVersion());
        jsonObject.put("coreVersion", ApplicationService.getSystemV4Version());
        jsonObject.put("osType", ApplicationService.getOSType());
        jsonObject.put("environment", applicationService.getEnvType().getValue());
        jsonObject.put("method", operLog.opt("f_method"));
        jsonObject.put("requestMethod", operLog.opt("f_request_method"));
        jsonObject.put("operIp", operLog.opt("f_oper_ip"));
        jsonObject.put("operUrl", operLog.opt("f_oper_url"));
        jsonObject.put("operName", operLog.opt("f_oper_name"));
        jsonObject.put("operParam", operLog.opt("f_oper_param"));
        jsonObject.put("errorMsg", operLog.opt("f_error_msg"));
        jsonObject.put("businessTitle", operLog.opt("f_title"));
        jsonObject.put("businessType", operLog.opt("f_business_type"));
        jsonObject.put("operatorType", operLog.opt("f_operator_type"));
        jsonObject.put("costTime", operLog.opt("f_cost_time"));

        try (CloseHttpLogHandler logHandler = CloseHttpLogHandler.build()) {
            logHandler.apply();
            if (applicationService.getApplicationName().equals(ServiceNameConstants.LIULI_SERVICE)) {
                // 琉璃自身调用
                try {
                    SpringUtils.getBean(ILogicServiceProxy.class).run("uploadErrorLog", jsonObject);
                } catch (ServiceException ignored) {
                }
            } else if (liuLiClientConfig.getLiuLiClientOperationalMode() == LiuLiClientOperationalModeEnum.INTRANET) {
                RemoteLiuLiService remoteLiuLiService = (RemoteLiuLiService) dynamicFeignClientFactory.getFeignClient(RemoteLiuLiService.class, ServiceNameConstants.LIULI_SERVICE);
                remoteLiuLiService.uploadErrorLog(jsonObject.toString(), SecurityConstants.INNER);
            } else {
                RestAsyncTools.post(LiuLiApi.UPLOAD_ERROR_LOG_PATH, jsonObject.toString(), null, new EmptyAsyncResponse());
            }
        }
    }

    /**
     * 检查琉璃架构版本
     */
    private void checkV4Version() {
        String nowVersion = ApplicationService.getSystemV4Version();
        if (nowVersion == null) {
            return;
        }
        LOGGER.info("\uD83C\uDF89\uD83C\uDF89\uD83C\uDF89欢迎使用琉璃架构，版本：V{}", nowVersion);
        String applicationName = applicationService.getApplicationName();
        if (applicationName.equals("af-runtime")) {
            return;
        }

        JSONObject params = new JSONObject().put("version", nowVersion);

        StandardAsyncResponse.SuccessHandler handler = response -> {
            R<JSONObject> result = R.build(new JSONObject(response));
            if (result.isSuccess()) {
                if (result.getData().getBoolean("hasUpdate")) {
                    LOGGER.warn(result.getData().getString("msg"));
                }
            } else {
                LOGGER.warn("检查琉璃架构版本失败:{}", result.getMsg());
            }
        };
        if (applicationName.equals(ServiceNameConstants.LIULI_SERVICE)) {
            // 琉璃自身调用
            JSONObject response = new JSONObject();
            try {
                Object localResult = SpringUtils.getBean(ILogicServiceProxy.class).run("checkV4Version", params);
                response.put("code", HttpStatus.SUCCESS);
                response.put("data", localResult);
            } catch (ServiceException e) {
                response.put("code", e.getCode());
                response.put("msg", e.getMessage());
            }
            handler.run(response.toString());
        } else if (liuLiClientConfig.getLiuLiClientOperationalMode() == LiuLiClientOperationalModeEnum.INTRANET) {
            RemoteLiuLiService remoteLiuLiService = (RemoteLiuLiService) dynamicFeignClientFactory.getFeignClient(RemoteLiuLiService.class, ServiceNameConstants.LIULI_SERVICE);
            R<Map<String, Object>> remoteResult = remoteLiuLiService.checkV4Version(params.toString(), SecurityConstants.INNER);
            handler.run(remoteResult.parseResponseJson().toString());
        } else {
            try (CloseHttpLogHandler logHandler = CloseHttpLogHandler.build()) {
                logHandler.apply();
                RestAsyncTools.post(LiuLiApi.CHECK_V4_VERSION_PATH, params.toString(), null,
                        new StandardAsyncResponse(handler, e -> LOGGER.error("检查琉璃架构版本失败:{}", e.getMessage())));
            }
        }
    }

    /**
     * 上报心跳
     */
    public void uploadHeartbeat() {
        if (liuLiClientConfig.getLiuLiClientOperationalMode() == LiuLiClientOperationalModeEnum.OFFLINE) {
            return;
        }
        if (!liuLiClientConfig.getHeartbeat()) {
            return;
        }
        String applicationName = applicationService.getApplicationName();
        if (applicationName.equals("af-runtime")) {
            return;
        }
        String applicationLockKey = geApplicationLockKey(applicationName);
        try {
            redisService.lock(applicationLockKey, 60, 300, () -> {
                //调用琉璃中心需传递的参数
                JSONObject params = new JSONObject()
                        .put("tenantUUID", applicationService.getTenantName())
                        .put("name", applicationName);

                StandardAsyncResponse.SuccessHandler handler = response -> {
                    R<JSONObject> result = R.build(new JSONObject(response));
                    if (!result.isSuccess()) {
                        LOGGER.warn("心跳上传失败:{}", result.getMsg());
                    }
                };
                if (applicationName.equals(ServiceNameConstants.LIULI_SERVICE)) {
                    // 琉璃自身调用
                    JSONObject response = new JSONObject();
                    try {
                        Object localResult = SpringUtils.getBean(ILogicServiceProxy.class).run("uploadHeartBeat", params);
                        response.put("code", HttpStatus.SUCCESS);
                        response.put("data", localResult);
                    } catch (ServiceException e) {
                        response.put("code", e.getCode());
                        response.put("msg", e.getMessage());
                    }
                    handler.run(response.toString());
                } else if (liuLiClientConfig.getLiuLiClientOperationalMode() == LiuLiClientOperationalModeEnum.INTRANET) {
                    RemoteLiuLiService remoteLiuLiService = (RemoteLiuLiService) dynamicFeignClientFactory.getFeignClient(RemoteLiuLiService.class, ServiceNameConstants.LIULI_SERVICE);
                    R<Map<String, Object>> remoteResult = remoteLiuLiService.uploadHeartBeat(params.toString(), SecurityConstants.INNER);
                    handler.run(remoteResult.parseResponseJson().toString());
                } else {
                    try (CloseHttpLogHandler logHandler = CloseHttpLogHandler.build()) {
                        logHandler.apply();
                        RestAsyncTools.post(LiuLiApi.UPLOAD_HEARTBEAT_PATH, params.toString(), null,
                                new StandardAsyncResponse(handler, e -> LOGGER.error("心跳上传失败:{}", e.getMessage())));
                    }
                }
            });
        } catch (InterruptedException ignored) {
        }
    }

}
